package com.tiviacz.travelersbackpack.inventory;

import com.mojang.datafixers.util.Pair;
import com.tiviacz.travelersbackpack.TravelersBackpack;
import com.tiviacz.travelersbackpack.blockentity.BackpackBlockEntity;
import com.tiviacz.travelersbackpack.client.screens.BackpackScreen;
import com.tiviacz.travelersbackpack.common.BackpackAbilities;
import com.tiviacz.travelersbackpack.component.ComponentUtils;
import com.tiviacz.travelersbackpack.components.*;
import com.tiviacz.travelersbackpack.config.TravelersBackpackConfig;
import com.tiviacz.travelersbackpack.init.ModDataComponents;
import com.tiviacz.travelersbackpack.init.ModItems;
import com.tiviacz.travelersbackpack.inventory.handler.ItemStackHandler;
import com.tiviacz.travelersbackpack.inventory.handler.StorageAccessWrapper;
import com.tiviacz.travelersbackpack.inventory.menu.BackpackBaseMenu;
import com.tiviacz.travelersbackpack.inventory.menu.BackpackItemMenu;
import com.tiviacz.travelersbackpack.inventory.menu.slot.BackpackSlotItemHandler;
import com.tiviacz.travelersbackpack.inventory.menu.slot.ToolSlotItemHandler;
import com.tiviacz.travelersbackpack.inventory.sorter.SortSelector;
import com.tiviacz.travelersbackpack.inventory.upgrades.IEnable;
import com.tiviacz.travelersbackpack.inventory.upgrades.ITickableUpgrade;
import com.tiviacz.travelersbackpack.inventory.upgrades.IUpgrade;
import com.tiviacz.travelersbackpack.inventory.upgrades.tanks.TanksUpgrade;
import com.tiviacz.travelersbackpack.item.upgrades.TanksUpgradeItem;
import com.tiviacz.travelersbackpack.item.upgrades.UpgradeItem;
import com.tiviacz.travelersbackpack.network.ClientboundSyncItemStackPacket;
import com.tiviacz.travelersbackpack.util.ItemStackUtils;
import com.tiviacz.travelersbackpack.util.PacketDistributor;
import com.tiviacz.travelersbackpack.util.Reference;
import com.tiviacz.travelersbackpack.util.RegistryHelper;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2371;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_3222;
import net.minecraft.class_5455;
import net.minecraft.class_9282;
import net.minecraft.class_9323;
import net.minecraft.class_9331;
import net.minecraft.class_9334;

public class BackpackWrapper {
    public static final BackpackWrapper DUMMY = new BackpackWrapper(ModItems.STANDARD_TRAVELERS_BACKPACK.method_7854(), Reference.BLOCK_ENTITY_SCREEN_ID, null, null);

    protected class_1799 stack;
    private ItemStackHandler inventory;
    private ItemStackHandler upgrades;
    private ItemStackHandler tools;

    public ItemStackHandler upgradesTracker;

    private final UpgradeManager upgradeManager;
    private class_1657 owner;
    public ArrayList<class_1657> playersUsing = new ArrayList<>();
    protected class_1937 level;
    private final int screenID;
    private long tanksCapacity = 0;
    public int index = -1;

    //Storage, Upgrades, Tools
    //0 - off, 1 - on
    public int[] dataLoad = new int[]{1, 1, 1};

    public Runnable saveHandler = () -> {
    };
    public Runnable abilityHandler = () -> {
    };
    public class_2338 backpackPos;

    public static final int STORAGE_ID = 0;
    public static final int UPGRADES_ID = 1;
    public static final int TOOLS_ID = 2;

    public BackpackWrapper(class_1799 stack, int screenID, @Nullable class_1657 player, @Nullable class_1937 level, int index) {
        this(stack, screenID, player, level);
        this.index = index;
    }

    public BackpackWrapper(class_1799 stack, int screenID, @Nullable class_1657 player, @Nullable class_1937 level) {
        this(stack, screenID, player, level, new int[]{1, 1, 1});
    }

    public BackpackWrapper(class_1799 stack, int screenID, @Nullable class_1657 player, @Nullable class_1937 level, int[] dataLoad) {
        if(player != null) {
            this.playersUsing.add(player);
        }
        if(screenID == Reference.WEARABLE_SCREEN_ID) {
            this.setBackpackOwner(player);
        }

        this.stack = stack;

        if(!isSizeInitialized(stack)) {
            initializeSize(stack);
        }

        this.screenID = screenID;
        this.level = level;
        this.dataLoad = dataLoad;

        this.loadHandlers();
        this.setBackpackTankCapacity();
        this.upgradeManager = new UpgradeManager(this);

        if(upgrades != null) {
            if(!this.stack.method_57826(ModDataComponents.RENDER_INFO)) {
                this.setRenderInfo(RenderInfo.EMPTY.compoundTag());
            }

            if(stack.method_57826(ModDataComponents.STARTER_UPGRADES)) {
                StarterUpgrades upgrades = stack.method_57824(ModDataComponents.STARTER_UPGRADES);
                if(upgrades != null) {
                    upgrades.upgrades().forEach(this::setStarterUpgrade);
                    stack.method_57381(ModDataComponents.STARTER_UPGRADES);
                }
            }

            //Old Data Conversion (Should not run in regular case)
            if(stack.method_57826(ModDataComponents.FLUID_TANKS_OLD)) {
                class_1799 oldTanks = ModItems.TANKS_UPGRADE.method_7854();
                oldTanks.method_57379(ModDataComponents.FLUIDS, new Fluids(new FluidVariantWrapper(stack.method_57824(ModDataComponents.FLUID_TANKS_OLD).leftTank().fluidVariant(), stack.method_57824(ModDataComponents.FLUID_TANKS_OLD).leftTank().amount()),
                        new FluidVariantWrapper(stack.method_57824(ModDataComponents.FLUID_TANKS_OLD).rightTank().fluidVariant(), stack.method_57824(ModDataComponents.FLUID_TANKS_OLD).rightTank().amount())));
                this.setStarterUpgrade(oldTanks);
                stack.method_57381(ModDataComponents.FLUID_TANKS_OLD);
            }
        }
    }

    //Create wrapper from the Backpack Stack
    public static BackpackWrapper fromStack(class_1799 backpackStack) {
        return new BackpackWrapper(backpackStack, Reference.ITEM_SCREEN_ID, null, null);
    }

    public void setBackpackStack(class_1799 backpack) {
        this.stack = backpack;

        //Update client tanks if present
        getUpgradeManager().getUpgrade(TanksUpgrade.class).ifPresent(tanksUpgrade -> tanksUpgrade.syncClients(backpack));
    }

    public class_1799 getBackpackStack() {
        return this.stack;
    }

    public int getBackpackSlotIndex() {
        return this.index;
    }

    public void setBackpackOwner(class_1657 player) {
        this.owner = player;
    }

    @Nullable
    public class_1657 getBackpackOwner() {
        return this.owner;
    }

    public ArrayList<class_1657> getPlayersUsing() {
        return this.playersUsing;
    }

    public void addUser(class_1657 player) {
        if(!this.playersUsing.contains(player)) {
            this.playersUsing.add(player);
        }
    }

    public class_1937 getLevel() {
        return this.level;
    }

    public ItemStackHandler loadHandler(class_9331<BackpackContainerContents> data, int defaultSize, int dataId, BiFunction<class_2371<class_1799>, Integer, ItemStackHandler> handlerFunction) {
        if(this.stack.method_57826(data)) {
            BackpackContainerContents contents = this.stack.method_57824(data);
            if(contents.getItems().size() < defaultSize) {
                contents = expandContents(contents, defaultSize, this.stack, data);
            }

            class_2371<class_1799> stacks = class_2371.method_10213(contents.getItems().size(), class_1799.field_8037);
            contents.copyInto(stacks);
            return handlerFunction.apply(stacks, dataId);
        }
        return handlerFunction.apply(class_2371.method_10213(defaultSize, class_1799.field_8037), dataId);
    }

    public void loadHandlers() {
        if(this.dataLoad[STORAGE_ID] == 1) {
            loadStorage();
        }
        if(this.dataLoad[UPGRADES_ID] == 1) {
            loadUpgrades();
        }
        if(this.dataLoad[TOOLS_ID] == 1) {
            loadTools();
        }
    }

    public void loadStorage() {
        this.inventory = loadHandler(ModDataComponents.BACKPACK_CONTAINER, getStorageSize(), STORAGE_ID, this::createHandler);
    }

    public void loadUpgrades() {
        this.upgrades = loadHandler(ModDataComponents.UPGRADES, getUpgradesSize(), UPGRADES_ID, this::createUpgradeHandler);
        this.upgradesTracker = loadHandler(ModDataComponents.UPGRADES, getUpgradesSize(), UPGRADES_ID, (stacks, data) -> new ItemStackHandler(stacks));
    }

    public void loadTools() {
        this.tools = loadHandler(ModDataComponents.TOOLS_CONTAINER, getToolSize(), TOOLS_ID, this::createHandler);
    }

    public void loadAdditionally(int type) {
        //Load handler additionally if not loaded in artificial wrapper
        if(dataLoad[type] == 0) {
            if(type == STORAGE_ID) loadStorage();
            if(type == UPGRADES_ID) loadUpgrades();
            if(type == TOOLS_ID) loadTools();
            dataLoad[type] = 1;
        }
    }

    public BackpackContainerContents expandContents(BackpackContainerContents contents, int size, class_1799 backpack, class_9331 type) {
        if(contents.getItems().size() < size) {
            List<class_1799> oldItems = contents.getItems();
            //Populate expanded items list with empty stacks
            ArrayList<class_1799> itemList = new ArrayList<>(Collections.nCopies(size, class_1799.field_8037));

            for(int i = 0; i < oldItems.size(); i++) {
                if(!oldItems.get(i).method_7960()) {
                    itemList.set(i, oldItems.get(i));
                }
            }
            //Expanded items
            BackpackContainerContents expandedContents = BackpackContainerContents.fromItems(size, itemList);
            backpack.method_57379(type, expandedContents);
            return expandedContents;
        }
        return contents;
    }

    public void setStarterUpgrade(class_1799 upgrade) {
        if(this.level == null) {
            return;
        }
        if(upgrade.method_7909().method_45382(this.level.method_45162())) {
            for(int i = 0; i < this.upgrades.getSlots(); i++) {
                if(this.upgrades.getStackInSlot(i).method_7960()) {
                    this.upgrades.setStackInSlot(i, upgrade);
                    this.upgradesTracker.setStackInSlot(i, upgrade);

                    if(upgrade.method_7909() instanceof TanksUpgradeItem) {
                        this.setRenderInfo(TanksUpgradeItem.writeToRenderData().compoundTag());
                    }
                    break;
                }
            }
        }
    }

    public int getStorageSize() {
        return this.stack.method_57825(ModDataComponents.STORAGE_SLOTS, Tiers.LEATHER.getStorageSlots());
    }

    public int getUpgradesSize() {
        return this.stack.method_57825(ModDataComponents.UPGRADE_SLOTS, Tiers.LEATHER.getUpgradeSlots());
    }

    public int getToolSize() {
        return this.stack.method_57825(ModDataComponents.TOOL_SLOTS, Tiers.LEATHER.getToolSlots());
    }

    public StorageAccessWrapper getStorageForInputOutput() {
        return new StorageAccessWrapper(this, getStorage());
    }

    public ItemStackHandler getStorage() {
        return this.inventory;
    }

    public ItemStackHandler getUpgrades() {
        return this.upgrades;
    }

    public ItemStackHandler getTools() {
        return this.tools;
    }

    public UpgradeManager getUpgradeManager() {
        return this.upgradeManager;
    }

    @Nullable
    public class_5455 getRegistriesAccess() {
        if(level != null) {
            return level.method_30349();
        }
        if(!playersUsing.isEmpty() && playersUsing.get(0).method_37908().method_30349() != null) {
            return playersUsing.get(0).method_37908().method_30349();
        }
        if(RegistryHelper.getRegistryAccess().isPresent()) {
            return RegistryHelper.getRegistryAccess().get();
        }
        return null;
    }

    public List<Integer> getUnsortableSlots() {
        return this.stack.method_57825(ModDataComponents.SLOTS, Slots.EMPTY).unsortables();
    }

    public List<Pair<Integer, Pair<class_1799, Boolean>>> getMemorySlots() {
        return this.stack.method_57825(ModDataComponents.SLOTS, Slots.EMPTY).memory();
    }

    public int getScreenID() {
        return this.screenID;
    }

    public class_2561 getBackpackScreenTitle() {
        return this.stack.method_57826(class_9334.field_49631) ? this.stack.method_57824(class_9334.field_49631) : class_2561.method_43471("screen.travelersbackpack.title");
    }

    public void setUnsortableSlots(List<Integer> unsortables) {
        Slots old = this.stack.method_57825(ModDataComponents.SLOTS, Slots.EMPTY);
        setData(ModDataComponents.SLOTS, Slots.updateUnsortables(old, unsortables));
    }

    public void setMemorySlots(List<Pair<Integer, Pair<class_1799, Boolean>>> memory) {
        Slots old = this.stack.method_57825(ModDataComponents.SLOTS, Slots.EMPTY);
        setData(ModDataComponents.SLOTS, Slots.updateMemory(old, memory));
    }

    public <T> void setDataAndSync(class_9331<T> dataComponentType, T value) {
        setData(dataComponentType, value);

        //Update on client
        sendDataToClients(dataComponentType);
    }

    public <T> void setData(class_9331<T> dataComponentType, T value) {
        this.stack.method_57379(dataComponentType, value);
        this.saveHandler.run();

        if(dataComponentType == ModDataComponents.ABILITY_ENABLED) {
            this.abilityHandler.run();
        }
    }

    public boolean showToolSlots() {
        return this.stack.method_57825(ModDataComponents.SHOW_TOOL_SLOTS, false);
    }

    public boolean showMoreButtons() {
        return this.stack.method_57825(ModDataComponents.SHOW_MORE_BUTTONS, false);
    }

    public boolean tanksVisible() {
        if(this.stack.method_57826(ModDataComponents.RENDER_INFO)) {
            return this.stack.method_57824(ModDataComponents.RENDER_INFO).hasTanks();
        }
        return getUpgradeManager().getUpgrade(TanksUpgrade.class).isPresent();
    }

    public long getBackpackTankCapacity() {
        return this.tanksCapacity;
    }

    public void setBackpackPos(class_2338 pos) {
        this.backpackPos = pos;
    }

    public class_2338 getBackpackPos() {
        return this.backpackPos;
    }

    public void setBackpackTankCapacity() {
        int rows = getRows() + (isExtended() ? 2 : 0);
        this.tanksCapacity = Tiers.of(this.stack.method_57825(ModDataComponents.TIER, 0)).getTankCapacityPerRow() * rows;
    }

    public RenderInfo getRenderInfo() {
        return this.stack.method_57825(ModDataComponents.RENDER_INFO, RenderInfo.EMPTY);
    }

    public void setRenderInfo(class_2487 compound) {
        setDataAndSync(ModDataComponents.RENDER_INFO, new RenderInfo(compound));
    }

    public void removeRenderInfo() {
        setRenderInfo(new class_2487());
    }

    public boolean isAbilityEnabled() {
        return this.stack.method_57825(ModDataComponents.ABILITY_ENABLED, TravelersBackpackConfig.getConfig().backpackAbilities.forceAbilityEnabled);
    }

    public SortSelector.SortType getSortType() {
        int type = this.stack.method_57825(ModDataComponents.SORT_TYPE, 0);
        return SortSelector.SortType.values()[type];
    }

    public void setNextSortType() {
        SortSelector.SortType type = getSortType();
        setDataAndSync(ModDataComponents.SORT_TYPE, type.next().ordinal());
    }

    public boolean hasSleepingBag() {
        return this.stack.method_57826(ModDataComponents.SLEEPING_BAG_COLOR);
    }

    public int getSleepingBagColor() {
        return this.stack.method_57825(ModDataComponents.SLEEPING_BAG_COLOR, -1);
    }

    public void setSleepingBagColor(int colorId) {
        setData(ModDataComponents.SLEEPING_BAG_COLOR, colorId);
    }

    public boolean isOwner(class_1657 player) {
        if(getBackpackOwner() != null) {
            return getBackpackOwner().method_5628() == player.method_5628();
        }
        return true;
    }

    public boolean isDyed() {
        return this.stack.method_57826(class_9334.field_49644);
    }

    public int getDyeColor() {
        return this.stack.method_57825(class_9334.field_49644, new class_9282(-1, false)).comp_2384();
    }

    public int getCooldown() {
        return this.stack.method_57825(ModDataComponents.COOLDOWN, 0);
    }

    public void setCooldown(int cooldownInSeconds) {
        setDataAndSync(ModDataComponents.COOLDOWN, cooldownInSeconds);
    }

    //Block Entity
    public void decreaseCooldown() {
        if(getCooldown() > 0) {
            this.stack.method_57368(ModDataComponents.COOLDOWN, 0, currentCooldown -> currentCooldown - 1);
            this.saveHandler.run();
        }
    }

    public boolean canUpgradeTick() {
        return this.stack.method_57826(ModDataComponents.UPGRADE_TICK_INTERVAL);
    }

    public boolean hasTickingUpgrade() {
        return this.upgradeManager.hasTickingUpgrade();
    }

    public int getUpgradeTickInterval() {
        return this.stack.method_57825(ModDataComponents.UPGRADE_TICK_INTERVAL, 100);
    }

    public void setUpgradeTickInterval(int ticks) {
        setDataAndSync(ModDataComponents.UPGRADE_TICK_INTERVAL, ticks);
    }

    public void removeUpgradeTickInterval() {
        this.stack.method_57381(ModDataComponents.UPGRADE_TICK_INTERVAL);
    }

    public boolean isExtended() {
        return getStorageSize() > 81;
    }

    public int getSlotsInRow() {
        if(isExtended()) {
            return 11;
        }
        return 9;
    }

    public int getRows() {
        return (int)Math.ceil((double)getStorageSize() / getSlotsInRow());
    }

    public void sendDataToClients(class_9331... dataComponentTypes) {
        //Other methods sync data for block entities
        if(getScreenID() == Reference.BLOCK_ENTITY_SCREEN_ID) return;

        //Sync stack in slot or hand
        if(getScreenID() == Reference.ITEM_SCREEN_ID && !getPlayersUsing().stream().filter(p -> !p.method_37908().field_9236).toList().isEmpty()) {
            int slotIndex = this.index == -1 ? getPlayersUsing().get(0).method_31548().field_7545 : this.index;
            PacketDistributor.sendToPlayer((class_3222)this.getPlayersUsing().get(0), new ClientboundSyncItemStackPacket(getPlayersUsing().get(0).method_5628(), slotIndex, getBackpackStack(), ItemStackUtils.createDataComponentMap(getBackpackStack(), dataComponentTypes)));
            return;
        }
        //Sync stack equipped in back slot
        if(TravelersBackpack.enableIntegration()) {
            //Sync backpack data on clients differently for integration, because of the way backpacks are handled
            if(getScreenID() == Reference.WEARABLE_SCREEN_ID && !getPlayersUsing().stream().filter(p -> !p.method_37908().field_9236).toList().isEmpty()) {
                for(class_1657 player : getPlayersUsing()) {
                    PacketDistributor.sendToPlayer((class_3222)player, new ClientboundSyncItemStackPacket(player.method_5628(), -1, getBackpackStack(), ItemStackUtils.createDataComponentMap(getBackpackStack(), dataComponentTypes)));
                }
            }
            return;
        }
        //Sync attachment stack
        if(getBackpackOwner() != null) {
            class_9323.class_9324 mapBuilder = class_9323.method_57827();
            class_1799 serverDataHolder = ComponentUtils.getWearingBackpack(getBackpackOwner()).method_7972();
            for(class_9331 type : dataComponentTypes) {
                class_1799 serverDataHolderCopy = ItemStackUtils.reduceSize(serverDataHolder);
                if(!serverDataHolderCopy.method_57826(type)) {
                    continue;
                }
                mapBuilder.method_57840(type, serverDataHolderCopy.method_57824(type));
            }
            ComponentUtils.getComponent(getBackpackOwner()).ifPresent(data -> data.synchronise(mapBuilder.method_57838()));
        }
    }

    public Optional<Pair<Integer, Pair<class_1799, Boolean>>> getMemorizedSlot(int slot) {
        return getMemorySlots().stream()
                .filter(pair -> pair.getFirst() == slot)
                .findFirst();
    }

    private ItemStackHandler createHandler(class_2371<class_1799> stacks, int dataId) {
        return new ItemStackHandler(stacks) {
            @Override
            protected void onContentsChanged(int slot) {
                setSlotChanged(slot, getStackInSlot(slot), dataId);

                if(dataId == TOOLS_ID) {
                    sendDataToClients(ModDataComponents.TOOLS_CONTAINER);
                }

                //Update comparator
                saveHandler.run();
            }

            @Override
            public boolean isItemValid(int slot, class_1799 stack) {
                if(dataId == 2) {
                    return ToolSlotItemHandler.isValid(stack);
                }
                return BackpackSlotItemHandler.isItemValid(stack);
            }
        };
    }

    public void setSlotChanged(int index, class_1799 stack, int dataId) {
        switch(dataId) {
            case STORAGE_ID:
                this.stack.method_57367(ModDataComponents.BACKPACK_CONTAINER, new BackpackContainerContents(this.getStorage().getSlots()), new BackpackContainerContents.Slot(index, stack), BackpackContainerContents::updateSlot);
                break;
            case UPGRADES_ID:
                this.stack.method_57367(ModDataComponents.UPGRADES, new BackpackContainerContents(this.getUpgrades().getSlots()), new BackpackContainerContents.Slot(index, stack), BackpackContainerContents::updateSlot);
                break;
            case TOOLS_ID:
                this.stack.method_57367(ModDataComponents.TOOLS_CONTAINER, new BackpackContainerContents(this.getTools().getSlots()), new BackpackContainerContents.Slot(index, stack), BackpackContainerContents::updateSlot);
                break;
        }
    }

    public void applyLowestTickInterval() {
        int minimalTickInterval = 100;
        for(int i = 0; i < this.upgrades.getSlots(); i++) {
            class_1799 upgrade = this.upgrades.getStackInSlot(i);
            if(!upgrade.method_7960()) {
                if(upgrade.method_57825(ModDataComponents.UPGRADE_ENABLED, true) && upgrade.method_57826(ModDataComponents.COOLDOWN)) {
                    minimalTickInterval = Math.min(minimalTickInterval, upgrade.method_57824(ModDataComponents.COOLDOWN));
                }
            }
        }
        if(!canUpgradeTick() || minimalTickInterval != getUpgradeTickInterval()) {
            setUpgradeTickInterval(minimalTickInterval);
        }
    }

    public void updateMinimalTickInterval(class_1799 newStack) {
        if(level != null && level.field_9236) return;

        boolean applyLowestTickInterval = false;
        if(newStack.method_7909() instanceof UpgradeItem upgradeItem) {
            if(upgradeItem.isTickingUpgrade()) {
                if(newStack.method_57825(ModDataComponents.UPGRADE_ENABLED, true)) {
                    int tickInterval = getUpgradeTickInterval();
                    if(newStack.method_57826(ModDataComponents.COOLDOWN)) {
                        tickInterval = newStack.method_57824(ModDataComponents.COOLDOWN);
                    }
                    if(!canUpgradeTick() || tickInterval < getUpgradeTickInterval()) {
                        setUpgradeTickInterval(tickInterval);
                    } else if(tickInterval > getUpgradeTickInterval()) {
                        applyLowestTickInterval = true;
                    }
                } else {
                    applyLowestTickInterval = true;
                }
            }
        } else {
            applyLowestTickInterval = true;
        }

        if(canUpgradeTick()) {
            if(!hasTickingUpgrade()) {
                removeUpgradeTickInterval();
            } else if(applyLowestTickInterval) {
                applyLowestTickInterval();
            }
        }
    }

    private ItemStackHandler createUpgradeHandler(class_2371<class_1799> stacks, int dataId) {
        return new ItemStackHandler(stacks) {
            @Override
            protected void onContentsChanged(int slot) {
                setSlotChanged(slot, getStackInSlot(slot), dataId);

                //Menu and screen updates
                if(!getPlayersUsing().isEmpty()) {
                    getUpgradeManager().detectedChange(upgradesTracker, slot);
                }

                updateMinimalTickInterval(getStackInSlot(slot));

                //Update client
                saveHandler.run();
            }

            @Override
            public int getSlotLimit(int slot) {
                return 1;
            }

            @Override
            public boolean isItemValid(int slot, class_1799 stack) {
                boolean isValid = true;
                //Check if upgrade is already present
                for(int i = 0; i < this.getSlots(); i++) {
                    if(getStackInSlot(i).method_7909() == stack.method_7909()) {
                        isValid = false;
                        break;
                    }
                }
                if(!isValid) {
                    return false;
                }
                if(stack.method_7909() instanceof TanksUpgradeItem) {
                    isValid = TanksUpgradeItem.canBePutInBackpack(getBackpackTankCapacity(), stack);
                }
                if(!checkIfUpgradeValid(stack)) {
                    isValid = false;
                }
                return isValid;
            }

            public boolean checkIfUpgradeValid(class_1799 upgradeStack) {
                if(upgradeStack.method_7909() instanceof UpgradeItem upgradeItem) {
                    class_1657 player = getPlayersUsing().isEmpty() ? null : getPlayersUsing().getFirst();
                    if(player == null) {
                        return false;
                    }
                    return upgradeItem.method_45382(player.method_37908().method_45162());
                }
                return false;
            }
        };
    }

    public static boolean isSizeInitialized(class_1799 stack) {
        return stack.method_57826(ModDataComponents.STORAGE_SLOTS) && stack.method_57826(ModDataComponents.UPGRADE_SLOTS) && stack.method_57826(ModDataComponents.TOOL_SLOTS);
    }

    public static void initializeSize(class_1799 stack) {
        Tiers.Tier tier = Tiers.LEATHER;
        if(stack.method_57826(ModDataComponents.TIER)) {
            tier = Tiers.of(stack.method_57824(ModDataComponents.TIER));
        }
        if(!stack.method_57826(ModDataComponents.STORAGE_SLOTS)) {
            stack.method_57379(ModDataComponents.STORAGE_SLOTS, tier.getStorageSlots());
        }
        if(!stack.method_57826(ModDataComponents.UPGRADE_SLOTS)) {
            stack.method_57379(ModDataComponents.UPGRADE_SLOTS, tier.getUpgradeSlots());
        }
        if(!stack.method_57826(ModDataComponents.TOOL_SLOTS)) {
            stack.method_57379(ModDataComponents.TOOL_SLOTS, tier.getToolSlots());
        }
    }

    //Used if slots are removed/added - reconstructs modifiable slots & updates screen
    public void requestMenuAndScreenUpdate() {
        requestMenuUpdate();
        requestScreenUpdate();
    }

    public void requestMenuUpdate() {
        if(!getPlayersUsing().isEmpty()) {
            getPlayersUsing().stream().filter(player -> player.field_7512 instanceof BackpackBaseMenu).forEach(player -> ((BackpackBaseMenu)player.field_7512).updateModifiableSlots());
        }
    }

    public void requestScreenUpdate() {
        if(!getPlayersUsing().isEmpty() && !getPlayersUsing().stream().filter(player -> player.method_37908().field_9236).toList().isEmpty()) {
            if(class_310.method_1551().field_1755 instanceof BackpackScreen screen) {
                screen.updateScreen(false);
            }
        }
    }

    public static void tickForBlockEntity(BackpackBlockEntity backpackBlockEntity) {
        BackpackWrapper wrapper = backpackBlockEntity.getWrapper();
        if(wrapper != BackpackWrapper.DUMMY) {
            if(wrapper.isAbilityEnabled() && BackpackAbilities.isOnList(BackpackAbilities.BLOCK_ABILITIES_LIST, wrapper.getBackpackStack())) {
                boolean decreaseCooldown = BackpackAbilities.ABILITIES.abilityTickBlock(backpackBlockEntity);
                if(wrapper.getCooldown() > 0) {
                    if(decreaseCooldown) {
                        wrapper.decreaseCooldown();
                    }
                }
            }
        }
    }

    @Nullable
    public static BackpackWrapper getBackpackWrapper(class_1657 player, class_1799 backpack, int[] dataLoad) {
        if(ComponentUtils.isWearingBackpack(player)) {
            if(player.field_7512 instanceof BackpackItemMenu menu && menu.getWrapper().getScreenID() == Reference.WEARABLE_SCREEN_ID) {
                return menu.getWrapper();
            } else {
                for(class_1657 otherPlayer : player.method_37908().method_18456()) {
                    if(otherPlayer.field_7512 instanceof BackpackItemMenu menu && menu.getWrapper().isOwner(player) && menu.getWrapper().getScreenID() == Reference.WEARABLE_SCREEN_ID) {
                        return menu.getWrapper();
                    }
                }
                return new BackpackWrapper(backpack, Reference.WEARABLE_SCREEN_ID, player, player.method_37908(), dataLoad);
            }
        }
        return null;
    }

    public static void tick(class_1799 stack, class_1657 player, boolean integration) {
        if(!integration) {
            if(TravelersBackpack.enableIntegration()) return;
        }

        if(player.method_5805() && ComponentUtils.isWearingBackpack(player)) {
            int ticks = (int)player.method_37908().method_8510();
            if(BackpackAbilities.isOnList(BackpackAbilities.ITEM_ABILITIES_LIST, ComponentUtils.getWearingBackpack(player))) {
                if(BackpackAbilities.isAbilityEnabledInConfig(stack)) {
                    if(stack.method_57825(ModDataComponents.ABILITY_ENABLED, TravelersBackpackConfig.getConfig().backpackAbilities.forceAbilityEnabled)) {
                        boolean decreaseCooldown = BackpackAbilities.ABILITIES.abilityTick(stack, player);
                        if(stack.method_57825(ModDataComponents.COOLDOWN, 0) > 0) {
                            BackpackWrapper wrapper;
                            if(ticks % 100 == 0) {
                                if(decreaseCooldown) {
                                    wrapper = ComponentUtils.getBackpackWrapper(player, stack, ComponentUtils.NO_ITEMS);
                                    int cooldown = wrapper.getCooldown();
                                    if(player.method_37908().field_9236) return;
                                    if(cooldown - 100 < 0) {
                                        wrapper.setCooldown(0);
                                    } else {
                                        wrapper.setCooldown(cooldown - 100);
                                    }
                                }
                            }
                        }
                    } else { //Tick cooldown even if ability switched off
                        if(stack.method_57825(ModDataComponents.COOLDOWN, 0) > 0) {
                            BackpackWrapper wrapper;
                            if(ticks % 100 == 0) {
                                wrapper = ComponentUtils.getBackpackWrapper(player, stack, ComponentUtils.NO_ITEMS);
                                int cooldown = wrapper.getCooldown();
                                if(player.method_37908().field_9236) return;
                                if(cooldown - 100 < 0) {
                                    wrapper.setCooldown(0);
                                } else {
                                    wrapper.setCooldown(cooldown - 100);
                                }
                            }
                        }
                    }
                }
            } else if(stack.method_57825(ModDataComponents.ABILITY_ENABLED, false)) {
                stack.method_57379(ModDataComponents.ABILITY_ENABLED, false);
            }

            if(stack.method_57826(ModDataComponents.UPGRADE_TICK_INTERVAL)) {
                int upgradeTicks = stack.method_57824(ModDataComponents.UPGRADE_TICK_INTERVAL);
                if(upgradeTicks == 0) return;
                BackpackWrapper wrapper;
                if(ticks % upgradeTicks == 0) {
                    wrapper = ComponentUtils.getBackpackWrapper(player, stack, ComponentUtils.UPGRADES_ONLY);
                    wrapper.getUpgradeManager().upgrades.forEach(upgradeBase -> {
                        if(upgradeBase instanceof ITickableUpgrade tickable) {
                            boolean tick = true;
                            if(upgradeBase instanceof IEnable enable) {
                                tick = enable.isEnabled(upgradeBase);
                            }
                            if(tick) {
                                tickable.tick(player, player.method_37908(), player.method_24515(), ticks);
                            }
                        }
                    });
                }
            }
        }
    }
}
