package com.tiviacz.travelersbackpack.inventory;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.tiviacz.travelersbackpack.init.ModDataComponents;
import com.tiviacz.travelersbackpack.inventory.handler.ItemStackHandler;
import com.tiviacz.travelersbackpack.inventory.upgrades.IEnable;
import com.tiviacz.travelersbackpack.inventory.upgrades.ITickableUpgrade;
import com.tiviacz.travelersbackpack.inventory.upgrades.UpgradeBase;
import com.tiviacz.travelersbackpack.item.upgrades.UpgradeItem;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import net.minecraft.class_1799;

public class UpgradeManager {
    public final BackpackWrapper wrapper;
    public final ItemStackHandler upgradesHandler;
    public BiMap<Integer, Optional<UpgradeBase<?>>> mappedUpgrades;
    public List<UpgradeBase<?>> upgrades = new ArrayList<>();

    public UpgradeManager(BackpackWrapper wrapper) {
        this.wrapper = wrapper;
        this.upgradesHandler = wrapper.getUpgrades();
        this.mappedUpgrades = HashBiMap.create();
        initializeUpgrades();
    }

    public BackpackWrapper getWrapper() {
        return this.wrapper;
    }

    public ItemStackHandler getUpgradesHandler() {
        return this.upgradesHandler;
    }

    public boolean hasUpgradeInSlot(int slot) {
        return this.mappedUpgrades.containsKey(slot);
    }

    public <T extends UpgradeBase<T>> Optional<T> getUpgrade(Class<T> upgradeClass) {
        return upgrades.stream()
                .filter(upgradeClass::isInstance)
                .map(upgradeClass::cast)
                .findFirst();
    }

    public boolean canAddUpgrade(UpgradeItem upgradeItem) {
        return upgrades.stream().noneMatch(u -> u.getClass().equals(upgradeItem.getUpgradeClass()));
    }

    public boolean invalidateUpgrade(int slot) {
        Optional<UpgradeBase<?>> upgrade = this.mappedUpgrades.get(slot);

        //Update upgrade tracker
        getWrapper().upgradesTracker.setStackInSlot(slot, class_1799.field_8037);

        //Error - item in slot is not an upgrade, just return
        if(upgrade == null) {
            return false;
        }

        upgrade.ifPresent(upg -> {
            this.mappedUpgrades.remove(slot);
            upg.remove();
            upgrades.remove(upg);
        });
        return true;
    }

    public void initializeUpgrades() {
        for(int i = 0; i < getUpgradesHandler().getSlots(); i++) {
            applyUpgrade(i);
        }
    }

    public void detectedChange(ItemStackHandler tracker, int slot) {
        boolean needsUpdate = applyUpgrade(slot);

        //Update if tab changed status
        if(getTabStatus(tracker.getStackInSlot(slot)) != getTabStatus(getUpgradesHandler().getStackInSlot(slot)) || isTagSelector(getUpgradesHandler().getStackInSlot(slot), tracker.getStackInSlot(slot))) {
            needsUpdate = true;
            class_1799 stackToSet = getUpgradesHandler().getStackInSlot(slot).method_7972();
            tracker.setStackInSlot(slot, stackToSet);
        }

        if(mappedUpgrades.containsKey(slot)) {
            if(!(getUpgradesHandler().getStackInSlot(slot).method_7909() instanceof UpgradeItem)) {
                needsUpdate = this.invalidateUpgrade(slot);
            }
        }

        //Update menu and screen
        if(needsUpdate) {
            if(!getWrapper().getPlayersUsing().isEmpty()) {
                getWrapper().getPlayersUsing().stream().filter(player -> !player.method_37908().field_9236).forEach(player -> player.field_7512.method_7623());
            }
            getWrapper().requestMenuAndScreenUpdate();
        }
    }

    public boolean applyUpgrade(int slot) {
        AtomicBoolean atomic = new AtomicBoolean(false);
        class_1799 upgradeStack = getUpgradesHandler().getStackInSlot(slot);
        if(upgradeStack.method_7909() instanceof UpgradeItem upgradeItem) {
            if(canAddUpgrade(upgradeItem)) {
                upgradeItem.getUpgrade().apply(this, slot, upgradeStack).ifPresent(upgrade -> {
                    this.upgrades.add(upgrade);
                    this.mappedUpgrades.put(slot, Optional.of(upgrade));
                    atomic.set(true);
                });
            }
        }
        return atomic.get();
    }

    public boolean getTabStatus(class_1799 stack) {
        return stack.method_57825(ModDataComponents.TAB_OPEN, false);
    }

    public boolean isTagSelector(class_1799 current, class_1799 tracker) {
        return isTagSelector(current) ^ isTagSelector(tracker);
    }

    public boolean isTagSelector(class_1799 stack) {
        return stack.method_57825(ModDataComponents.FILTER_SETTINGS, List.of(1, 0, 1)).get(1) == 2;
    }

    public boolean hasTickingUpgrade() {
        return this.upgrades.stream()
                .filter(upgradeBase -> upgradeBase instanceof ITickableUpgrade && upgradeBase instanceof IEnable)
                .anyMatch(upgrade -> ((IEnable)upgrade).isEnabled(upgrade));
    }
}