/*
 * Decompiled with CFR 0.152.
 */
package net.carbonmc.graphene.async.redstone;

import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import net.carbonmc.graphene.async.AsyncSystemInitializer;
import net.carbonmc.graphene.event.AsyncHandler;
import net.minecraft.core.BlockPos;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.BlockGetter;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraftforge.event.TickEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@AsyncHandler(threadPool="compute", fallbackToSync=true)
public class AsyncRedstone {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final BlockingQueue<RedstoneTask> redstoneQueue = new LinkedBlockingQueue<RedstoneTask>(1000);
    private static final Map<BlockPos, RedstoneSnapshot> activeTasks = new ConcurrentHashMap<BlockPos, RedstoneSnapshot>();
    private static final int MAX_SNAPSHOT_AGE_MS = 100;
    private static final int MAX_UPDATES_PER_TICK = 100;
    private static final int REDSTONE_RADIUS = 10;
    private static final Semaphore taskSemaphore = new Semaphore(500);

    public static void init() {
        LOGGER.info("Async Redstone System initialized");
    }

    public static void shutdown() {
        redstoneQueue.clear();
        activeTasks.clear();
        LOGGER.info("Async Redstone System shutdown completed");
    }

    public static void computeRedstoneAsync(ServerLevel level, BlockPos sourcePos) {
        if (!level.m_46749_(sourcePos)) {
            LOGGER.warn("Attempted redstone computation at unloaded position {}", (Object)sourcePos);
            return;
        }
        if (activeTasks.containsKey(sourcePos)) {
            LOGGER.debug("Redstone computation already in progress for {}", (Object)sourcePos);
            return;
        }
        if (!taskSemaphore.tryAcquire()) {
            LOGGER.warn("Too many concurrent redstone computations, skipping {}", (Object)sourcePos);
            return;
        }
        try {
            RedstoneSnapshot snapshot = new RedstoneSnapshot(level, sourcePos);
            if (!snapshot.isValid()) {
                LOGGER.warn("Failed to create valid snapshot for {}", (Object)sourcePos);
                return;
            }
            RedstoneTask task = new RedstoneTask(snapshot);
            if (redstoneQueue.offer(task)) {
                activeTasks.put(sourcePos, snapshot);
            } else {
                LOGGER.warn("Redstone queue full, computation skipped for {}", (Object)sourcePos);
                taskSemaphore.release();
            }
        }
        catch (Exception e) {
            LOGGER.error("Error creating redstone snapshot for {}", (Object)sourcePos, (Object)e);
            taskSemaphore.release();
        }
    }

    @SubscribeEvent
    public static void onServerTick(TickEvent.ServerTickEvent event) {
        if (event.phase == TickEvent.Phase.END) {
            AsyncRedstone.processRedstoneTasks();
            AsyncRedstone.applyRedstoneUpdates();
        }
    }

    private static void processRedstoneTasks() {
        int processed = 0;
        while (processed < 100 && !redstoneQueue.isEmpty()) {
            RedstoneTask task = (RedstoneTask)redstoneQueue.poll();
            if (task == null) continue;
            ++processed;
            AsyncSystemInitializer.getThreadPool("compute").execute(() -> {
                try {
                    AsyncRedstone.processRedstoneTask(task);
                }
                finally {
                    taskSemaphore.release();
                }
            });
        }
    }

    private static void processRedstoneTask(RedstoneTask task) {
        try {
            if (!task.snapshot().isStillValid()) {
                LOGGER.debug("Redstone snapshot expired during computation");
                return;
            }
            HashMap<BlockPos, Integer> powerLevels = new HashMap<BlockPos, Integer>();
            ArrayDeque<BlockPos> queue = new ArrayDeque<BlockPos>();
            queue.add(task.snapshot().sourcePos());
            while (!queue.isEmpty()) {
                BlockPos current = (BlockPos)queue.poll();
                int power = AsyncRedstone.calculatePower(task.snapshot(), current);
                powerLevels.put(current, power);
                task.results().add(new RedstoneResult(current, power));
                for (BlockPos neighbor : AsyncRedstone.getNeighbors(current)) {
                    if (powerLevels.containsKey(neighbor) || !task.snapshot().containsPosition(neighbor)) continue;
                    queue.add(neighbor);
                }
            }
            task.completed().set(true);
        }
        catch (Exception e) {
            LOGGER.error("Redstone computation failed", (Throwable)e);
            task.error().set(e);
        }
    }

    private static void applyRedstoneUpdates() {
        Iterator<Map.Entry<BlockPos, RedstoneSnapshot>> it = activeTasks.entrySet().iterator();
        int updatesApplied = 0;
        block2: while (it.hasNext() && updatesApplied < 100) {
            Map.Entry<BlockPos, RedstoneSnapshot> entry = it.next();
            RedstoneSnapshot snapshot = entry.getValue();
            for (RedstoneTask task : snapshot.assignedTasks()) {
                if (!task.completed().get() || updatesApplied >= 100) continue;
                if (task.error().get() == null && snapshot.isStillValid()) {
                    for (RedstoneResult result : task.results()) {
                        if (updatesApplied++ >= 100) break;
                        BlockPos pos = result.pos();
                        if (!snapshot.level().m_46749_(pos)) continue;
                        try {
                            snapshot.level().m_46672_(pos, snapshot.level().m_8055_(pos).m_60734_());
                        }
                        catch (Exception e) {
                            LOGGER.error("Failed to update redstone at {}", (Object)pos, (Object)e);
                        }
                    }
                    LOGGER.debug("Applied {} redstone updates", (Object)task.results().size());
                }
                it.remove();
                continue block2;
            }
        }
    }

    private static int calculatePower(RedstoneSnapshot snapshot, BlockPos pos) {
        try {
            BlockState state = snapshot.getBlockState(pos);
            return state != null ? state.m_60746_((BlockGetter)snapshot.level(), pos, null) : 0;
        }
        catch (Exception e) {
            LOGGER.error("Error calculating power at {}", (Object)pos, (Object)e);
            return 0;
        }
    }

    private static List<BlockPos> getNeighbors(BlockPos pos) {
        return List.of(pos.m_7494_(), pos.m_7495_(), pos.m_122012_(), pos.m_122019_(), pos.m_122029_(), pos.m_122024_());
    }

    private static class RedstoneSnapshot {
        private final ServerLevel level;
        private final BlockPos sourcePos;
        private final Map<BlockPos, BlockState> stateMap = new ConcurrentHashMap<BlockPos, BlockState>();
        private final Set<RedstoneTask> tasks = ConcurrentHashMap.newKeySet();
        private final long creationTime;

        public RedstoneSnapshot(ServerLevel level, BlockPos sourcePos) {
            this.level = level;
            this.sourcePos = sourcePos;
            this.creationTime = System.nanoTime();
            for (int x = -10; x <= 10; ++x) {
                for (int y = -10; y <= 10; ++y) {
                    for (int z = -10; z <= 10; ++z) {
                        BlockPos pos = sourcePos.m_7918_(x, y, z);
                        if (!level.m_46749_(pos)) continue;
                        this.stateMap.put(pos, level.m_8055_(pos));
                    }
                }
            }
        }

        public boolean isValid() {
            return !this.stateMap.isEmpty();
        }

        public boolean isStillValid() {
            return System.nanoTime() - this.creationTime < TimeUnit.MILLISECONDS.toNanos(100L);
        }

        public boolean containsPosition(BlockPos pos) {
            return this.stateMap.containsKey(pos);
        }

        public BlockState getBlockState(BlockPos pos) {
            return this.stateMap.get(pos);
        }

        public void assignTask(RedstoneTask task) {
            this.tasks.add(task);
        }

        public Set<RedstoneTask> assignedTasks() {
            return this.tasks;
        }

        public ServerLevel level() {
            return this.level;
        }

        public BlockPos sourcePos() {
            return this.sourcePos;
        }
    }

    private static class RedstoneTask {
        private final RedstoneSnapshot snapshot;
        private final List<RedstoneResult> results = new CopyOnWriteArrayList<RedstoneResult>();
        private final AtomicBoolean completed = new AtomicBoolean(false);
        private final AtomicReference<Exception> error = new AtomicReference();

        public RedstoneTask(RedstoneSnapshot snapshot) {
            this.snapshot = snapshot;
            snapshot.assignTask(this);
        }

        public RedstoneSnapshot snapshot() {
            return this.snapshot;
        }

        public List<RedstoneResult> results() {
            return this.results;
        }

        public AtomicBoolean completed() {
            return this.completed;
        }

        public AtomicReference<Exception> error() {
            return this.error;
        }
    }

    private record RedstoneResult(BlockPos pos, int power) {
    }
}

