/*
 * Decompiled with CFR 0.152.
 */
package net.carbonmc.graphene.event;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import net.carbonmc.graphene.config.CoolConfig;
import net.minecraftforge.eventbus.api.Event;
import net.minecraftforge.fml.loading.FMLEnvironment;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class AsyncEventSystem {
    public static final Logger LOGGER = LogManager.getLogger((String)"Graph-Async");
    private static final int CPU_CORES = Runtime.getRuntime().availableProcessors();
    private static final int MIN_POOL_SIZE = 2;
    private static final int DEFAULT_CORE_POOL_SIZE = Math.min(4, Math.max((Integer)CoolConfig.maxCPUPro.get(), CPU_CORES));
    private static final int MAX_POOL_SIZE = Math.min(16, CPU_CORES * 2);
    private static final long KEEP_ALIVE_TIME = 30L;
    private static final int WORK_QUEUE_SIZE = 2000;
    private static final long POOL_ADJUST_INTERVAL_MS = 30000L;
    private static final long SLOW_TASK_THRESHOLD_NS = TimeUnit.MILLISECONDS.toNanos(100L);
    private static final int MAX_FAILURES_BEFORE_DISABLE = 3;
    private static final long FAILURE_COOLDOWN_MS = 60000L;
    private static final ThreadPoolExecutor ASYNC_EXECUTOR = AsyncEventSystem.createThreadPool();
    private static final ConcurrentHashMap<Class<? extends Event>, EventTypeInfo> EVENT_TYPE_INFOS = new ConcurrentHashMap(64);
    private static volatile boolean initialized = false;
    private static final AtomicLong totalAsyncTasks = new AtomicLong(0L);
    private static final AtomicLong failedAsyncTasks = new AtomicLong(0L);
    private static final AtomicLong lastAdjustTime = new AtomicLong(System.currentTimeMillis());

    private static ThreadPoolExecutor createThreadPool() {
        return new ThreadPoolExecutor(DEFAULT_CORE_POOL_SIZE, MAX_POOL_SIZE, 30L, TimeUnit.SECONDS, (BlockingQueue)new LinkedBlockingQueue(2000), (ThreadFactory)new AsyncEventThreadFactory(), (RejectedExecutionHandler)new AsyncEventRejectedHandler()){

            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                AsyncEventSystem.adjustPoolSize();
            }
        };
    }

    private static void adjustPoolSize() {
        long now = System.currentTimeMillis();
        if (now - lastAdjustTime.get() < 30000L) {
            return;
        }
        int activeCount = ASYNC_EXECUTOR.getActiveCount();
        int queueSize = ASYNC_EXECUTOR.getQueue().size();
        int newCoreSize = AsyncEventSystem.calculateNewPoolSize(activeCount, queueSize);
        ASYNC_EXECUTOR.setCorePoolSize(newCoreSize);
        lastAdjustTime.set(now);
    }

    private static int calculateNewPoolSize(int activeCount, int queueSize) {
        return Math.min(MAX_POOL_SIZE, Math.max(2, activeCount + queueSize / 100));
    }

    public static void initialize() {
        if (!CoolConfig.isEnabled() || initialized) {
            return;
        }
        AsyncEventSystem.registerCommonAsyncEvents();
        initialized = true;
        LOGGER.info("Initialized with core: {}, max: {}, queue: {}", (Object)DEFAULT_CORE_POOL_SIZE, (Object)MAX_POOL_SIZE, (Object)2000);
    }

    private static void registerCommonAsyncEvents() {
        if (!CoolConfig.isEnabled()) {
            return;
        }
        String[] asyncEvents = new String[]{"net.minecraftforge.event.entity.player.PlayerEvent", "net.minecraftforge.event.entity.player.AdvancementEvent", "net.minecraftforge.event.entity.player.AnvilRepairEvent", "net.minecraftforge.event.entity.player.PlayerInteractEvent", "net.minecraftforge.event.entity.player.PlayerXpEvent", "net.minecraftforge.event.level.BlockEvent", "net.minecraftforge.event.level.ChunkEvent", "net.minecraftforge.event.level.ExplosionEvent", "net.minecraftforge.event.entity.EntityEvent", "net.minecraftforge.event.entity.EntityJoinLevelEvent", "net.minecraftforge.event.entity.EntityLeaveLevelEvent", "net.minecraftforge.event.entity.EntityMountEvent", "net.minecraftforge.event.entity.EntityTeleportEvent", "net.minecraftforge.event.entity.item.ItemEvent", "net.minecraftforge.event.entity.item.ItemExpireEvent", "net.minecraftforge.event.entity.item.ItemTossEvent", "net.minecraftforge.event.level.LevelEvent", "net.minecraftforge.event.level.BlockEvent", "net.minecraftforge.event.level.ChunkEvent", "net.minecraftforge.event.network.CustomPayloadEvent", "net.minecraftforge.event.CommandEvent", "net.minecraftforge.event.TagsUpdatedEvent", "net.minecraftforge.event.LootTableLoadEvent", "net.minecraftforge.event.RegisterCommandsEvent"};
        String[] syncEvents = new String[]{"net.minecraftforge.event.TickEvent", "net.minecraftforge.event.level.LevelTickEvent", "net.minecraftforge.event.entity.living.LivingEvent", "net.minecraftforge.event.entity.living.LivingAttackEvent", "net.minecraftforge.event.entity.living.LivingDamageEvent", "net.minecraftforge.event.entity.living.LivingDeathEvent", "net.minecraftforge.event.entity.living.LivingDropsEvent", "net.minecraftforge.event.entity.living.LivingExperienceDropEvent", "net.minecraftforge.event.entity.living.LivingHealEvent", "net.minecraftforge.event.entity.living.LivingKnockBackEvent", "net.minecraftforge.event.server.ServerStartingEvent", "net.minecraftforge.event.server.ServerStoppingEvent", "net.minecraftforge.event.server.ServerStartedEvent"};
        AsyncEventSystem.registerEvents(asyncEvents, true);
        AsyncEventSystem.registerEvents(syncEvents, false);
        LOGGER.info("Registered {} async event types", (Object)EVENT_TYPE_INFOS.size());
    }

    private static void registerEvents(String[] classNames, boolean async) {
        for (String className : classNames) {
            try {
                Class<? extends Event> eventClass = AsyncEventSystem.loadEventClass(className);
                if (async && AsyncEventSystem.isClientOnlyEvent(eventClass)) {
                    LOGGER.debug("Skipping client event: {}", (Object)className);
                    continue;
                }
                if (async) {
                    AsyncEventSystem.registerAsyncEvent(eventClass);
                    continue;
                }
                AsyncEventSystem.registerSyncEvent(eventClass);
            }
            catch (ClassNotFoundException e) {
                AsyncEventSystem.handleEventRegistrationError(className, async, e);
            }
        }
    }

    private static Class<? extends Event> loadEventClass(String className) throws ClassNotFoundException {
        try {
            Class<?> clazz = Class.forName(className);
            if (!Event.class.isAssignableFrom(clazz)) {
                throw new IllegalArgumentException("Class does not extend Event: " + className);
            }
            return clazz;
        }
        catch (ClassNotFoundException e) {
            ClassLoader forgeLoader = Event.class.getClassLoader();
            Class<?> clazz = Class.forName(className, true, forgeLoader);
            return clazz;
        }
    }

    private static void handleEventRegistrationError(String className, boolean async, ClassNotFoundException e) {
        if (async) {
            LOGGER.warn("[Fallback] Failed to load async event: {}, falling back to SYNC", (Object)className);
            try {
                Class<? extends Event> eventClass = AsyncEventSystem.loadEventClass(className);
                AsyncEventSystem.registerSyncEvent(eventClass);
            }
            catch (ClassNotFoundException ex) {
                LOGGER.error("[Critical] Event class not found: {}", (Object)className);
            }
        } else {
            LOGGER.error("[Critical] Sync event class not found: {}", (Object)className);
        }
    }

    public static boolean isClientOnlyEvent(Class<? extends Event> eventClass) {
        String name = eventClass.getName();
        return name.startsWith("client") || name.contains(".client.") || name.startsWith("net.minecraft.client.");
    }

    public static void registerAsyncEvent(Class<? extends Event> eventType) {
        if (!CoolConfig.isEnabled()) {
            return;
        }
        EVENT_TYPE_INFOS.compute(eventType, (k, v) -> {
            EventTypeInfo info = v == null ? new EventTypeInfo(true) : v;
            info.async = true;
            info.healthy = true;
            info.failedCount.set(0);
            info.isClientEvent = AsyncEventSystem.isClientOnlyEvent(eventType);
            return info;
        });
        LOGGER.debug("Registered async event: {}", (Object)eventType.getName());
    }

    public static void registerSyncEvent(Class<? extends Event> eventType) {
        if (!CoolConfig.isEnabled()) {
            return;
        }
        EVENT_TYPE_INFOS.compute(eventType, (k, v) -> {
            EventTypeInfo info = v == null ? new EventTypeInfo(false) : v;
            info.async = false;
            info.isClientEvent = AsyncEventSystem.isClientOnlyEvent(eventType);
            return info;
        });
        LOGGER.debug("Registered sync event: {}", (Object)eventType.getName());
    }

    public static boolean shouldHandleAsync(Class<? extends Event> eventType) {
        EventTypeInfo info = EVENT_TYPE_INFOS.get(eventType);
        if (info == null) {
            return eventType.getSimpleName().contains("Async");
        }
        return (!info.isClientEvent || !FMLEnvironment.dist.isDedicatedServer()) && info.async && info.healthy;
    }

    public static CompletableFuture<Void> executeAsync(Class<? extends Event> eventType, Runnable task) {
        if (!CoolConfig.isEnabled() || AsyncEventSystem.shouldExecuteImmediately(eventType)) {
            task.run();
            return CompletableFuture.completedFuture(null);
        }
        totalAsyncTasks.incrementAndGet();
        EventTypeInfo info = AsyncEventSystem.getEventTypeInfo(eventType);
        if (!info.async || !info.healthy) {
            task.run();
            return CompletableFuture.completedFuture(null);
        }
        return AsyncEventSystem.submitAsyncTask(eventType, task, info);
    }

    private static CompletableFuture<Void> submitAsyncTask(Class<? extends Event> eventType, Runnable task, EventTypeInfo info) {
        info.pendingTasks.incrementAndGet();
        ClassLoader contextLoader = Thread.currentThread().getContextClassLoader();
        return CompletableFuture.runAsync(() -> {
            Thread.currentThread().setContextClassLoader(contextLoader);
            try {
                AsyncEventSystem.executeAndMonitorTask(eventType, task);
            }
            catch (Throwable t) {
                AsyncEventSystem.handleTaskFailure(eventType, info, t);
                throw t;
            }
            finally {
                info.pendingTasks.decrementAndGet();
            }
        }, ASYNC_EXECUTOR).exceptionally(ex -> {
            LOGGER.warn("Retrying {} synchronously", (Object)eventType.getSimpleName());
            task.run();
            return null;
        });
    }

    private static boolean shouldExecuteImmediately(Class<? extends Event> eventType) {
        return eventType.getName().contains("Client") && FMLEnvironment.dist.isDedicatedServer() || !initialized;
    }

    private static EventTypeInfo getEventTypeInfo(Class<? extends Event> eventType) {
        return EVENT_TYPE_INFOS.computeIfAbsent(eventType, k -> new EventTypeInfo(AsyncEventSystem.shouldHandleAsync(eventType)));
    }

    private static void executeAndMonitorTask(Class<? extends Event> eventType, Runnable task) {
        long startTime = System.nanoTime();
        task.run();
        long elapsed = System.nanoTime() - startTime;
        if (elapsed > SLOW_TASK_THRESHOLD_NS) {
            LOGGER.debug("Slow task {}: {}ms", (Object)eventType.getSimpleName(), (Object)TimeUnit.NANOSECONDS.toMillis(elapsed));
        }
    }

    private static void handleTaskFailure(Class<? extends Event> eventType, EventTypeInfo info, Throwable t) {
        failedAsyncTasks.incrementAndGet();
        info.failedCount.incrementAndGet();
        info.lastFailureTime = System.currentTimeMillis();
        LOGGER.error("Task failed: {}", (Object)eventType.getSimpleName(), (Object)t);
        if (((Boolean)CoolConfig.DISABLE_ASYNC_ON_ERROR.get()).booleanValue() || info.failedCount.get() >= 3) {
            info.healthy = false;
            LOGGER.warn("Disabled async for {}", (Object)eventType.getSimpleName());
        }
    }

    public static void shutdown() {
        if (!CoolConfig.isEnabled()) {
            return;
        }
        if (!initialized) {
            return;
        }
        LOGGER.info("Shutting down async event system. Total tasks: {}, Failed: {}", (Object)totalAsyncTasks.get(), (Object)failedAsyncTasks.get());
        ASYNC_EXECUTOR.shutdown();
        try {
            if (!ASYNC_EXECUTOR.awaitTermination(10L, TimeUnit.SECONDS)) {
                LOGGER.warn("Forcing async event executor shutdown");
                ASYNC_EXECUTOR.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            ASYNC_EXECUTOR.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public static int getQueueSize() {
        return ASYNC_EXECUTOR.getQueue().size();
    }

    public static int getActiveCount() {
        return ASYNC_EXECUTOR.getActiveCount();
    }

    public static int getPoolSize() {
        return ASYNC_EXECUTOR.getPoolSize();
    }

    public static int getMaxPoolSize() {
        return MAX_POOL_SIZE;
    }

    public static int getAsyncEventCount() {
        return EVENT_TYPE_INFOS.size();
    }

    public static void tryRegisterAsyncEvent(Consumer<?> consumer) {
        try {
            for (Type type : consumer.getClass().getGenericInterfaces()) {
                Class eventClass;
                Type[] typeArgs;
                ParameterizedType paramType;
                if (!(type instanceof ParameterizedType) || !(paramType = (ParameterizedType)type).getRawType().equals(Consumer.class) || (typeArgs = paramType.getActualTypeArguments()).length <= 0 || !(typeArgs[0] instanceof Class) || !Event.class.isAssignableFrom(eventClass = (Class)typeArgs[0])) continue;
                Class eventType = eventClass;
                AsyncEventSystem.registerAsyncEvent(eventType);
            }
        }
        catch (Exception e) {
            LOGGER.warn("Failed to determine event type for consumer: {}", (Object)consumer.getClass().getName(), (Object)e);
        }
    }

    public static void resetEventTypeHealth(Class<? extends Event> eventType) {
        EVENT_TYPE_INFOS.computeIfPresent(eventType, (k, v) -> {
            v.healthy = true;
            v.failedCount.set(0);
            return v;
        });
    }

    static {
        ASYNC_EXECUTOR.prestartAllCoreThreads();
    }

    private static class AsyncEventThreadFactory
    implements ThreadFactory {
        private final AtomicInteger counter = new AtomicInteger(0);

        private AsyncEventThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r, "Graphene-Async-Worker-" + this.counter.incrementAndGet());
            thread.setDaemon(true);
            thread.setPriority(4);
            thread.setUncaughtExceptionHandler((t, e) -> LOGGER.error("Uncaught exception in async thread", e));
            return thread;
        }
    }

    private static class AsyncEventRejectedHandler
    implements RejectedExecutionHandler {
        private AsyncEventRejectedHandler() {
        }

        @Override
        public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
            LOGGER.warn("Queue full ({} tasks), executing on caller thread", (Object)executor.getQueue().size());
            if (!executor.isShutdown()) {
                r.run();
            }
        }
    }

    private static class EventTypeInfo {
        volatile boolean async;
        volatile boolean healthy = true;
        final AtomicInteger pendingTasks = new AtomicInteger(0);
        final AtomicInteger failedCount = new AtomicInteger(0);
        volatile boolean isClientEvent = false;
        volatile long lastFailureTime = 0L;

        EventTypeInfo(boolean async) {
            this.async = async;
        }

        boolean shouldRetryAsync() {
            return this.async && this.healthy && (this.failedCount.get() < 3 || System.currentTimeMillis() - this.lastFailureTime > 60000L);
        }
    }
}

