/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.map;

import io.github.axolotlclient.AxolotlClientConfig.impl.util.DrawUtil;
import io.github.axolotlclient.bridge.render.AxoRenderContext;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.AxolotlClientWaypointsCommon;
import io.github.axolotlclient.waypoints.HudCreator;
import io.github.axolotlclient.waypoints.util.ARGB;
import io.github.axolotlclient.waypoints.waypoints.Waypoint;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_2441996;
import net.minecraft.unmapped.C_2691939;
import net.minecraft.unmapped.C_3628668;
import net.minecraft.unmapped.C_3674802;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3779718;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_6849228;
import net.minecraft.unmapped.C_7799337;
import net.minecraft.unmapped.C_8049104;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_9335057;
import net.ornithemc.osl.lifecycle.api.client.MinecraftClientEvents;
import org.joml.Matrix4fStack;
import org.joml.Vector3f;

public class Minimap extends MinimapCommon {

	private static final C_0561170 texLocation = AxolotlClientWaypoints.rl("minimap");
	public static final C_0561170 arrowLocation = AxolotlClientWaypoints.rl("textures/gui/sprites/arrow.png");
	private int[] pixels;
	public long updateDuration = -1;
	private C_3779718 tex;
	private int mapCenterX, mapCenterZ;
	private boolean usingHud;
	public boolean allowCaves = true;
	private final Matrix4fStack matrixStack = new Matrix4fStack(5);

	private final C_8105098 minecraft = C_8105098.m_0408063();

	public void init() {
		minimap.add(enabled, lockMapToNorth, arrowScale, minimapOutline, outlineColor, enableBiomeBlending, mapScale, showWaypoints, showCardinalDirections);
		AxolotlClientWaypoints.category.add(Minimap.minimap);
		if (AxolotlClientWaypointsCommon.AXOLOTLCLIENT_PRESENT) {
			usingHud = true;
			var save = HudCreator.createHud(this);
			MinecraftClientEvents.STOP.register(mc -> save.run());
		}
	}

	public boolean isEnabled() {
		return enabled.get();
	}

	public void setup() {
		minecraft.m_1218956().m_0381205(texLocation, tex = new C_3779718(viewDistance, viewDistance));
		pixels = tex.m_1823577();
		this.x = new C_7799337(minecraft).m_1713651() - size - 10;
		this.y = 10;
	}

	public void renderMapOverlay() {
		if (usingHud) {
			return;
		}
		var guiWidth = new C_7799337(minecraft).m_1713651();
		this.x = guiWidth - size - 10;
		this.y = 10;
		if (minecraft.m_7633905()) {
			this.y += 15;
		}
		/*if (!minecraft.player.getActiveEffects().isEmpty() && (minecraft.screen == null ||
			!(this.minecraft.screen instanceof EffectRenderingInventoryScreen<?> effectRenderingInventoryScreen && effectRenderingInventoryScreen.canSeeEffects()))) {
			if (minecraft.player.getActiveEffects().stream().anyMatch(e -> !e.getEffect().value().isBeneficial())) {
				this.y += 26;
			}
			this.y += 20;
		}*/
		renderMap();
	}

	public void renderMap(AxoRenderContext ctx) {
		renderMap();
	}

	public void renderMap() {
		if (!isEnabled()) {
			return;
		}
		matrixStack.clear().pushMatrix();
		C_3754158.m_8373640();
		{
			DrawUtil.pushScissor(x, y, size, size);
			C_3754158.m_8373640();
			C_3754158.m_3172490(x, y, 0);
			C_3754158.m_3172490(radius, radius, 0);
			if (!lockMapToNorth.get()) {
				C_3754158.m_8616673(-(minecraft.f_7663840.m_4815215() + 180), 0, 0, 1);
			}
			C_3754158.m_4552250((float) Math.sqrt(2), (float) Math.sqrt(2), 1);
			C_3754158.m_4552250(mapScale.get(), mapScale.get(), 1);
			C_3754158.m_3172490(-radius, -radius, 0);
			float offX, offZ;
			offX = -(float) (minecraft.f_7663840.f_6638345 - mapCenterX);
			offZ = -(float) (minecraft.f_7663840.f_9103758 - mapCenterZ);
			C_3754158.m_3172490(offX, offZ, 0);
			minecraft.m_1218956().m_5325521(texLocation);
			C_2691939.m_5935491(0, 0, 0, 0, viewDistance, viewDistance, size, size);
			C_3754158.m_2041265();
			DrawUtil.popScissor();
		}

		if (minimapOutline.get() && !usingHud) {
			DrawUtil.outlineRect(x, y, size, size, outlineColor.get().toInt());
		}
		if (showWaypoints.get()) {
			renderMapWaypoints();
		}
		if (showCardinalDirections.get()) {
			Vector3f pos = new Vector3f();
			matrixStack.pushMatrix();
			var directions = new String[]{"N", "W", "E", "S"};
			for (int i : new int[]{-2, 1, 2, -1}) {
				var label = directions[i < 0 ? i + 2 : i + 1];
				var labelWidth = minecraft.f_0426313.m_0040387(label);
				var labelHeight = minecraft.f_0426313.f_6725889;
				matrixStack.pushMatrix();
				matrixStack.identity();
				matrixStack.translate(x + radius, y + radius, 0);
				if (!lockMapToNorth.get()) {
					matrixStack.rotate((float) -(((minecraft.f_7663840.m_4815215() + 180) / 180) * Math.PI), 0, 0, 1);
				}
				matrixStack.translate((i % 2) * size, ((int) (i / 2f)) * size, 0);
				pos.zero();
				matrixStack.transformPosition(pos);
				matrixStack.popMatrix();
				pos.x = C_4976084.m_7164829(pos.x, x, x + size);
				pos.y = C_4976084.m_7164829(pos.y, y, y + size);
				C_3754158.m_8373640();
				C_3754158.m_3172490(pos.x, pos.y, pos.z);
				C_3754158.m_4552250(0.5f, 0.5f, 1);
				C_2691939.m_7865719(-(labelWidth / 2 + 2), -(labelHeight / 2 + 2), labelWidth / 2 + 2, labelHeight / 2 + 2, 0x77888888);
				minecraft.f_0426313.m_4683674(label, -labelWidth / 2, -labelHeight / 2, -1);
				C_3754158.m_2041265();
			}
			matrixStack.popMatrix();
		}

		C_3754158.m_8373640();
		C_3754158.m_3172490(x + radius, y + radius, 0);
		if (lockMapToNorth.get()) {
			C_3754158.m_8616673(minecraft.f_7663840.m_4815215() + 180, 0, 0, 1);
		}
		C_3754158.m_4552250(0.5f * arrowScale.get(), 0.5f * arrowScale.get(), 1);
		int arrowSize = 15;
		C_3754158.m_3172490(-arrowSize / 2f, -arrowSize / 2f, 1);
		minecraft.m_1218956().m_5325521(arrowLocation);
		C_2691939.m_5935491(0, 0, 0, 0, arrowSize, arrowSize, arrowSize, arrowSize);
		C_3754158.m_2041265();

		matrixStack.popMatrix();
		C_3754158.m_2041265();
	}

	private void renderMapWaypoints() {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		matrixStack.pushMatrix();
		C_3754158.m_8373640();
		Vector3f pos = new Vector3f();
		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			matrixStack.pushMatrix();
			C_3754158.m_8373640();
			float posX = (float) (waypoint.x() - minecraft.f_7663840.f_6638345);
			float posY = (float) (waypoint.z() - minecraft.f_7663840.f_9103758);

			{
				pos.zero();
				matrixStack.pushMatrix();
				matrixStack.identity();
				matrixStack.translate(x, y, 0);
				matrixStack.translate(radius, radius, 0);
				matrixStack.scale((float) Math.sqrt(2), (float) Math.sqrt(2), 1);
				matrixStack.scale(mapScale.get(), mapScale.get(), 1);
				if (!lockMapToNorth.get()) {
					matrixStack.rotate((float) -Math.toRadians(minecraft.f_7663840.f_7005130 + 180), 0, 0, 1);
				}
				matrixStack.translate(posX, posY, 1);
				matrixStack.transformPosition(pos);
				matrixStack.popMatrix();
			}

			{
				pos.x = C_4976084.m_7164829(pos.x, x, x + size);
				pos.y = C_4976084.m_7164829(pos.y, y, y + size);
				C_3754158.m_3172490(pos.x, pos.y, pos.z());
			}

			C_3754158.m_6326777(1, 1, 1);
			int textWidth = minecraft.f_0426313.m_0040387(waypoint.display());
			int textHeight = minecraft.f_0426313.f_6725889;
			C_2691939.m_7865719(-(textWidth / 2) - Waypoint.displayXOffset(), -(textHeight / 2) - Waypoint.displayYOffset(), (textWidth / 2) + Waypoint.displayXOffset(), (textHeight / 2) + Waypoint.displayYOffset(), waypoint.color().toInt());
			minecraft.f_0426313.m_4413321(waypoint.display(), -(textWidth / 2f), -textHeight / 2f, -1, false);
			matrixStack.popMatrix();
			C_3754158.m_2041265();
		}
		matrixStack.popMatrix();
		C_3754158.m_2041265();
	}

	private static int blockToSectionCoord(int c) {
		return c >> 4;
	}

	public void updateMapView() {
		if (!isEnabled()) {
			updateDuration = -1;
			return;
		}
		long start = System.currentTimeMillis();
		int centerX = (int) (minecraft.f_7663840.f_6638345 + 0.5);
		int centerZ = (int) (minecraft.f_7663840.f_9103758 + 0.5);
		mapCenterX = centerX;
		mapCenterZ = centerZ;
		int texHalfWidth = size / 2;

		C_3674802.C_0067708 mutableBlockPos = new C_3674802.C_0067708();
		C_3674802.C_0067708 mutableBlockPos2 = new C_3674802.C_0067708();

		var level = minecraft.f_4601986;
		var centerChunk = level.m_0750002(blockToSectionCoord(centerX), blockToSectionCoord(centerZ));
		var surface = centerChunk.m_5956776(centerX & 15, centerZ & 15);
		mutableBlockPos.m_1540202(centerX, surface, centerZ);
		int solidBlocksAbovePlayer = 0;
		boolean atSurface = false;
		if (level.f_6669533.m_8976359()) {
			atSurface = (int) (minecraft.f_7663840.f_1187082 + 0.5) >= level.m_8568350();
		} else if (surface + 1 <= (int) (minecraft.f_7663840.f_1187082 + 0.5)) {
			atSurface = true;
		} else {
			while (solidBlocksAbovePlayer <= 3 && surface > (int) (minecraft.f_7663840.f_1187082 + 0.5) && surface > WorldMapScreen.MIN_BUILD_HEIGHT) {
				C_2441996 state = centerChunk.m_9971171(mutableBlockPos);
				mutableBlockPos.m_1540202(centerX, surface--, centerZ);
				if (!(state.m_0999604().m_4952508() || !state.m_0999604().m_4839982() || !state.m_0999604().m_8740684())) {
					solidBlocksAbovePlayer++;
				}
			}
			if (solidBlocksAbovePlayer <= 2) {
				atSurface = true;
			}
		}

		boolean updated = false;
		for (int x = 0; x < size; x++) {
			double d = 0.0;
			for (int z = -1; z < size; z++) {
				int chunkX = (centerX + x - texHalfWidth);
				int chunkZ = (centerZ + z - texHalfWidth);
				C_6849228 levelChunk = level.m_0750002(blockToSectionCoord(chunkX), blockToSectionCoord(chunkZ));
				if (levelChunk != null) {
					int fluidDepth = 0;
					double e = 0.0;
					mutableBlockPos.m_1540202(chunkX, 0, chunkZ);
					int y = levelChunk.m_5956776(mutableBlockPos.m_9150363() & 15, mutableBlockPos.m_3900258() & 15) + 1;
					if (!atSurface) {
						y = Math.min(y, (int) (minecraft.f_7663840.f_1187082 + 0.5));
					}
					C_2441996 blockState;
					if (y <= WorldMapScreen.MIN_BUILD_HEIGHT) {
						blockState = C_3628668.f_3097723.m_9077732();
					} else {
						do {
							mutableBlockPos.m_1540202(chunkX, --y, chunkZ);
							blockState = levelChunk.m_9971171(mutableBlockPos);
						} while (blockState.m_0999604().m_3247759(blockState) == C_9335057.f_8898614 && y > WorldMapScreen.MIN_BUILD_HEIGHT);

						if (y > WorldMapScreen.MIN_BUILD_HEIGHT && blockState.m_0999604().m_8228353().m_8583295()) {
							int highestFullBlockY = y - 1;
							mutableBlockPos2.m_1540202(mutableBlockPos.m_9150363(), mutableBlockPos.m_4798774(), mutableBlockPos.m_3900258());

							C_2441996 blockState2;
							do {
								mutableBlockPos2.m_1540202(mutableBlockPos.m_9150363(), highestFullBlockY--, mutableBlockPos.m_3900258());
								blockState2 = levelChunk.m_9971171(mutableBlockPos2);
								fluidDepth++;
							} while (highestFullBlockY > WorldMapScreen.MIN_BUILD_HEIGHT && blockState2.m_0999604().m_8228353().m_8583295());
						}
					}

					e += y;
					var mapColor = blockState.m_0999604().m_3247759(blockState);

					int color;
					if (mapColor == C_9335057.f_2086259) {
						var floorBlock = levelChunk.m_9971171(mutableBlockPos2);
						var floorColor = floorBlock.m_0999604().m_3247759(floorBlock).f_4989652;
						int biomeColor = enableBiomeBlending.get() ? C_8049104.m_8522175(level, mutableBlockPos) : mapColor.f_4989652;
						float shade = 1.0f;
						int waterColor = biomeColor;
						waterColor = ARGB.colorFromFloat(1f, ARGB.redFloat(waterColor) * shade, ARGB.greenFloat(waterColor) * shade, ARGB.blueFloat(waterColor) * shade);
						waterColor = ARGB.average(waterColor, ARGB.scaleRGB(floorColor, 1f - fluidDepth / 15f));
						color = waterColor;
					} else {
						double f = (e - d) * 4.0 / (1 + 4) + ((x + z & 1) - 0.5) * 0.4;
						int brightness;
						if (f > 0.6) {
							brightness = 2;
						} else if (f < -0.6) {
							brightness = 0;
						} else {
							brightness = 1;
						}
						color = mapColor.m_4724233(brightness);
					}

					d = e;

					if (z >= 0 && pixels[x + z * size] != color) {
						pixels[x + z * size] = ARGB.opaque(color);
						updated = true;
					}
				} else {
					if (z >= 0 && pixels[x + z * size] != 0) {
						pixels[x + z * size] = ARGB.opaque(0);
						updated = true;
					}
				}
			}
		}
		if (updated) {
			tex.m_4420833();
		}
		updateDuration = System.currentTimeMillis() - start;
	}
}
