/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.map;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.mojang.blaze3d.buffers.GpuBuffer;
import com.mojang.blaze3d.systems.CommandEncoder;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.util.ARGB;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.fabricmc.fabric.api.resource.v1.ResourceLoader;
import net.minecraft.class_1011;
import net.minecraft.class_1058;
import net.minecraft.class_1087;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3264;
import net.minecraft.class_3486;
import net.minecraft.class_3620;
import net.minecraft.class_4013;

public class TextureSampler {
	private static final class_310 minecraft = class_310.method_1551();

	private static final Object2IntMap<class_2680> sampleCache = new Object2IntOpenHashMap<>();
	private static final Object2ObjectMap<class_2960, class_1011> atlasCache = new Object2ObjectOpenHashMap<>();

	private static boolean sampling;
	private static final List<Supplier<CompletableFuture<Integer>>> sampleQueue = new ArrayList<>();

	static {
		ResourceLoader.get(class_3264.field_14188).registerReloader(AxolotlClientWaypoints.rl("reload_listener"),
			(class_4013) rm -> {
				atlasCache.values().forEach(class_1011::close);
				atlasCache.clear();
				sampleCache.clear();
				sampleQueue.clear();
				sampling = false;
			});
	}

	public static CompletableFuture<Integer> getSample(class_2680 state, class_1937 level, class_2338 pos, class_3620.class_6594 brightness) {
		if (state.method_26215()) {
			return CompletableFuture.completedFuture(0);
		}
		if (sampleCache.containsKey(state)) {
			return CompletableFuture.completedFuture(applyTransforms(sampleCache.getInt(state), state, level, pos, brightness));
		}

		Supplier<CompletableFuture<Integer>> sup = () -> {
			if (sampleCache.containsKey(state)) {
				int cached = sampleCache.getInt(state);
				if (!sampleQueue.isEmpty()) {
					var list = sampleQueue.toArray(Supplier[]::new);
					sampleQueue.clear();
					for (var e : list) {
						e.get();
					}
				}
				sampling = false;
				return CompletableFuture.completedFuture(applyTransforms(cached, state, level, pos, brightness));
			}
			class_1087 model = minecraft.method_1554().method_4743().method_3335(state);
			var tex = model.method_68511();
			return downloadAtlas(tex.method_45852()).thenApply(i -> sampleSprite(i, tex))
				.thenApply(x -> {
					sampleCache.put(state, (int) x);
					if (!sampleQueue.isEmpty()) {
						var list = sampleQueue.toArray(Supplier[]::new);
						sampleQueue.clear();
						for (var e : list) {
							e.get();
						}
					}
					sampling = false;

					return applyTransforms(x, state, level, pos, brightness);
				});
		};
		if (!sampling) {
			sampling = true;
			return sup.get();
		} else {
			CompletableFuture<Integer> cf = new CompletableFuture<>();
			sampleQueue.add(() -> sup.get().thenApply(i -> {
				cf.complete(i);
				return i;
			}));
			return cf;
		}
	}

	private static int applyTransforms(int color, class_2680 state, class_1937 level, class_2338 pos, class_3620.class_6594 brightness) {
		int col = minecraft.method_1505().method_1697(state, level, pos, 0);
		var fluid = state.method_26227();
		if (!fluid.method_15769() && fluid.method_15767(class_3486.field_15517)) {
			int b = minecraft.field_1687.method_23753(pos).comp_349().method_8687();
			float shade = minecraft.field_1687.method_24852(class_2350.field_11036, true);
			int f = ARGB.colorFromFloat(1f, ARGB.redFloat(b) * shade, ARGB.greenFloat(b) * shade, ARGB.blueFloat(b) * shade);
			color = ARGB.color(ARGB.alpha(color), f);
		}
		if (col != -1 && isGrayscaleColor(color)) {
			color = ARGB.color(ARGB.alpha(color), col);
		}
		return ARGB.scaleRGB(color, brightness.field_34764);
	}

	private static boolean isGrayscaleColor(int color) {
		return ARGB.red(color) == ARGB.green(color) && ARGB.red(color) == ARGB.blue(color);
	}

	private static CompletableFuture<class_1011> downloadAtlas(class_2960 loc) {
		if (atlasCache.containsKey(loc)) {
			return CompletableFuture.completedFuture(atlasCache.get(loc));
		}
		RenderSystem.assertOnRenderThread();
		var cf = new CompletableFuture<class_1011>();

		var gpuTexture = minecraft.method_1531().method_4619(loc).method_68004();
		int bufferSize = gpuTexture.getFormat().pixelSize() * gpuTexture.getWidth(0) * gpuTexture.getHeight(0);

		GpuBuffer gpuBuffer = RenderSystem.getDevice().createBuffer(() -> "Texture Atlas buffer", 9, bufferSize);
		CommandEncoder commandEncoder = RenderSystem.getDevice().createCommandEncoder();

		commandEncoder.copyTextureToBuffer(gpuTexture, gpuBuffer, 0, () -> {
			try (GpuBuffer.MappedView mappedView = commandEncoder.mapBuffer(gpuBuffer, true, false)) {

				int textureWidth = gpuTexture.getWidth(0);
				int textureHeight = gpuTexture.getHeight(0);

				var img = new class_1011(textureWidth, textureHeight, false);

				for (int n = 0; n < textureHeight; n++) {
					for (int o = 0; o < textureWidth; o++) {
						img.method_4305(o, n, mappedView.data().getInt((o + n * textureWidth) * gpuTexture.getFormat().pixelSize()));
					}
				}

				atlasCache.put(loc, img);
				cf.complete(img);
			}
			gpuBuffer.close();
		}, 0);
		return cf;
	}

	private static int sampleSprite(class_1011 image, class_1058 sprite) {
		int sampled = 0;

		for (int n = sprite.method_35807(); n < sprite.method_35807() + sprite.method_45851().method_45815(); n++) {
			for (int o = sprite.method_35806(); o < sprite.method_35806() + sprite.method_45851().method_45807(); o++) {
				int c = image.method_61940(o, n);
				sampled = getColorAvg(sampled, c);
			}
		}
		return sampled;
	}

	private static int getColorAvg(int a, int b) {
		if (a == 0) {
			return b;
		} else if (ARGB.alpha(b) == 0) {
			return a;
		} else {
			return ARGB.average(a, b);
		}
	}
}
