/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.waypoints;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import net.minecraft.class_10209;
import net.minecraft.class_10366;
import net.minecraft.class_10799;
import net.minecraft.class_12247;
import net.minecraft.class_1921;
import net.minecraft.class_243;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_4184;
import net.minecraft.class_4587;
import net.minecraft.class_4597;
import net.minecraft.class_9779;
import net.minecraft.class_9909;
import net.minecraft.class_9922;
import net.minecraft.class_9960;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.DrawUtil;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.mixin.GameRendererAccessor;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;
import org.joml.Quaternionf;
import org.joml.Vector2f;
import org.joml.Vector4f;

public class WaypointRenderer {

	private static final double CUTOFF_DIST = 5;
	private final class_310 minecraft = class_310.method_1551();
	private final class_1921 QUADS = class_1921.method_75940("waypoint_quads", class_12247.method_75927(class_10799.field_56879)
		.method_75929(192*8).method_75938());
	private final Matrix4f view = new Matrix4f();
	private final Vector4f viewProj = new Vector4f();
	private final Set<Waypoint> worldRendererWaypoints = new HashSet<>();

	public void render(class_9922 allocator, class_9960 targets, class_9779 deltaTracker) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		if (minecraft.field_1687 == null) return;
		var profiler = class_10209.method_64146();
		profiler.method_15405("waypoints");
		float f = deltaTracker.method_60637(true);
		class_9909 frameGraphBuilder = new class_9909();
		targets.field_53091 = frameGraphBuilder.method_61914("main", this.minecraft.method_1522());
		var pass = frameGraphBuilder.method_61911("waypoints");
		targets.field_53091 = pass.method_61933(targets.field_53091);
		pass.method_61929(() -> {
			class_4597.class_4598 bufferSource = minecraft.method_22940().method_23000();
			float fov = ((GameRendererAccessor) minecraft.field_1773).invokeGetFov(minecraft.field_1773.method_19418(), f, true);
			RenderSystem.setProjectionMatrix(((GameRendererAccessor) minecraft.field_1773).getHud3dProjectionMatrixBuffer().method_71095(this.minecraft.method_22683().method_4489(),
				this.minecraft.method_22683().method_4506(), fov), class_10366.field_54953);
			var stack = new class_4587();
			var cam = minecraft.field_1773.method_19418();

			stack.method_22903();
			stack.method_22907(cam.method_23767().invert());
			var camPos = AxolotlClientWaypoints.WAYPOINT_RENDERER.minecraft.field_1773.method_19418().method_71156();
			worldRendererWaypoints.clear();

			for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
				profiler.method_15396(waypoint.name());
				renderWaypoint(waypoint, stack, camPos, cam, bufferSource, fov);
				profiler.method_15407();
			}

			stack.method_22909();
			bufferSource.method_37104();
			if (!stack.method_67795()) {
				throw new IllegalStateException("Pose stack not empty");
			}
		});
		frameGraphBuilder.method_61909(allocator);
	}

	private void renderWaypoint(Waypoint waypoint, class_4587 stack, class_243 camPos, class_4184 cam, class_4597.class_4598 bufferSource, float fov) {
		int textWidth = minecraft.field_1772.method_1727(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.field_1772.field_2000;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		var displayStart = projectToScreen(cam, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(-(width / 2f * 0.04f), (height / 2f * 0.04f)));
		if (displayStart == null) return;
		var displayEnd = projectToScreen(cam, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(width / 2f * 0.04f, -(height / 2f * 0.04f)));
		if (displayEnd == null) return;
		float projWidth = Math.abs(displayEnd.x() - displayStart.x());
		float projHeight = Math.abs(displayEnd.y() - displayStart.y());
		if (projWidth < width && projHeight < height) {
			return;
		}
		worldRendererWaypoints.add(waypoint);

		stack.method_22903();
		stack.method_22904(waypoint.x() - camPos.method_10216(), waypoint.y() - camPos.method_10214(), waypoint.z() - camPos.method_10215());
		stack.method_22907(cam.method_23767().invert(new Quaternionf()));
		float scale = 0.04F;
		stack.method_22905(scale, -scale, scale);
		fillRect(stack, bufferSource, -width / 2f, -height / 2f, 0.00f, width / 2f, height / 2f, waypoint.color().toInt());
		drawFontBatch(waypoint.display(), -textWidth / 2f, -textHeight / 2f, stack.method_23760().method_23761(), bufferSource);
		stack.method_22909();
	}

	private void fillRect(class_4587 stack, class_4597.class_4598 source, float x, float y, float z, float x2, float y2, int color) {
		var buf = source.method_73477(QUADS);
		var matrix = stack.method_23760().method_23761();
		buf.method_22918(matrix, x, y, z).method_39415(color);
		buf.method_22918(matrix, x, y2, z).method_39415(color);
		buf.method_22918(matrix, x2, y2, z).method_39415(color);
		buf.method_22918(matrix, x2, y, z).method_39415(color);
	}

	private void drawFontBatch(String text, float x, float y, Matrix4f matrix, class_4597 bufferSource) {
		minecraft.field_1772.method_27521(text, x, y, -1, false, matrix, bufferSource, class_327.class_6415.field_33993, 0, 0xF000F0);
	}

	public void renderWaypoints(class_332 graphics, class_9779 deltaTracker) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		var profiler = class_10209.method_64146();
		var cam = minecraft.field_1773.method_19418();
		profiler.method_15396("waypoints");

		graphics.method_51448().pushMatrix();
		var waypoints = AxolotlClientWaypoints.getCurrentWaypoints();
		var positionDrawers = new ArrayList<Runnable>(waypoints.size());
		for (Waypoint waypoint : waypoints) {
			graphics.method_51448().pushMatrix();
			renderWaypoint(waypoint, graphics, deltaTracker, cam, positionDrawers);
			graphics.method_51448().popMatrix();
		}
		if (!positionDrawers.isEmpty()) {
			positionDrawers.forEach(Runnable::run);
		}
		graphics.method_51448().popMatrix();
		profiler.method_15407();
	}

	private void renderWaypoint(Waypoint waypoint, class_332 graphics, class_9779 tracker, class_4184 camera, List<Runnable> positionDrawers) {
		var tick = tracker.method_60637(true);
		var fov = ((GameRendererAccessor) minecraft.field_1773).invokeGetFov(camera, tick, true);
		var pose = graphics.method_51448();

		var textWidth = minecraft.field_1772.method_1727(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.field_1772.field_2000;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		var camPos = camera.method_71156();

		var displayStart = projectToScreen(camera, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(-(width / 2f * 0.04f), (height / 2f * 0.04f)));
		var displayEnd = projectToScreen(camera, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f((width / 2f * 0.04f), -(height / 2f * 0.04f)));
		Result result = projectToScreen(camera, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), null);
		if (result == null) return;
		float projWidth;
		float projHeight;
		if (displayStart != null && displayEnd != null) {
			projWidth = Math.abs(displayEnd.x() - displayStart.x());
			projHeight = Math.abs(displayEnd.y() - displayStart.y());
		} else {
			projWidth = 0;
			projHeight = 0;
		}

		pose.translate(result.x(), result.y());
		boolean outOfView = result.x() < -width / 2f || result.x() > graphics.method_51421() + width / 2f || result.y() < -height / 2f || result.y() > graphics.method_51443() + height / 2f;
		if (!AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get() && outOfView) {
			return;
		}

		boolean _3dOnScreen;
		if (displayEnd != null && displayStart != null) {
			float minX = displayStart.x();
			float minY = displayStart.y();
			float maxX = displayEnd.x();
			float maxY = displayEnd.y();
			int guiWidth = graphics.method_51421();
			int guiHeight = graphics.method_51443();
			_3dOnScreen = minX > 0 && minY > 0 && minX < guiWidth && minY < guiHeight ||
				minX > 0 && maxY > 0 && minX < guiWidth && maxY < guiHeight ||
				maxX > 0 && maxY > 0 && maxX < guiWidth && maxY < guiHeight ||
				maxX > 0 && minY > 0 && maxX < guiWidth && minY < guiHeight ||
				minX < guiWidth && maxX > 0 && minY < guiHeight && maxY > 0;
		} else {
			_3dOnScreen = false;
		}

		boolean displayX = Math.abs(result.x() - graphics.method_51421() / 2f) < (_3dOnScreen ? Math.max(projWidth, width) : width) / 2f + graphics.method_51421() / 4f;
		boolean displayY = Math.abs(result.y() - graphics.method_51443() / 2f) < (_3dOnScreen ? Math.max(height, projHeight) : height) / 2f + graphics.method_51443() / 4f;
		if (displayX && displayY) {
			pose.pushMatrix();
			pose.translate(0, Math.max(height, projHeight + 4) / 2f + 4);
			var pos = pose.transformPosition(new Vector2f());
			if ((projWidth >= width || projHeight >= height) && _3dOnScreen) {
				pos.y = Math.min(pos.y, displayEnd.y() + 6);
			}
			positionDrawers.add(() -> {
				var line1 = waypoint.name();
				pose.pushMatrix();
				pose.translate(pos);
				graphics.method_51448().scale(AxolotlClientWaypoints.waypointTitleScale.get());
				int line1W = minecraft.field_1772.method_1727(line1);
				graphics.method_25294(-line1W / 2 - 2, -2, line1W / 2 + 2, minecraft.field_1772.field_2000 + 2, Colors.GRAY.withAlpha(100).toInt());
				DrawUtil.outlineRect(graphics, -line1W / 2 - 2, -2, line1W + 4, minecraft.field_1772.field_2000 + 4, Colors.GRAY.toInt());
				graphics.method_51433(minecraft.field_1772, line1, -line1W / 2, 0, -1, true);
				if (!waypoint.closerToThan(camPos.method_10216(), camPos.method_10214(), camPos.method_10215(), CUTOFF_DIST)) {
					pose.translate(0, minecraft.field_1772.field_2000 + 4);
					var line2 = AxolotlClientWaypoints.tr("distance", "%.2f".formatted(waypoint.distTo(camPos.method_10216(), camPos.method_10214(), camPos.method_10215())));
					graphics.method_51439(minecraft.field_1772, line2, -minecraft.field_1772.method_27525(line2) / 2, 0, -1, false);
				}
				pose.popMatrix();
			});
			pose.popMatrix();
		}

		if ((projWidth >= width || projHeight >= height) && _3dOnScreen && worldRendererWaypoints.contains(waypoint)) {
			return;
		}

		graphics.method_25294(-width / 2, -height / 2, width / 2, height / 2, waypoint.color().toInt());
		graphics.method_51433(minecraft.field_1772, waypoint.display(), -textWidth / 2, -textHeight / 2, -1, false);
	}

	private @Nullable Result projectToScreen(class_4184 camera, float fov, float xScreenMargin, float yScreenMargin, double x, double y, double z, Vector2f orthoOffset) {
		viewProj.set(x, y, z, 1);
		if (orthoOffset != null) {
			var vec = new Matrix4f();
			vec.rotate(camera.method_23767().invert(new Quaternionf()));
			vec.translate(orthoOffset.x(), orthoOffset.y(), 0);
			vec.rotate(camera.method_23767());
			vec.transform(viewProj);
		}

		view.rotation(camera.method_23767()).translate(camera.method_71156().method_46409().negate());

		Matrix4f projection = minecraft.field_1773.method_22973(fov);
		projection.mul(view);
		viewProj.mul(projection);

		if (orthoOffset == null && AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get()) {
			viewProj.w = Math.max(Math.abs(viewProj.x()), Math.max(Math.abs(viewProj.y()), viewProj.w()));
		}

		if (viewProj.w() <= 0) {
			return null;
		}
		viewProj.div(viewProj.w());

		float projX = viewProj.x();
		float projY = viewProj.y();

		//float x = (graphics.guiWidth()/2f) + ((graphics.guiWidth() - xScreenMargin) * (viewProj.x() / 2f));
		float resultX = 0.5f * (minecraft.method_22683().method_4486() * (projX + 1) - xScreenMargin * projX);
		//float y = graphics.guiHeight() - (graphics.guiHeight()/2f + (graphics.guiHeight()-yScreenMargin) * (viewProj.y() / 2f));
		float resultY = minecraft.method_22683().method_4502() * (0.5f - projY / 2) + (yScreenMargin * projY) / 2f;
		return new Result(resultX, resultY);
	}

	private record Result(float x, float y) {
	}
}
