/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Base64;
import java.util.Comparator;
import java.util.List;

import io.github.axolotlclient.AxolotlClientConfig.api.AxolotlClientConfig;
import io.github.axolotlclient.AxolotlClientConfig.api.manager.ConfigManager;
import io.github.axolotlclient.AxolotlClientConfig.api.options.OptionCategory;
import io.github.axolotlclient.AxolotlClientConfig.impl.managers.VersionedJsonConfigManager;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.ConfigStyles;
import io.github.axolotlclient.waypoints.map.Minimap;
import io.github.axolotlclient.waypoints.map.WorldMapScreen;
import io.github.axolotlclient.waypoints.network.Listener;
import io.github.axolotlclient.waypoints.waypoints.Waypoint;
import io.github.axolotlclient.waypoints.waypoints.WaypointRenderer;
import io.github.axolotlclient.waypoints.waypoints.WaypointStorage;
import io.github.axolotlclient.waypoints.waypoints.gui.CreateWaypointScreen;
import io.github.axolotlclient.waypoints.waypoints.gui.WaypointsScreen;
import lombok.extern.slf4j.Slf4j;
import net.fabricmc.api.ClientModInitializer;
import net.fabricmc.loader.api.FabricLoader;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_3020744;
import net.minecraft.unmapped.C_4576025;
import net.minecraft.unmapped.C_7778778;
import net.minecraft.unmapped.C_8105098;
import net.ornithemc.osl.keybinds.api.KeyBindingEvents;
import net.ornithemc.osl.lifecycle.api.client.MinecraftClientEvents;
import org.lwjgl.input.Keyboard;

@Slf4j
public class AxolotlClientWaypoints implements ClientModInitializer {

	public static final Minimap MINIMAP = new Minimap();
	public static final WaypointStorage WAYPOINT_STORAGE = new WaypointStorage();
	public static final WaypointRenderer WAYPOINT_RENDERER = new WaypointRenderer();
	public static final Listener NETWORK_LISTENER = new Listener();

	public static OptionCategory category = OptionCategory.create(AxolotlClientWaypointsCommon.MODID);
	private final OptionCategory waypoints = OptionCategory.create("waypoints");
	public static BooleanOption renderWaypoints = new BooleanOption("render_waypoints", true);
	public static BooleanOption renderWaypointsInWorld = new BooleanOption("render_waypoints_in_world", true);
	public static BooleanOption renderOutOfViewWaypointsOnScreenEdge = new BooleanOption("render_out_of_view_waypoints", true);

	private final C_7778778 map = new C_7778778(AxolotlClientWaypointsCommon.MODID + ".world_map", Keyboard.KEY_M, AxolotlClientWaypointsCommon.MODID);
	private final C_7778778 manageWaypoints = new C_7778778(AxolotlClientWaypointsCommon.MODID + ".waypoints_menu", Keyboard.KEY_K, AxolotlClientWaypointsCommon.MODID);
	private final C_7778778 newWaypoint = new C_7778778(AxolotlClientWaypointsCommon.MODID + ".create_waypoint", Keyboard.KEY_N, AxolotlClientWaypointsCommon.MODID);

	@Override
	public void onInitializeClient() {
		AxolotlClientWaypointsCommon.init();
		MINIMAP.init();

		category.add(waypoints);
		waypoints.add(renderWaypoints, renderWaypointsInWorld, renderOutOfViewWaypointsOnScreenEdge);

		try {
			Files.createDirectories(FabricLoader.getInstance().getConfigDir().resolve(AxolotlClientWaypointsCommon.MODID));
		} catch (IOException e) {
			log.warn("Failed to create config dir, options may not save correctly!", e);
		}
		ConfigManager configManager;
		AxolotlClientConfig.getInstance().register(configManager = new VersionedJsonConfigManager(AxolotlClientWaypointsCommon.OPTIONS_PATH, category, 1,
			(oldVersion, newVersion, rootCategory, json) -> json));
		configManager.load();
		configManager.save();

		KeyBindingEvents.REGISTER_KEYBINDS.register(reg -> {
			reg.register(map);
			reg.register(manageWaypoints);
			reg.register(newWaypoint);
		});
		MinecraftClientEvents.TICK_END.register(mc -> {
			if (map.m_4823804()) {
				mc.m_6408915(new WorldMapScreen());
			} else if (manageWaypoints.m_4823804()) {
				mc.m_6408915(new WaypointsScreen(mc.f_0723335));
			} else if (newWaypoint.m_4823804()) {
				mc.m_6408915(new CreateWaypointScreen(mc.f_0723335));
			}
		});
		NETWORK_LISTENER.init();
	}

	public static C_3020744 createOptionsScreen(C_3020744 parent) {
		return ConfigStyles.createScreen(parent, category);
	}

	public static C_0561170 rl(String path) {
		return new C_0561170(AxolotlClientWaypointsCommon.MODID, path);
	}

	public static String tr(String key, Object... args) {
		return C_4576025.m_5618193(AxolotlClientWaypointsCommon.MODID + "." + key, args);
	}

	public static String tra(String key) {
		return C_4576025.m_3547652(AxolotlClientWaypointsCommon.MODID + "." + key);
	}

	public static List<Waypoint> getCurrentWaypoints() {
		return getCurrentWaypoints(true);
	}

	public static List<Waypoint> getCurrentWaypoints(boolean dimension) {
		var mc = C_8105098.m_0408063();
		var player = mc.f_7663840;
		return WAYPOINT_STORAGE.getCurrentlyAvailableWaypoints(dimension ? String.valueOf(mc.f_4601986.f_6669533.m_9165548()) : null)
			.sorted(Comparator.comparingDouble(w -> w.squaredDistTo(player.f_6638345, player.f_1187082, player.f_9103758)))
			.toList();
	}

	private static String getB64(String s) {
		return new String(Base64.getUrlEncoder().encode(s.getBytes(StandardCharsets.UTF_8)));
	}

	public static Path getCurrentStorageDir() {
		var mc = C_8105098.m_0408063();
		return getCurrentWorldStorageDir()
			.resolve(getB64(mc.f_4601986.f_6669533.m_9165548()));
	}

	public static Path getCurrentWorldStorageDir() {
		var mc = C_8105098.m_0408063();
		String str;
		if (mc.m_4351464() == null) {
			str = mc.m_1862495().f_7619058;
		} else {
			str = mc.m_4351464().m_0808780();
		}
		return AxolotlClientWaypointsCommon.MOD_STORAGE_DIR.resolve(getB64(str));
	}

	public static boolean playerHasOp() {
		return C_8105098.m_0408063().m_4351464().m_8526012().m_6078293(C_8105098.m_0408063().f_7663840.m_2101945());
	}
}
