/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.waypoints;

import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.unmapped.C_0539808;
import net.minecraft.unmapped.C_0557736;
import net.minecraft.unmapped.C_2691939;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_4461663;
import net.minecraft.unmapped.C_5786166;
import net.minecraft.unmapped.C_7799337;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8373595;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.DrawUtil;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.mixin.GameRendererAccessor;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;
import org.joml.Quaternionf;
import org.joml.Vector2f;
import org.joml.Vector4f;

public class WaypointRenderer {

	private static final double CUTOFF_DIST = 5;
	private final C_8105098 minecraft = C_8105098.m_0408063();
	private final Matrix4f view = new Matrix4f();
	private final Vector4f viewProj = new Vector4f();

	public void render(float f) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		if (minecraft.f_4601986 == null) return;
		var profiler = C_8105098.m_0408063().f_3960058;
		profiler.m_3614560("waypoints");

		var cam = minecraft.m_5661944();

		C_3754158.m_6326777(1, 1, 1);
		C_3754158.m_8373640();

		var camPos = cam.m_4800803(f);
		var cpos = camPos.m_8163229(0, cam.m_6131251(), 0);
		//var camPos = new Vector3d(cam.x, cam.y, cam.z);
		//var prevCamPos = new Vector3d(cam.prevTickX, cam.prevTickY, cam.prevTickZ);
		//camPos.sub(prevCamPos).mul(f).add(prevCamPos);

		C_3754158.m_7643110(false);
		C_3754158.m_7027522();
		float fov = ((GameRendererAccessor) minecraft.f_4267957).invokeGetFov(f, true);
		var win = new C_7799337(minecraft);

		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			profiler.m_8238460(waypoint.name());
			renderWaypoint(waypoint, camPos, cpos, fov, win.m_1713651(), win.m_5421693());
			profiler.m_8388219();
		}

		C_3754158.m_7643110(true);
		C_3754158.m_9078814();
		C_3754158.m_2041265();
	}

	private void renderWaypoint(Waypoint waypoint, C_0557736 camPos, C_0557736 camEyePos, float fov, int guiWidth, int guiHeight) {
		int textWidth = minecraft.f_0426313.m_0040387(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.f_0426313.f_6725889;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		var displayStart = projectToScreen(guiWidth, guiHeight, camPos, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(-(width / 2f * 0.04f), (height / 2f * 0.04f)));
		if (displayStart == null) return;
		var displayEnd = projectToScreen(guiWidth, guiHeight, camPos, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(width / 2f * 0.04f, -(height / 2f * 0.04f)));
		if (displayEnd == null) return;
		float projWidth = Math.abs(displayEnd.x() - displayStart.x());
		float projHeight = Math.abs(displayEnd.y() - displayStart.y());
		if (projWidth < width && projHeight < height) {
			return;
		}

		C_3754158.m_8373640();
		C_3754158.m_5322104(waypoint.x() - camEyePos.f_8797516, waypoint.y() - camEyePos.f_7064947, waypoint.z() - camEyePos.f_1767139);
		var dispatcher = minecraft.m_5440205();
		C_3754158.m_8616673(-dispatcher.f_5107786, 0.0F, 1.0F, 0.0F);
		C_3754158.m_8616673(dispatcher.f_2592027, 1.0F, 0.0F, 0.0F);
		float scale = 0.04F;
		C_3754158.m_4552250(-scale, -scale, scale);
		C_3754158.m_9671730();
		fillRect(-width / 2f, -height / 2f, -0.1f, width / 2f, height / 2f, waypoint.color().toInt());
		drawFontBatch(waypoint.display(), -textWidth / 2f, -textHeight / 2f);
		C_3754158.m_2041265();
	}

	private void fillRect(float x, float y, float z, float x2, float y2, int color) {
		C_5786166 tessellator = C_5786166.m_2065116();
		C_8373595 bufferBuilder = tessellator.m_1454391();
		C_3754158.m_9671730();
		C_3754158.m_2090124();
		C_3754158.m_0833259(770, 771, 1, 0);
		C_3754158.m_3323122((color >> 16 & 0xFF) / 255.0F, (color >> 8 & 0xFF) / 255.0F, (color & 0xFF) / 255.0F, (color >> 24 & 0xFF) / 255.0F);
		bufferBuilder.m_0421390(7, C_4461663.f_6859334);
		bufferBuilder.m_3299851(x, y2, z).m_4365807();
		bufferBuilder.m_3299851(x2, y2, z).m_4365807();
		bufferBuilder.m_3299851(x2, y, z).m_4365807();
		bufferBuilder.m_3299851(x, y, z).m_4365807();
		tessellator.m_8222644();
		C_3754158.m_7547086();
		C_3754158.m_6326777(1, 1, 1);
		C_3754158.m_5313301();
	}

	private void drawFontBatch(String text, float x, float y) {
		minecraft.f_0426313.m_4413321(text, x, y, -1, false);
	}

	public void renderWaypoints(float tick) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		var profiler = C_8105098.m_0408063().f_3960058;
		var cam = minecraft.m_5661944();
		profiler.m_8238460("waypoints");
		var win = new C_7799337(minecraft);

		C_3754158.m_8373640();
		var positionDrawer = new AtomicReference<Runnable>();
		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			C_3754158.m_8373640();
			renderWaypoint(waypoint, tick, cam, positionDrawer, win.m_1713651(), win.m_5421693());
			C_3754158.m_2041265();
		}
		if (positionDrawer.get() != null) {
			positionDrawer.get().run();
		}
		C_3754158.m_2041265();
		profiler.m_8388219();
	}

	private Matrix4f getProjectionMatrix(float fov) {
		Matrix4f matrix4f = new Matrix4f();
		return matrix4f.perspective(fov * ((float) Math.PI / 180F), (float) this.minecraft.f_0545414 / (float) this.minecraft.f_5990000, 0.05F, minecraft.f_9967940.f_7110074 * 4);
	}

	private void renderWaypoint(Waypoint waypoint, float tick, C_0539808 camera, AtomicReference<Runnable> positionDrawn, int guiWidth, int guiHeight) {
		var fov = ((GameRendererAccessor) minecraft.f_4267957).invokeGetFov(tick, false);

		var textWidth = minecraft.f_0426313.m_0040387(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.f_0426313.f_6725889;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		var camPos = camera.m_4800803(tick);

		var displayStart = projectToScreen(guiWidth, guiHeight, camPos, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(-(width / 2f * 0.04f), (height / 2f * 0.04f)));
		var displayEnd = projectToScreen(guiWidth, guiHeight, camPos, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f((width / 2f * 0.04f), -(height / 2f * 0.04f)));
		Result result = projectToScreen(guiWidth, guiHeight, camPos, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), null);
		if (result == null) return;
		float projWidth;
		float projHeight;
		if (displayStart != null && displayEnd != null) {
			projWidth = Math.abs(displayEnd.x() - displayStart.x());
			projHeight = Math.abs(displayEnd.y() - displayStart.y());
		} else {
			projWidth = 0;
			projHeight = 0;
		}

		C_3754158.m_3172490(result.x(), result.y(), 0);
		boolean outOfView = result.x() < -width / 2f || result.x() > guiWidth + width / 2f || result.y() < -height / 2f || result.y() > guiHeight + height / 2f;
		if (!AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get() && outOfView) {
			return;
		}

		boolean _3dOnScreen;
		if (displayEnd != null && displayStart != null) {
			float minX = displayStart.x();
			float minY = displayStart.y();
			float maxX = displayEnd.x();
			float maxY = displayEnd.y();
			_3dOnScreen = minX > 0 && minY > 0 && minX < guiWidth && minY < guiHeight ||
				minX > 0 && maxY > 0 && minX < guiWidth && maxY < guiHeight ||
				maxX > 0 && maxY > 0 && maxX < guiWidth && maxY < guiHeight ||
				maxX > 0 && minY > 0 && maxX < guiWidth && minY < guiHeight;
		} else {
			_3dOnScreen = false;
		}
		if (positionDrawn.get() == null && Math.abs(result.x() - guiWidth / 2f) < (_3dOnScreen ? Math.max(projWidth, width) : width) / 2f && Math.abs(result.y() - guiHeight / 2f) < (_3dOnScreen ? Math.max(height, projHeight) : height) / 2f) {
			positionDrawn.set(() -> {
				var line1 = waypoint.name();
				C_3754158.m_8373640();
				C_3754158.m_3172490(result.x(), result.y(), 0);
				C_3754158.m_3172490(0, Math.max(height, projHeight + 4) / 2f + 4, 0);
				if (_3dOnScreen) {
					float y = result.y()+Math.max(height, projHeight + 4) / 2f + 4;
					var y2 = Math.min(y, displayEnd.y()+6);
					C_3754158.m_3172490(0, y2-y, 0);
				}
				int line1W = minecraft.f_0426313.m_0040387(line1);
				C_2691939.m_7865719(-line1W / 2 - 2, -2, line1W / 2 + 2, minecraft.f_0426313.f_6725889 + 2, Colors.GRAY.withAlpha(100).toInt());
				DrawUtil.outlineRect(-line1W / 2 - 2, -2, line1W + 4, minecraft.f_0426313.f_6725889 + 4, Colors.GRAY.toInt());
				minecraft.f_0426313.m_4413321(line1, -line1W / 2f, 0, -1, true);
				if (!waypoint.closerToThan(camPos.f_8797516, camPos.f_7064947, camPos.f_1767139, CUTOFF_DIST)) {
					C_3754158.m_3172490(0, minecraft.f_0426313.f_6725889 + 4, 0);
					var line2 = AxolotlClientWaypoints.tr("distance", "%.2f".formatted(waypoint.distTo(camPos.f_8797516, camPos.f_7064947, camPos.f_1767139)));
					minecraft.f_0426313.m_4413321(line2, -minecraft.f_0426313.m_0040387(line2) / 2f, 0, -1, false);
				}
				C_3754158.m_2041265();
			});
		}

		if ((projWidth >= width || projHeight >= height) && _3dOnScreen) {
			return;
		}

		C_2691939.m_7865719(-width / 2, -height / 2, width / 2, height / 2, waypoint.color().toInt());
		minecraft.f_0426313.m_4413321(waypoint.display(), -textWidth / 2f, -textHeight / 2f, -1, false);
	}

	private @Nullable Result projectToScreen(int guiWidth, int guiHeight, C_0557736 camPos, float fov, int width, int height, double x, double y, double z, Vector2f orthoOffset) {
		viewProj.set(x, y, z, 1);
		var dispatcher = minecraft.m_5440205();
		if (orthoOffset != null) {
			var vec = new Matrix4f();
			var camRot = new Quaternionf()
				.rotationYXZ(-dispatcher.f_5107786 * (float) (Math.PI / 180.0), dispatcher.f_2592027 * (float) (Math.PI / 180.0), 0.0F)
				.rotateY((float) -(Math.PI));
			vec.rotate(camRot.invert(new Quaternionf()));
			vec.translate(orthoOffset.x(), orthoOffset.y(), 0);
			vec.rotate(camRot);
			vec.transform(viewProj);
		}

		view.rotation(new Quaternionf()
			.rotationYXZ(-dispatcher.f_5107786 * (float) (Math.PI / 180.0), dispatcher.f_2592027 * (float) (Math.PI / 180.0), 0.0F)
			.rotateY((float) -(Math.PI)).invert()).translate((float) -camPos.f_8797516, (float) -camPos.f_7064947, (float) -camPos.f_1767139);

		Matrix4f projection = getProjectionMatrix(fov);
		projection.mul(view);
		viewProj.mul(projection);

		if (orthoOffset == null && AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get()) {
			viewProj.w = Math.max(Math.abs(viewProj.x()), Math.max(Math.abs(viewProj.y()), viewProj.w()));
		}

		if (viewProj.w() <= 0) {
			return null;
		}
		viewProj.div(viewProj.w());

		float projX = viewProj.x();
		float projY = viewProj.y();

		//float x = (graphics.guiWidth()/2f) + ((graphics.guiWidth() - width) * (viewProj.x() / 2f));
		float resultX = 0.5f * (guiWidth * (projX + 1) - width * projX);
		//float y = graphics.guiHeight() - (graphics.guiHeight()/2f + (graphics.guiHeight()-height) * (viewProj.y() / 2f));
		float resultY = guiHeight * (0.5f - projY / 2) + (height * projY) / 2f;
		return new Result(resultX, resultY);
	}

	private record Result(float x, float y) {
	}
}
