/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.map;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ForkJoinPool;
import java.util.function.Supplier;

import com.google.common.collect.Lists;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.Element;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.Screen;
import io.github.axolotlclient.AxolotlClientConfig.impl.ui.Selectable;
import io.github.axolotlclient.AxolotlClientConfig.impl.util.DrawUtil;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.map.util.LevelChunkStorage;
import io.github.axolotlclient.waypoints.map.widgets.AbstractSliderButton;
import io.github.axolotlclient.waypoints.map.widgets.DropdownButton;
import io.github.axolotlclient.waypoints.map.widgets.ImageButton;
import io.github.axolotlclient.waypoints.mixin.LevelAccessor;
import io.github.axolotlclient.waypoints.util.ARGB;
import io.github.axolotlclient.waypoints.waypoints.Waypoint;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import net.minecraft.unmapped.C_0561170;
import net.minecraft.unmapped.C_2441996;
import net.minecraft.unmapped.C_3628668;
import net.minecraft.unmapped.C_3674802;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_3779718;
import net.minecraft.unmapped.C_3831727;
import net.minecraft.unmapped.C_4976084;
import net.minecraft.unmapped.C_5553933;
import net.minecraft.unmapped.C_6849228;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_9335057;
import org.joml.Matrix4fStack;
import org.joml.Vector2i;
import org.joml.Vector3f;

@Slf4j
public class WorldMapScreen extends Screen {
	public static final int MIN_BUILD_HEIGHT = 0;
	private static final int TILE_SIZE = 16;
	private static final C_0561170 OPTIONS_SPRITE = AxolotlClientWaypoints.rl("textures/gui/sprites/options.png");
	private static final C_0561170 OPTIONS_HOVERED_SPRITE = AxolotlClientWaypoints.rl("textures/gui/sprites/options_hovered.png");

	private final Map<Vector2i, LazyTile> tiles = new ConcurrentHashMap<>();
	private final Vector3f dragOffset = new Vector3f();
	private float scale = 1f;
	public static boolean allowCaves = true, allowCavesNether;
	private boolean atSurface;
	private int caveY;
	private String dimension;
	private Waypoint hoveredWaypoint = null;
	private boolean initializedOnce = false;
	private Runnable optionUpdate;
	private final C_3831727 font = C_8105098.m_0408063().f_0426313;
	private final Matrix4fStack matrixStack = new Matrix4fStack(16);

	public WorldMapScreen() {
		super(AxolotlClientWaypoints.tr("worldmap"));
	}

	@Override
	public void m_7817195() {

	}

	@Override
	public void m_7261014(int mouseX, int mouseY, float partialTick) {
		super.m_4184721(0);
		C_3754158.m_8373640();
		matrixStack.clear();
		matrixStack.pushMatrix();
		C_3754158.m_3172490(f_5465691 / 2f, f_3080061 / 2f, 0);
		matrixStack.translate(f_5465691 / 2f, f_3080061 / 2f, 0);
		C_3754158.m_3172490(dragOffset.x(), dragOffset.y(), dragOffset.z());
		matrixStack.translate(dragOffset.x(), dragOffset.y(), dragOffset.z());
		C_3754158.m_8373640();
		matrixStack.pushMatrix();
		C_3754158.m_4552250(scale, scale, 1);
		matrixStack.scale(scale, scale, 1);

		var player = f_7153641.f_7663840;

		for (LazyTile tile : tiles.values()) {
			tile.render(matrixStack, (float) player.f_6638345, (float) player.f_9103758, scale, partialTick, caveY, atSurface, f_5465691, f_3080061);
		}

		matrixStack.popMatrix();
		C_3754158.m_2041265();
		renderMapWaypoints(mouseX, mouseY);

		C_3754158.m_8616673(f_7153641.f_7663840.m_4815215() + 180, 0, 0, 1);
		C_3754158.m_4552250(0.5f * AxolotlClientWaypoints.MINIMAP.arrowScale.get(), 0.5f * AxolotlClientWaypoints.MINIMAP.arrowScale.get(), 1);
		int arrowSize = 15;
		C_3754158.m_3172490(-arrowSize / 2f, -arrowSize / 2f, 5);
		f_7153641.m_1218956().m_5325521(Minimap.arrowLocation);
		m_5935491(0, 0, 0, 0, arrowSize, arrowSize, arrowSize, arrowSize);
		matrixStack.popMatrix();
		C_3754158.m_2041265();
		super.m_7261014(mouseX, mouseY, partialTick);

		if (mouseX > -1 && mouseY > -1) {
			int x = getWorldX(mouseX);
			int z = getWorldZ(mouseY);
			m_2717572(font, AxolotlClientWaypoints.tr("position", String.valueOf(x), String.valueOf(getY(x, z)), String.valueOf(z)), f_5465691 / 2, f_3080061 - 15, Colors.GRAY.toInt());
		}
	}

	private static int blockToSectionCoord(int c) {
		return c >> 4;
	}

	private int getY(int x, int z) {
		int tileX = x / TILE_SIZE;
		int tileY = z / TILE_SIZE;
		var tile = tiles.get(new Vector2i(tileX - 1, tileY - 1));
		C_6849228 c;
		if (tile == null || tile.tile == null) {
			c = ((LevelAccessor) f_7153641.f_4601986).invokeChunkLoadedAt(blockToSectionCoord(x), blockToSectionCoord(z), true) ? f_7153641.f_4601986.m_0750002(blockToSectionCoord(x), blockToSectionCoord(z)) : null;
		} else c = tile.tile.chunk.chunk();
		if (c == null) return MIN_BUILD_HEIGHT;
		int y = c.m_5956776(x & 15, z & 15);
		if (atSurface) {
			return y;
		}
		y = Math.min(y, caveY);
		C_2441996 blockState;
		var mutableBlockPos = new C_3674802.C_0067708(x, 0, z);
		do {
			mutableBlockPos.m_1540202(mutableBlockPos.m_9150363(), --y, mutableBlockPos.m_3900258());
			blockState = c.m_9971171(mutableBlockPos);
		} while (blockState.m_0999604().m_3247759(blockState) == C_9335057.f_8898614 && y > MIN_BUILD_HEIGHT);
		return y;
	}

	private int getWorldX(double guiX) {
		return C_4976084.m_8723286(f_7153641.f_7663840.f_9103758 - ((f_5465691 / 2f + dragOffset.x()) - guiX) / scale);
	}

	private int getWorldZ(double guiZ) {
		return C_4976084.m_8723286(f_7153641.f_7663840.f_9103758 - ((f_3080061 / 2f + dragOffset.y()) - guiZ) / scale);
	}

	private void renderMapWaypoints(int mouseX, int mouseY) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		C_3754158.m_8373640();
		matrixStack.pushMatrix();
		var pos = new Vector3f();
		hoveredWaypoint = null;
		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			C_3754158.m_8373640();
			matrixStack.pushMatrix();
			float posX = (float) (waypoint.x() - f_7153641.f_7663840.f_6638345) + 1;
			float posY = (float) (waypoint.z() - f_7153641.f_7663840.f_9103758) + 1;

			matrixStack.translate(posX * scale, posY * scale, 0);
			C_3754158.m_3172490(posX * scale, posY * scale, 0);
			pos.zero();
			matrixStack.transformPosition(pos);

			int textWidth = font.m_0040387(waypoint.display());
			int width = textWidth + Waypoint.displayXOffset() * 2;
			int textHeight = font.f_6725889;
			int height = textHeight + Waypoint.displayYOffset() * 2;
			pos.sub(width / 2f, height / 2f, 0);
			m_7865719(-(width / 2), -(height / 2), (width / 2), (height / 2), waypoint.color().toInt());
			font.m_4413321(waypoint.display(), -(textWidth / 2f), -textHeight / 2f, -1, false);
			if (hoveredWaypoint == null) {
				if (mouseX >= pos.x() && mouseY >= pos.y() && mouseX < pos.x() + width && mouseY < pos.y() + height) {
					hoveredWaypoint = waypoint;
					C_3754158.m_3172490(0, 0, 2);
					DrawUtil.outlineRect(-width / 2, -height / 2, width, height, Colors.WHITE.toInt());
				}
			}
			matrixStack.popMatrix();
			C_3754158.m_2041265();
		}
		matrixStack.popMatrix();
		C_3754158.m_2041265();
	}

	private void collectPlayerYData() {
		var level = f_7153641.f_4601986;
		if (allowCaves || (allowCavesNether && level.f_6669533.m_8976359())) {
			int playerX = (int) (f_7153641.f_7663840.f_6638345 + 0.5);
			int playerZ = (int) (f_7153641.f_7663840.f_9103758 + 0.5);
			var centerChunk = level.m_0750002(blockToSectionCoord(playerX), blockToSectionCoord(playerZ));
			var surface = centerChunk.m_5956776(playerX & 15, playerZ & 15);
			C_3674802.C_0067708 mutableBlockPos = new C_3674802.C_0067708(playerX, surface, playerZ);
			int solidBlocksAbovePlayer = 0;
			atSurface = false;
			if (level.f_6669533.m_8976359()) {
				atSurface = (int) (f_7153641.f_7663840.f_1187082 + 0.5) >= level.m_8568350();
			} else if (surface + 1 <= (int) (f_7153641.f_7663840.f_1187082 + 0.5)) {
				atSurface = true;
			} else {
				while (solidBlocksAbovePlayer <= 3 && surface > (int) (f_7153641.f_7663840.f_1187082 + 0.5) && surface > MIN_BUILD_HEIGHT) {
					C_2441996 state = centerChunk.m_9971171(mutableBlockPos);
					mutableBlockPos.m_1540202(playerX, surface--, playerZ);
					if (!(state.m_0999604().m_4952508() || !state.m_0999604().m_4839982() || !state.m_0999604().m_8740684())) {
						solidBlocksAbovePlayer++;
					}
				}
				if (solidBlocksAbovePlayer <= 2) {
					atSurface = true;
				}
			}
		} else {
			atSurface = true;
		}

		caveY = (int) (f_7153641.f_7663840.f_1187082 + 0.5);
		dimension = level.f_6669533.m_9165548();
	}

	private void createTiles() {
		int playerX = (int) (f_7153641.f_7663840.f_6638345 + 0.5);
		int playerZ = (int) (f_7153641.f_7663840.f_9103758 + 0.5);

		var playerTile = createTile(playerX, playerZ, atSurface, caveY);
		if (playerTile != null) {
			Map<Vector2i, LazyTile> loadedTiles = new HashMap<>(tiles);
			tiles.clear();
			tiles.put(new Vector2i(playerTile.tilePosX(), playerTile.tilePosY()), playerTile);
			f_7153641.m_2167033(() -> {
				triggerNeighbourLoad(playerTile, atSurface, caveY, playerTile);
				loadedTiles.forEach((v, t) -> {
					if (!tiles.containsKey(v)) {
						tiles.put(v, t);
					}
				});
			});
		}
	}

	private void triggerNeighbourLoad(LazyTile tile, boolean atSurface, int caveY, LazyTile origin) {
		List<Runnable> queue = new ArrayList<>();
		loadNeighbour(tile, -1, -1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, -1, 0, atSurface, caveY, queue, origin);
		loadNeighbour(tile, -1, 1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 0, 1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 1, 1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 1, 0, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 1, -1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 0, -1, atSurface, caveY, queue, origin);
		if (!queue.isEmpty()) {
			for (Runnable runnable : Lists.reverse(queue)) {
				runnable.run();
			}
		}
	}

	private void loadNeighbour(LazyTile origin, int tileOffsetX, int tileOffsetY, boolean atSurface, int caveY, List<Runnable> loadQueue, LazyTile mapOrigin) {
		if (tileOffsetY != 0 || tileOffsetX != 0) {
			var tileXLeft = origin.tilePosX() + tileOffsetX;
			var tileYLeft = origin.tilePosY() + tileOffsetY;
			var vec = new Vector2i(tileXLeft, tileYLeft);
			if (!tiles.containsKey(vec)) {
				var anchorXLeft = tileXLeft * TILE_SIZE;
				var anchorYLeft = tileYLeft * TILE_SIZE;
				var tile = createTile(anchorXLeft, anchorYLeft, atSurface, caveY);
				if (tile != null) {
					tiles.put(vec, tile);
					loadQueue.add(() -> triggerNeighbourLoad(tile, atSurface, caveY, mapOrigin));
				}
			}
		}
	}

	private static C_0561170 getTileRl(int tileX, int tileY) {
		return AxolotlClientWaypoints.rl("world_map/tile_" + tileX + "_" + tileY);
	}

	private LazyTile createTile(int anchorX, int anchorZ, boolean atSurface, int caveY) {
		return createTile(f_7153641, anchorX, anchorZ, atSurface, caveY);
	}

	private static LazyTile createTile(C_8105098 minecraft, int anchorX, int anchorZ, boolean atSurface, int caveY) {
		anchorZ = anchorZ - anchorZ % TILE_SIZE;
		anchorX = anchorX - anchorX % TILE_SIZE;
		var level = minecraft.f_4601986;
		int tileX = anchorX / TILE_SIZE;
		int tileY = anchorZ / TILE_SIZE;
		C_6849228 tileChunk = ((LevelAccessor) level).invokeChunkLoadedAt(tileX, tileY, false) ? level.m_0750002(tileX, tileY) : null;
		if (tileChunk != null) {
			return new LazyTile(tileX, tileY, () -> {
				var t = Tile.create(tileX, tileY, tileChunk);
				t.update(caveY, atSurface, level);
				return t;
			});
		}
		return null;
	}

	public static void saveLoadedChunkTile(int chunkX, int chunkZ) {
		C_8105098 minecraft = C_8105098.m_0408063();
		var tile = createTile(minecraft, chunkX << 4, chunkZ << 4, true, 0);
		if (tile != null) {
			tile.load(0, true)
				.thenRun(() -> {
					var dir = getCurrentLevelMapSaveDir();
					try {
						Files.createDirectories(dir);

						saveTile(tile, dir);
					} catch (IOException e) {
						log.error("Failed to create world map save dir!", e);
					}
				});
		}
	}

	private double dragMouseX;
	private double dragMouseY;

	@Override
	public boolean mouseClicked(double mouseX, double mouseY, int button) {
		if (!super.mouseClicked(mouseX, mouseY, button)) {
			if (button == 1) {
				if (hoveredWaypoint != null) {
					f_7153641.m_6408915(new ContextMenuScreen(this, mouseX, mouseY, new ContextMenuScreen.Type.Waypoint(hoveredWaypoint)));
				} else {
					int worldX = getWorldX(mouseX);
					int worldZ = getWorldZ(mouseY);
					f_7153641.m_6408915(new ContextMenuScreen(this, mouseX, mouseY, new ContextMenuScreen.Type.Map(dimension, worldX, getY(worldX, worldZ), worldZ)));
				}
				return true;
			} else if (button == 0) {
				dragMouseX = (mouseX - dragOffset.x);
				dragMouseY = (mouseY - dragOffset.y);
			}
			return false;
		}
		return true;
	}

	@Override
	public boolean mouseDragged(double mouseX, double mouseY, int button, double dragX, double dragY) {
		if (!super.mouseDragged(mouseX, mouseY, button, dragX, dragY)) {
			if (button == 0) {
				dragOffset.set((float) (mouseX - dragMouseX), (float) (mouseY - dragMouseY), 0);
				return true;
			}
			return false;
		}
		return true;
	}

	@Override
	public boolean mouseScrolled(double mouseX, double mouseY, double scrollX, double scrollY) {
		if (!super.mouseScrolled(mouseX, mouseY, scrollX, scrollY)) {
			if (scrollY > 0) {
				scale *= 2;
				var offsetX = f_5465691 / 2f + dragOffset.x();
				var offsetY = f_3080061 / 2f + dragOffset.y();
				var mirroredOnOffsetX = offsetX - (mouseX - offsetX);
				var mirroredOnOffsetY = offsetY - (mouseY - offsetY);
				dragOffset.set(mirroredOnOffsetX - f_5465691 / 2f, mirroredOnOffsetY - f_3080061 / 2f, 0);
			} else {
				scale /= 2;
				var offsetX = f_5465691 / 2f + dragOffset.x();
				var offsetY = f_3080061 / 2f + dragOffset.y();
				var mirroredOnOffsetX = offsetX + (mouseX - offsetX) / 2f;
				var mirroredOnOffsetY = offsetY + (mouseY - offsetY) / 2f;
				dragOffset.set(mirroredOnOffsetX - f_5465691 / 2f, mirroredOnOffsetY - f_3080061 / 2f, 0);
			}
		}
		return true;
	}

	@Override
	public void m_3593494() {
		class Overlay implements Element, Selectable {

			@Override
			public boolean isFocused() {
				return true;
			}

			@Override
			public void setFocused(boolean focused) {

			}

			@Override
			public SelectionType getType() {
				return SelectionType.NONE;
			}

			@Override
			public boolean isMouseOver(double mouseX, double mouseY) {
				return true;
			}

			boolean called = false;

			@Override
			public boolean mouseScrolled(double mouseX, double mouseY, double amountX, double amountY) {
				if (called) return false;
				called = true;
				var bl = WorldMapScreen.this.mouseScrolled(mouseX, mouseY, amountX, amountY);
				called = false;
				return bl;
			}

			@Override
			public boolean mouseDragged(double mouseX, double mouseY, int button, double deltaX, double deltaY) {
				if (called) return false;
				called = true;
				var bl = WorldMapScreen.super.mouseDragged(mouseX, mouseY, button, deltaX, deltaY);
				called = false;
				return bl;
			}
		}
		addSelectableChild(new Overlay());
		addDrawableChild(new ImageButton(4, f_3080061 - 20, 16, 16, new ImageButton.WidgetSprites(OPTIONS_SPRITE, OPTIONS_SPRITE, OPTIONS_HOVERED_SPRITE),
			btn -> f_7153641.m_6408915(AxolotlClientWaypoints.createOptionsScreen(this)), AxolotlClientWaypoints.tr("options")));
		var slider = addDrawableChild(new AbstractSliderButton(f_5465691 - 150, 20, 150, 20, AxolotlClientWaypoints.tr("player_y"), 0) {
			final int min = MIN_BUILD_HEIGHT - 1;
			final int max = f_7153641.f_4601986.m_8568350() + 1;

			@Override
			protected void updateMessage() {
				if (value == 0) {
					setMessage(AxolotlClientWaypoints.tr("player_y"));
				} else {
					setMessage(String.valueOf(caveY - 1));
				}
			}

			@Override
			protected void applyValue() {
				caveY = (int) (min + (max - min) * value);
				atSurface = false;
				if (value == 0) {
					collectPlayerYData();
				}
				CompletableFuture.runAsync(() -> tiles.values().forEach(t -> t.update(caveY, atSurface, f_7153641.f_4601986)));
			}
		});
		addDrawableChild(new DropdownButton(f_5465691 - 20, 0, 20, 20,
			AxolotlClientWaypoints.tr("open_dropdown"), (btn, val) -> slider.visible = val));
		slider.visible = false;
		optionUpdate = () -> {
			var allowsCaves = allowCaves || (allowCavesNether && f_7153641.f_4601986.f_6669533.m_8976359());
			boolean updated = slider.active != allowsCaves;
			if (updated) {
				slider.active = allowsCaves;
				if (!allowsCaves) {
					atSurface = true;
					CompletableFuture.runAsync(() -> tiles.values().forEach(t -> t.update(caveY, atSurface, f_7153641.f_4601986)));
				} else {
					slider.applyValue();
				}
			}
		};
		AxolotlClientWaypoints.NETWORK_LISTENER.postReceive.add(optionUpdate);
		optionUpdate.run();
		if (tiles.isEmpty()) {
			f_7153641.m_2167033(() -> {
				if (!initializedOnce) {
					collectPlayerYData();
				}
				loadSavedTiles();
				createTiles();
			});
		}
		initializedOnce = true;
	}

	@Override
	public void m_2450377() {

	}

	private void loadSavedTiles() {
		var dir = getCurrentLevelMapSaveDir();
		if (Files.exists(dir)) {
			try (var s = Files.list(dir)) {
				s.forEach(file -> {
					var name = file.getFileName().toString();
					if (!name.endsWith(Tile.FILE_EXTENSION)) {
						return;
					}
					name = name.substring(0, name.indexOf("."));
					var coords = name.split("_");
					int x = Integer.parseInt(coords[0]);
					int y = Integer.parseInt(coords[1]);
					var key = new Vector2i(x, y);
					if (!tiles.containsKey(key)) {
						try {
							var tile = new LazyTile(x, y, () -> {
								try {
									return Tile.read(file, minecraft.world);
								} catch (IOException e) {
									return null;
								}
							});
							tiles.put(key, tile);
						} catch (Exception e) {
							log.warn("Failed to load tile at {}, {}", x, y, e);
						}
					}
				});
			} catch (IOException e) {
				log.info("Failed to load saved world map tiles!", e);
			}
		}
	}

	private void saveTiles() {
		var dir = getCurrentLevelMapSaveDir();
		try {
			Files.createDirectories(dir);

			tiles.values().forEach((tile) -> saveTile(tile, dir));
		} catch (IOException e) {
			log.error("Failed to create world map save dir!", e);
		}
	}

	private static void saveTile(LazyTile tile, Path dir) {
		try {
			tile.save(dir);
		} catch (IOException e) {
			log.warn("Failed to save tile at {}, {}", tile.tilePosX(), tile.tilePosY(), e);
		}
	}

	private static Path getCurrentLevelMapSaveDir() {
		return AxolotlClientWaypoints.getCurrentStorageDir().resolve("worldmap");
	}

	@Override
	public void m_8984281() {
		if (optionUpdate != null) {
			AxolotlClientWaypoints.NETWORK_LISTENER.postReceive.remove(optionUpdate);
		}
		if (f_7153641.f_0723335 == null) {
			saveTiles();
			tiles.values().forEach(LazyTile::release);
			tiles.clear();
		}
	}

	@RequiredArgsConstructor
	private static class LazyTile {
		private Tile tile;
		@Getter
		@Accessors(fluent = true)
		private final int tilePosX, tilePosY;
		private final Supplier<Tile> supplier;
		private final Vector3f pos = new Vector3f();
		private boolean loaded;

		public void render(Matrix4fStack matrixStack, float playerX, float playerZ, float scale, float delta, int caveY, boolean atSurface, int guiWidth, int guiHeight) {
			float x = tilePosX() * TILE_SIZE - playerX;
			float y = tilePosY() * TILE_SIZE - playerZ;
			C_3754158.m_8373640();
			matrixStack.pushMatrix();
			C_3754158.m_3172490(x, y, 0);
			matrixStack.translate(x, y, 0);
			pos.zero();
			matrixStack.transformPosition(pos);
			if (pos.x + TILE_SIZE * scale >= 0 && pos.x < guiWidth && pos.y + TILE_SIZE * scale >= 0 && pos.y < guiHeight) {
				if (tile == null) {
					if (!loaded) {
						load(caveY, atSurface);
					}
				} else {
					// FIXME - floating point precision errors?
					C_8105098.m_0408063().m_1218956().m_5325521(tile.rl());
					m_5935491(0, 0, 0, 0, TILE_SIZE, TILE_SIZE, TILE_SIZE, TILE_SIZE);
				}
			}
			matrixStack.popMatrix();
			C_3754158.m_2041265();
		}

		public CompletableFuture<?> load(int caveY, boolean atSurface) {
			if (!loaded) {
				loaded = true;
				return CompletableFuture.supplyAsync(supplier, C_8105098.m_0408063()::m_2167033).thenApplyAsync(t -> {
					t.update(caveY, atSurface, C_8105098.m_0408063().f_4601986);
					return t;
				}, ForkJoinPool.commonPool()).thenApply(t -> tile = t);
			}
			return CompletableFuture.completedFuture(null);
		}

		public void release() {
			if (tile != null) {
				tile.release();
			}
		}

		public void save(Path dir) throws IOException {
			if (tile != null) {
				tile.save(dir);
			}
		}

		public void update(int caveY, boolean atSurface, C_5553933 level) {
			if (tile != null) {
				tile.update(caveY, atSurface, level);
			}
		}
	}

	private record Tile(int tilePosX, int tilePosY, C_0561170 rl, C_3779718 tex,
						LevelChunkStorage.Entry chunk) {
		public static final String FILE_EXTENSION = ".bin";

		public void release() {
			C_8105098.m_0408063().m_1218956().m_3775266(rl);
		}

		public void save(Path dir) throws IOException {
			var out = dir.resolve("%d_%d%s".formatted(tilePosX, tilePosY, FILE_EXTENSION));
			chunk.write(out);
		}

		public static Tile create(int x, int y, C_6849228 chunk) {
			return create(x, y, new LevelChunkStorage.Entry(chunk));
		}

		public static Tile create(int x, int y, LevelChunkStorage.Entry chunk) {
			var rl = getTileRl(x, y);
			var tex = new C_3779718(TILE_SIZE, TILE_SIZE);
			Arrays.fill(tex.m_1823577(), Colors.BLACK.toInt());
			C_8105098.m_0408063().m_1218956().m_0381205(rl, tex);
			return new Tile(x, y, rl, tex, chunk);
		}

		public static Tile read(Path p, C_5553933 level) throws IOException {
			var name = p.getFileName().toString();
			name = name.substring(0, name.indexOf("."));
			var coords = name.split("_");
			int x = Integer.parseInt(coords[0]);
			int y = Integer.parseInt(coords[1]);
			return create(x, y, LevelChunkStorage.Entry.read(p, level));
		}

		public void update(int caveY, boolean atSurface, C_5553933 level) {
			if (chunk.chunk().m_5909771()) {
				Arrays.fill(tex.m_1823577(), Colors.BLACK.toInt());
				C_8105098.m_0408063().m_2167033(tex::m_4420833);
				return;
			}
			int levelMinY = MIN_BUILD_HEIGHT;
			int centerX = (tilePosX * TILE_SIZE) + TILE_SIZE / 2;
			int centerZ = (tilePosY * TILE_SIZE) + TILE_SIZE / 2;

			var pixels = tex.m_1823577();
			int size = TILE_SIZE;
			int texHalfWidth = size / 2;

			C_3674802.C_0067708 mutableBlockPos = new C_3674802.C_0067708();
			C_3674802.C_0067708 mutableBlockPos2 = new C_3674802.C_0067708();
			boolean updated = false;
			for (int x = 0; x < size; x++) {
				double d = 0;
				for (int z = -1; z < size; z++) {
					int chunkX = (centerX + x - texHalfWidth);
					int chunkZ = (centerZ + z - texHalfWidth);

					int fluidDepth = 0;
					double e = 0.0;
					mutableBlockPos.m_1540202(chunkX, 0, chunkZ);
					C_6849228 levelChunk;
					int y;
					if (z < 0) {
						mutableBlockPos.m_1540202(chunkX, 0, chunkZ + 1);
					}
					levelChunk = chunk.chunk();
					y = levelChunk.m_5956776(mutableBlockPos.m_9150363() & 15, mutableBlockPos.m_3900258() & 15) + 1;
					if (!atSurface) {
						y = Math.min(y, caveY);
					}
					C_2441996 blockState;
					if (y <= levelMinY) {
						blockState = C_3628668.f_3097723.m_9077732();
					} else {
						do {
							mutableBlockPos.m_1540202(mutableBlockPos.m_9150363(), --y, mutableBlockPos.m_3900258());
							blockState = levelChunk.m_9971171(mutableBlockPos);
						} while (blockState.m_0999604().m_3247759(blockState) == C_9335057.f_8898614 && y > levelMinY);

						if (y > levelMinY && blockState.m_0999604().m_8228353().m_8583295()) {
							int highestFullBlockY = y - 1;
							mutableBlockPos2.m_1540202(mutableBlockPos.m_9150363(), mutableBlockPos.m_4798774(), mutableBlockPos.m_3900258());

							C_2441996 blockState2;
							do {
								mutableBlockPos2.m_1540202(mutableBlockPos2.m_9150363(), highestFullBlockY--, mutableBlockPos2.m_3900258());
								blockState2 = levelChunk.m_9971171(mutableBlockPos2);
								fluidDepth++;
							} while (highestFullBlockY > levelMinY && blockState2.m_0999604().m_8228353().m_8583295());

						}
					}

					e += y;
					var mapColor = blockState.m_0999604().m_3247759(blockState);

					int brightness;
					if (mapColor == C_9335057.f_2086259) {
						double f = fluidDepth * 0.1 + (x + z & 1) * 0.2;
						if (f < 0.5) {
							brightness = 2;
						} else if (f > 0.9) {
							brightness = 0;
						} else {
							brightness = 1;
						}
					} else {
						double f = (e - d) * 4.0 / (1 + 4) + ((x + z & 1) - 0.5) * 0.4;
						if (f > 0.6) {
							brightness = 2;
						} else if (f < -0.6) {
							brightness = 0;
						} else {
							brightness = 1;
						}
					}

					d = e;
					/*if (Minimap.useTextureSampling.get()) {
						if (z >= 0 && !blockState.isAir()) {
							final int fz = z, fx = x;
							TextureSampler.getSample(blockState, level, mutableBlockPos, brightness).thenAccept(color -> {
								color = ARGB.opaque(color);
								if (Integer.rotateRight(pixels.getPixelRGBA(fx, fz), 4) != color) {
									pixels.setPixelRGBA(fx, fz, Integer.rotateLeft(color, 4));
									tex.upload();
								}
							});
						}
					} else*/
					{
						int color = mapColor.m_4724233(brightness);
						if (z >= 0 && pixels[x + z * TILE_SIZE] != color) {
							pixels[x + z * TILE_SIZE] = ARGB.opaque(color);
							updated = true;
						}
					}
				}
			}

			if (updated) {
				C_8105098.m_0408063().m_2167033(tex::m_4420833);
			}
		}
	}
}
