/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.waypoints;

import java.util.concurrent.atomic.AtomicBoolean;
import net.minecraft.unmapped.C_0539808;
import net.minecraft.unmapped.C_2691939;
import net.minecraft.unmapped.C_3754158;
import net.minecraft.unmapped.C_4461663;
import net.minecraft.unmapped.C_5786166;
import net.minecraft.unmapped.C_7799337;
import net.minecraft.unmapped.C_8105098;
import net.minecraft.unmapped.C_8373595;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.mixin.GameRendererAccessor;
import org.joml.Matrix4f;
import org.joml.Quaternionf;
import org.joml.Vector3d;
import org.joml.Vector4f;

public class WaypointRenderer {

	private static final double CUTOFF_DIST = 12;
	private final C_8105098 minecraft = C_8105098.m_0408063();
	private final Matrix4f view = new Matrix4f();
	private final Vector4f viewProj = new Vector4f();

	public void render(float f) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		if (minecraft.f_4601986 == null) return;
		var profiler = C_8105098.m_0408063().f_3960058;
		profiler.m_3614560("waypoints");

		var cam = minecraft.m_5661944();

		C_3754158.m_6326777(1, 1, 1);
		C_3754158.m_8373640();

		var camPos = new Vector3d(cam.f_6638345, cam.f_1187082, cam.f_9103758);
		var prevCamPos = new Vector3d(cam.f_1098674, cam.f_9198678, cam.f_0414261);
		camPos.sub(prevCamPos).mul(f).add(prevCamPos);

		C_3754158.m_7643110(false);
		C_3754158.m_7027522();
		var guiScale = new C_7799337(minecraft).m_8052133();

		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			if (waypoint.closerToThan(camPos.x, camPos.y + cam.m_6131251(), camPos.z, CUTOFF_DIST / guiScale)) {
				profiler.m_8238460(waypoint.name());
				renderWaypoint(waypoint, camPos);
				profiler.m_8388219();
			}
		}

		C_3754158.m_7643110(true);
		C_3754158.m_9078814();
		C_3754158.m_2041265();
	}

	private void renderWaypoint(Waypoint waypoint, Vector3d camPos) {
		C_3754158.m_8373640();
		C_3754158.m_5322104(waypoint.x() - camPos.x, waypoint.y() - (camPos.y), waypoint.z() - camPos.z);
		var dispatcher = minecraft.m_5440205();
		C_3754158.m_8616673(-dispatcher.f_5107786, 0.0F, 1.0F, 0.0F);
		C_3754158.m_8616673(dispatcher.f_2592027, 1.0F, 0.0F, 0.0F);
		float scale = 0.04F;
		C_3754158.m_4552250(-scale, -scale, scale);
		C_3754158.m_9671730();
		int textWidth = minecraft.f_0426313.m_0040387(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.f_0426313.f_6725889;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		fillRect(-width / 2f, -height / 2f, -0.1f, width / 2f, height / 2f, waypoint.color().toInt());
		drawFontBatch(waypoint.display(), -textWidth / 2f, -textHeight / 2f);
		C_3754158.m_2041265();
	}

	private void fillRect(float x, float y, float z, float x2, float y2, int color) {
		C_5786166 tessellator = C_5786166.m_2065116();
		C_8373595 bufferBuilder = tessellator.m_1454391();
		C_3754158.m_9671730();
		C_3754158.m_2090124();
		C_3754158.m_0833259(770, 771, 1, 0);
		C_3754158.m_3323122((color >> 16 & 0xFF) / 255.0F, (color >> 8 & 0xFF) / 255.0F, (color & 0xFF) / 255.0F, (color >> 24 & 0xFF) / 255.0F);
		bufferBuilder.m_0421390(7, C_4461663.f_6859334);
		bufferBuilder.m_3299851(x, y2, z).m_4365807();
		bufferBuilder.m_3299851(x2, y2, z).m_4365807();
		bufferBuilder.m_3299851(x2, y, z).m_4365807();
		bufferBuilder.m_3299851(x, y, z).m_4365807();
		tessellator.m_8222644();
		C_3754158.m_7547086();
		C_3754158.m_6326777(1, 1, 1);
		C_3754158.m_5313301();
	}

	private void drawFontBatch(String text, float x, float y) {
		minecraft.f_0426313.m_4413321(text, x, y, -1, false);
	}

	public void renderWaypoints(float tick) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		var profiler = C_8105098.m_0408063().f_3960058;
		var cam = minecraft.m_5661944();
		profiler.m_8238460("waypoints");
		var win = new C_7799337(minecraft);

		C_3754158.m_8373640();
		var positionDrawn = new AtomicBoolean();
		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			C_3754158.m_8373640();
			renderWaypoint(waypoint, tick, cam, positionDrawn, win.m_1713651(), win.m_5421693(), win.m_8052133());
			C_3754158.m_2041265();
		}
		C_3754158.m_2041265();
		profiler.m_8388219();
	}

	private Matrix4f getProjectionMatrix(float fov) {
		Matrix4f matrix4f = new Matrix4f();
		return matrix4f.perspective(fov * ((float) Math.PI / 180F), (float) this.minecraft.f_0545414 / (float) this.minecraft.f_5990000, 0.05F, minecraft.f_9967940.f_7110074 * 4);
	}

	private void renderWaypoint(Waypoint waypoint, float tick, C_0539808 camera, AtomicBoolean positionDrawn, int guiWidth, int guiHeight, int guiScale) {
		var fov = ((GameRendererAccessor) minecraft.f_4267957).invokeGetFov(tick, true);

		var textWidth = minecraft.f_0426313.m_0040387(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.f_0426313.f_6725889;
		int height = textHeight + Waypoint.displayYOffset() * 2;

		viewProj.set(waypoint.x(), waypoint.y(), waypoint.z(), 1);
		var dispatcher = minecraft.m_5440205();
		var camPos = camera.m_4800803(tick);
		view.rotation(new Quaternionf()
			.rotationYXZ(-dispatcher.f_5107786 * (float) (Math.PI / 180.0), dispatcher.f_2592027 * (float) (Math.PI / 180.0), 0.0F)
			.rotateY((float) -(Math.PI)).invert()).translate((float) -camPos.f_8797516, (float) -camPos.f_7064947, (float) -camPos.f_1767139);

		Matrix4f projection = getProjectionMatrix(fov);
		projection.mul(view);
		viewProj.mul(projection);

		if (AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get()) {
			viewProj.w = Math.max(Math.abs(viewProj.x()), Math.max(Math.abs(viewProj.y()), viewProj.w()));
		}

		if (viewProj.w() <= 0) {
			return;
		}
		viewProj.div(viewProj.w());

		float projX = viewProj.x();
		float projY = viewProj.y();

		//float x = (graphics.guiWidth()/2f) + ((graphics.guiWidth() - width) * (viewProj.x() / 2f));
		float x = 0.5f * (guiWidth * (projX + 1) - width * projX);
		//float y = graphics.guiHeight() - (graphics.guiHeight()/2f + (graphics.guiHeight()-height) * (viewProj.y() / 2f));
		float y = guiHeight * (0.5f - projY / 2) + (height * projY) / 2f;

		C_3754158.m_3172490(x, y, 0);

		if (!AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get() && (x < -width / 2f || x > guiWidth + width / 2f || y < -height / 2f || y > guiHeight + height / 2f)) {
			return;
		}
		if (waypoint.closerToThan(camPos.f_8797516, camPos.f_7064947, camPos.f_1767139, CUTOFF_DIST / guiScale)) {
			return;
		}

		if (!positionDrawn.get() && Math.abs(x - guiWidth / 2f) < 8 && Math.abs(y - guiHeight / 2f) < 8) {
			positionDrawn.set(true);
			C_3754158.m_8373640();
			C_3754158.m_3172490(0, height / 2f + 2, 0);
			var line1 = waypoint.name();
			minecraft.f_0426313.m_4413321(line1, -minecraft.f_0426313.m_0040387(line1) / 2f, 0, -1, false);
			C_3754158.m_3172490(0, minecraft.f_0426313.f_6725889 + 2, 0);
			var line2 = AxolotlClientWaypoints.tr("distance", "%.2f".formatted(waypoint.distTo(camPos.f_8797516, camPos.f_7064947, camPos.f_1767139)));
			minecraft.f_0426313.m_4413321(line2, -minecraft.f_0426313.m_0040387(line2) / 2f, 0, -1, false);
			C_3754158.m_2041265();
		}

		C_2691939.m_7865719(-width / 2, -height / 2, width / 2, height / 2, waypoint.color().toInt());
		minecraft.f_0426313.m_4413321(waypoint.display(), -textWidth / 2f, -textHeight / 2f, -1, false);
	}
}
