/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.map;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import com.google.common.collect.Lists;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.map.util.LevelChunkStorage;
import io.github.axolotlclient.waypoints.map.widgets.DropdownButton;
import io.github.axolotlclient.waypoints.map.widgets.ImageButton;
import io.github.axolotlclient.waypoints.map.widgets.WidgetSprites;
import io.github.axolotlclient.waypoints.util.ARGB;
import io.github.axolotlclient.waypoints.waypoints.Waypoint;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_1163;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2561;
import net.minecraft.class_2680;
import net.minecraft.class_2791;
import net.minecraft.class_2806;
import net.minecraft.class_2818;
import net.minecraft.class_2902;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_357;
import net.minecraft.class_3610;
import net.minecraft.class_3620;
import net.minecraft.class_4076;
import net.minecraft.class_437;
import org.joml.Vector2i;
import org.joml.Vector3f;

@SuppressWarnings({"DataFlowIssue"})
@Slf4j
public class WorldMapScreen extends class_437 {
	private static final int TILE_SIZE = 16;
	private static final class_2960 OPTIONS_SPRITE = AxolotlClientWaypoints.rl("textures/gui/sprites/options.png");
	private static final class_2960 OPTIONS_HOVERED_SPRITE = AxolotlClientWaypoints.rl("textures/gui/sprites/options_hovered.png");

	private final Map<Vector2i, LazyTile> tiles = new ConcurrentHashMap<>();
	private final Vector3f dragOffset = new Vector3f();
	private float scale = 1f;
	public static boolean allowCaves = true, allowCavesNether;
	private boolean atSurface;
	private int caveY;
	private class_2960 dimension;
	private Waypoint hoveredWaypoint = null;
	private boolean initializedOnce = false;
	private Runnable optionUpdate;

	public WorldMapScreen() {
		super(AxolotlClientWaypoints.tr("worldmap"));
	}

	@Override
	public void method_25394(class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
		method_25420(guiGraphics);
		guiGraphics.method_51448().method_22903();
		guiGraphics.method_51448().method_46416(field_22789 / 2f, field_22790 / 2f, 0);
		guiGraphics.method_51448().method_23760().method_23761().translate(dragOffset);
		guiGraphics.method_51448().method_22903();
		guiGraphics.method_51448().method_22905(scale, scale, 1);

		var playerPos = field_22787.field_1724.method_19538();

		for (LazyTile tile : tiles.values()) {
			tile.render(guiGraphics, (float) playerPos.method_10216(), (float) playerPos.method_10215(), scale, partialTick, caveY, atSurface);
		}
		int x = getWorldX(mouseX);
		int z = getWorldZ(mouseY);
		{
			guiGraphics.method_51448().method_22903();
			guiGraphics.method_51448().method_46416((float) -playerPos.method_10216(), (float) -playerPos.method_10215(), 0);
			int tileX = x / TILE_SIZE;
			int tileY = z / TILE_SIZE;
			if (x < 0 && x % TILE_SIZE != 0) {
				tileX -= 1;
			}
			if (z < 0 && z % TILE_SIZE != 0) {
				tileY -= 1;
			}
			guiGraphics.method_51448().method_46416(tileX * TILE_SIZE, tileY * TILE_SIZE, 0);
			guiGraphics.method_25294(0, 0, TILE_SIZE, TILE_SIZE, 0x33FFFFFF);
			guiGraphics.method_49601(0, 0, TILE_SIZE, TILE_SIZE, 0x33FFFFFF);
			guiGraphics.method_51448().method_22909();
		}

		guiGraphics.method_51448().method_22909();

		renderMapWaypoints(guiGraphics, mouseX, mouseY);

		guiGraphics.method_51448().method_23760().method_23761().rotate((float) (((field_22787.field_1724.method_43078() + 180) / 180) * Math.PI), 0, 0, 1);
		guiGraphics.method_51448().method_22905(0.5f * AxolotlClientWaypoints.MINIMAP.arrowScale.get(), 0.5f * AxolotlClientWaypoints.MINIMAP.arrowScale.get(), 1);
		int arrowSize = 15;
		guiGraphics.method_51448().method_46416(-arrowSize / 2f, -arrowSize / 2f, 0);
		guiGraphics.method_25290(Minimap.arrowLocation, 0, 0, 0, 0, arrowSize, arrowSize, arrowSize, arrowSize);
		guiGraphics.method_51448().method_22909();
		super.method_25394(guiGraphics, mouseX, mouseY, partialTick);

		guiGraphics.method_27534(field_22793, AxolotlClientWaypoints.tr("position", String.valueOf(x), String.valueOf(getY(x, z)), String.valueOf(z)), field_22789 / 2, field_22790 - 15, Colors.GRAY.toInt());
	}

	private int getY(int x, int z) {
		int tileX = x / TILE_SIZE;
		int tileY = z / TILE_SIZE;
		if (x < 0 && x % TILE_SIZE != 0) {
			tileX -= 1;
		}
		if (z < 0 && z % TILE_SIZE != 0) {
			tileY -= 1;
		}
		var tile = tiles.get(new Vector2i(tileX, tileY));
		class_2791 c;
		if (tile == null || tile.tile == null) {
			c = field_22787.field_1687.method_8402(class_4076.method_18675(x), class_4076.method_18675(z), class_2806.field_12803, false);
		} else c = tile.tile.chunk.chunk();
		if (c == null) return field_22787.field_1687.method_31607();
		int y = c.method_12005(class_2902.class_2903.field_13202, x, z);
		if (atSurface) {
			return y;
		}
		y = Math.min(y, caveY);
		class_2680 blockState;
		var mutableBlockPos = new class_2338.class_2339(x, 0, z);
		do {
			mutableBlockPos.method_33098(--y);
			blockState = c.method_8320(mutableBlockPos);
		} while (blockState.method_26205(field_22787.field_1687, mutableBlockPos) == class_3620.field_16008 && y > field_22787.field_1687.method_31607());
		return y;
	}

	private int getWorldX(int guiX) {
		return class_3532.method_15357(field_22787.field_1724.method_23317() - ((field_22789 / 2f + dragOffset.x()) - guiX) / scale);
	}

	private int getWorldZ(int guiZ) {
		return class_3532.method_15357(field_22787.field_1724.method_23321() - ((field_22790 / 2f + dragOffset.y()) - guiZ) / scale);
	}

	private void renderMapWaypoints(class_332 graphics, int mouseX, int mouseY) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		graphics.method_51448().method_22903();
		var pos = new Vector3f();
		hoveredWaypoint = null;
		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			graphics.method_51448().method_22903();
			float posX = (float) (waypoint.x() - field_22787.field_1724.method_23317()) + 1;
			float posY = (float) (waypoint.z() - field_22787.field_1724.method_23321()) + 1;

			graphics.method_51448().method_46416(posX * scale, posY * scale, 0);
			pos.zero();
			graphics.method_51448().method_23760().method_23761().transformPosition(pos);

			int textWidth = field_22793.method_1727(waypoint.display());
			int width = textWidth + Waypoint.displayXOffset() * 2;
			int textHeight = field_22793.field_2000;
			int height = textHeight + Waypoint.displayYOffset() * 2;
			pos.sub(width / 2f, height / 2f, 0);
			graphics.method_25294(-(width / 2), -(height / 2), (width / 2), (height / 2), waypoint.color().toInt());
			graphics.method_51433(field_22793, waypoint.display(), -(textWidth / 2), -textHeight / 2, -1, false);
			if (hoveredWaypoint == null) {
				if (mouseX >= pos.x() && mouseY >= pos.y() && mouseX < pos.x() + width && mouseY < pos.y() + height) {
					hoveredWaypoint = waypoint;
					graphics.method_49601(-width / 2, -height / 2, width, height, Colors.WHITE.toInt());
				}
			}
			graphics.method_51448().method_22909();
		}
		graphics.method_51448().method_22909();
	}

	private void collectPlayerYData() {
		var level = field_22787.field_1687;
		if (allowCaves || (allowCavesNether && level.method_8597().comp_643())) {
			int playerX = field_22787.field_1724.method_31477();
			int playerZ = field_22787.field_1724.method_31479();
			var centerChunk = level.method_8497(class_4076.method_18675(playerX), class_4076.method_18675(playerZ));
			var surface = centerChunk.method_12005(class_2902.class_2903.field_13202, playerX, playerZ);
			class_2338.class_2339 mutableBlockPos = new class_2338.class_2339(playerX, surface, playerZ);
			int solidBlocksAbovePlayer = 0;
			atSurface = false;
			if (level.method_8597().comp_643()) {
				atSurface = field_22787.field_1724.method_31478() >= level.method_8597().comp_653();
			} else if (surface + 1 <= field_22787.field_1724.method_31478()) {
				atSurface = true;
			} else {
				while (solidBlocksAbovePlayer <= 3 && surface > field_22787.field_1724.method_31478() && surface > level.method_31607()) {
					class_2680 state = centerChunk.method_8320(mutableBlockPos);
					mutableBlockPos.method_33098(surface--);
					if (!(state.method_26167(level, mutableBlockPos.method_10074()) || !state.method_26225() || !state.method_26230(level, mutableBlockPos))) {
						solidBlocksAbovePlayer++;
					}
				}
				if (solidBlocksAbovePlayer <= 2) {
					atSurface = true;
				}
			}
		} else {
			atSurface = true;
		}

		caveY = field_22787.field_1724.method_31478();
		dimension = level.method_27983().method_29177();
	}

	private void createTiles() {
		int playerX = field_22787.field_1724.method_31477();
		int playerZ = field_22787.field_1724.method_31479();

		var playerTile = createTile(playerX, playerZ, atSurface, caveY);
		if (playerTile != null) {
			Map<Vector2i, LazyTile> loadedTiles = new HashMap<>(tiles);
			tiles.clear();
			tiles.put(new Vector2i(playerTile.tilePosX(), playerTile.tilePosY()), playerTile);
			triggerNeighbourLoad(playerTile, atSurface, caveY, playerTile);
			loadedTiles.forEach((v, t) -> {
				if (!tiles.containsKey(v)) {
					tiles.put(v, t);
				}
			});
		}
	}

	private void triggerNeighbourLoad(LazyTile tile, boolean atSurface, int caveY, LazyTile origin) {
		List<Runnable> queue = new ArrayList<>();
		loadNeighbour(tile, -1, -1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, -1, 0, atSurface, caveY, queue, origin);
		loadNeighbour(tile, -1, 1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 0, 1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 1, 1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 1, 0, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 1, -1, atSurface, caveY, queue, origin);
		loadNeighbour(tile, 0, -1, atSurface, caveY, queue, origin);
		if (!queue.isEmpty()) {
			Lists.reverse(queue).forEach(Runnable::run);
		}
	}

	private void loadNeighbour(LazyTile origin, int tileOffsetX, int tileOffsetY, boolean atSurface, int caveY, List<Runnable> loadQueue, LazyTile mapOrigin) {
		if (tileOffsetY != 0 || tileOffsetX != 0) {
			var tileXLeft = origin.tilePosX() + tileOffsetX;
			var tileYLeft = origin.tilePosY() + tileOffsetY;
			var vec = new Vector2i(tileXLeft, tileYLeft);
			if (!tiles.containsKey(vec)) {
				var anchorXLeft = tileXLeft * TILE_SIZE;
				var anchorYLeft = tileYLeft * TILE_SIZE;
				var tile = createTile(anchorXLeft, anchorYLeft, atSurface, caveY);
				if (tile != null) {
					tiles.put(vec, tile);
					loadQueue.add(() -> triggerNeighbourLoad(tile, atSurface, caveY, mapOrigin));
				}
			}
		}
	}

	private static class_2960 getTileRl(int tileX, int tileY) {
		return AxolotlClientWaypoints.rl("world_map/tile_" + tileX + "_" + tileY);
	}

	private LazyTile createTile(int anchorX, int anchorZ, boolean atSurface, int caveY) {
		return createTile(field_22787, anchorX, anchorZ, atSurface, caveY);
	}

	private static LazyTile createTile(class_310 minecraft, int anchorX, int anchorZ, boolean atSurface, int caveY) {
		anchorZ = anchorZ - anchorZ % TILE_SIZE;
		anchorX = anchorX - anchorX % TILE_SIZE;
		var level = minecraft.field_1687;
		int tileX = anchorX / TILE_SIZE;
		int tileY = anchorZ / TILE_SIZE;
		if (anchorX < 0 && anchorX % TILE_SIZE != 0) {
			tileX -= 1;
		}
		if (anchorZ < 0 && anchorZ % TILE_SIZE != 0) {
			tileY -= 1;
		}
		class_2791 tileChunk = level.method_8402(tileX, tileY, class_2806.field_12803, false);
		if (tileChunk != null) {
			int finalTileX = tileX;
			int finalTileY = tileY;
			return new LazyTile(tileX, tileY, () -> Tile.create(finalTileX, finalTileY, tileChunk));
		}
		return null;
	}

	public static void saveLoadedChunkTile(class_1923 pos) {
		class_310 minecraft = class_310.method_1551();
		var anchorX = pos.method_8326();
		var anchorZ = pos.method_8328();
		anchorZ = anchorZ - anchorZ % TILE_SIZE;
		anchorX = anchorX - anchorX % TILE_SIZE;
		var level = minecraft.field_1687;
		int tileX = anchorX / TILE_SIZE;
		int tileY = anchorZ / TILE_SIZE;
		if (anchorX < 0 && anchorX % TILE_SIZE != 0) {
			tileX -= 1;
		}
		if (anchorZ < 0 && anchorZ % TILE_SIZE != 0) {
			tileY -= 1;
		}
		class_2791 tileChunk = level.method_8402(tileX, tileY, class_2806.field_12803, false);
		if (tileChunk != null) {
			var dir = getCurrentLevelMapSaveDir();
			var out = dir.resolve("%d_%d%s".formatted(tileX, tileY, Tile.FILE_EXTENSION));
			try {
				Files.createDirectories(dir);
				new LevelChunkStorage.Entry(tileChunk).write(out);
			} catch (IOException e) {
				log.warn("Failed to save tile at {}, {}", tileX, tileY, e);
			}
		}
	}

	@Override
	public boolean method_25402(double mouseX, double mouseY, int button) {
		if (!super.method_25402(mouseX, mouseY, button)) {
			if (button == 1) {
				if (hoveredWaypoint != null) {
					field_22787.method_1507(new ContextMenuScreen(this, mouseX, mouseY, new ContextMenuScreen.Type.Waypoint(hoveredWaypoint)));
				} else {
					int worldX = getWorldX((int) mouseX);
					int worldZ = getWorldZ((int) mouseY);
					field_22787.method_1507(new ContextMenuScreen(this, mouseX, mouseY, new ContextMenuScreen.Type.Map(dimension.toString(), worldX, getY(worldX, worldZ), worldZ)));
				}
				return true;
			}
			return false;
		}
		return true;
	}

	@Override
	public boolean method_25403(double mouseX, double mouseY, int button, double dragX, double dragY) {
		if (!super.method_25403(mouseX, mouseY, button, dragX, dragY)) {
			if (button == 0) {
				dragOffset.add((float) dragX, (float) dragY, 0);
				return true;
			}
			return false;
		}
		return true;
	}

	@Override
	public boolean method_25401(double mouseX, double mouseY, double scrollY) {
		if (!super.method_25401(mouseX, mouseY, scrollY)) {
			if (scrollY > 0) {
				scale *= 2;
				var offsetX = field_22789 / 2f + dragOffset.x();
				var offsetY = field_22790 / 2f + dragOffset.y();
				var mirroredOnOffsetX = offsetX - (mouseX - offsetX);
				var mirroredOnOffsetY = offsetY - (mouseY - offsetY);
				dragOffset.set(mirroredOnOffsetX - field_22789 / 2f, mirroredOnOffsetY - field_22790 / 2f, 0);
			} else {
				scale /= 2;
				var offsetX = field_22789 / 2f + dragOffset.x();
				var offsetY = field_22790 / 2f + dragOffset.y();
				var mirroredOnOffsetX = offsetX + (mouseX - offsetX) / 2f;
				var mirroredOnOffsetY = offsetY + (mouseY - offsetY) / 2f;
				dragOffset.set(mirroredOnOffsetX - field_22789 / 2f, mirroredOnOffsetY - field_22790 / 2f, 0);
			}
		}
		return true;
	}

	@Override
	protected void method_25426() {
		method_37063(new ImageButton(4, field_22790 - 20, 16, 16, new WidgetSprites(OPTIONS_SPRITE, OPTIONS_SPRITE, OPTIONS_HOVERED_SPRITE),
			btn -> field_22787.method_1507(AxolotlClientWaypoints.createOptionsScreen(this)), AxolotlClientWaypoints.tr("options")));
		var slider = method_37063(new class_357(field_22789 - 150, 20, 150, 20, AxolotlClientWaypoints.tr("player_y"), 0) {
			final int min = field_22787.field_1687.method_31607();
			final int max = field_22787.field_1687.method_31600() + 1;

			@Override
			protected void method_25346() {
				if (field_22753 == 0 || Math.floor((max - min) * field_22753) == 0) {
					method_25355(AxolotlClientWaypoints.tr("player_y"));
				} else {
					method_25355(class_2561.method_43470(String.valueOf(caveY - 1)));
				}
			}

			@Override
			protected void method_25344() {
				caveY = (int) (min + (max - min) * field_22753);
				atSurface = false;
				if (field_22753 == 0 || Math.floor((max - min) * field_22753) == 0) {
					collectPlayerYData();
				}
				updateTiles();
			}
		});
		method_37063(new DropdownButton(field_22789 - 20, 0, 20, 20,
			AxolotlClientWaypoints.tr("open_dropdown"), (btn, val) -> slider.field_22764 = val));
		slider.field_22764 = false;
		optionUpdate = () -> {
			var allowsCaves = allowCaves || (allowCavesNether && field_22787.field_1687.method_8597().comp_643());
			boolean updated = slider.field_22763 != allowsCaves;
			if (updated) {
				slider.field_22763 = allowsCaves;
				if (!allowsCaves) {
					atSurface = true;
					updateTiles();
				} else {
					slider.method_25344();
				}
			}
		};
		AxolotlClientWaypoints.NETWORK_LISTENER.postReceive.add(optionUpdate);
		optionUpdate.run();
		if (tiles.isEmpty()) {
			if (!initializedOnce) {
				collectPlayerYData();
			}
			CompletableFuture.runAsync(() -> {
				loadSavedTiles();
				createTiles();
			});
		}
		initializedOnce = true;
	}

	private void updateTiles() {
		CompletableFuture.runAsync(() -> tiles.values().forEach(t -> t.update(caveY, atSurface, field_22787.field_1687)));
	}

	private void loadSavedTiles() {
		var dir = getCurrentLevelMapSaveDir();
		if (Files.exists(dir)) {
			try (var s = Files.list(dir)) {
				s.forEach(file -> {
					var name = file.getFileName().toString();
					if (!name.endsWith(Tile.FILE_EXTENSION)) {
						return;
					}
					name = name.substring(0, name.indexOf("."));
					var coords = name.split("_");
					int x = Integer.parseInt(coords[0]);
					int y = Integer.parseInt(coords[1]);
					var key = new Vector2i(x, y);
					if (!tiles.containsKey(key)) {
						try {
							var tile = new LazyTile(x, y, () -> {
								try {
									return Tile.read(file, minecraft.level);
								} catch (IOException e) {
									return null;
								}
							});
							tiles.put(key, tile);
						} catch (Exception e) {
							log.warn("Failed to load tile at {}, {}", x, y, e);
						}
					}
				});
			} catch (IOException e) {
				log.info("Failed to load saved world map tiles!", e);
			}
		}
	}

	private void saveTiles() {
		var dir = getCurrentLevelMapSaveDir();
		try {
			Files.createDirectories(dir);

			tiles.values().forEach((tile) -> saveTile(tile, dir));
		} catch (IOException e) {
			log.error("Failed to create world map save dir!", e);
		}
	}

	private static void saveTile(LazyTile tile, Path dir) {
		try {
			tile.save(dir);
		} catch (IOException e) {
			log.warn("Failed to save tile at {}, {}", tile.tilePosX(), tile.tilePosY(), e);
		}
	}

	private static Path getCurrentLevelMapSaveDir() {
		return AxolotlClientWaypoints.getCurrentStorageDir().resolve("worldmap");
	}

	@Override
	public void method_25432() {
		if (optionUpdate != null) {
			AxolotlClientWaypoints.NETWORK_LISTENER.postReceive.remove(optionUpdate);
		}
		if (field_22787.field_1755 == null) {
			saveTiles();
			tiles.values().forEach(LazyTile::release);
			tiles.clear();
		}
	}

	@RequiredArgsConstructor
	private static class LazyTile {
		private Tile tile;
		@Getter
		@Accessors(fluent = true)
		private final int tilePosX, tilePosY;
		private final Supplier<Tile> supplier;
		private final Vector3f pos = new Vector3f();
		private boolean loaded;

		public void render(class_332 guiGraphics, float playerX, float playerZ, float scale, float delta, int caveY, boolean atSurface) {
			float x = tilePosX() * TILE_SIZE - playerX;
			float y = tilePosY() * TILE_SIZE - playerZ;
			guiGraphics.method_51448().method_22903();
			guiGraphics.method_51448().method_46416(x, y, 0);
			pos.zero();
			guiGraphics.method_51448().method_23760().method_23761().transformPosition(pos);
			if (pos.x + TILE_SIZE * scale >= 0 && pos.x < guiGraphics.method_51421() && pos.y + TILE_SIZE * scale >= 0 && pos.y < guiGraphics.method_51443()) {
				if (tile == null) {
					if (!loaded) {
						load().thenRunAsync(() -> {
							if (tile != null) {
								tile.update(caveY, atSurface, class_310.method_1551().field_1687);
							}
						});
					}
				} else {
					// FIXME - floating point precision errors?
					guiGraphics.method_25290(tile.rl(), 0, 0, 0, 0, TILE_SIZE, TILE_SIZE, TILE_SIZE, TILE_SIZE);
				}
			}
			guiGraphics.method_51448().method_22909();
		}

		public CompletableFuture<?> load() {
			if (!loaded) {
				loaded = true;
				return class_310.method_1551().method_5385(supplier).thenApply(t -> tile = t);
			}
			return CompletableFuture.completedFuture(null);
		}

		public void release() {
			if (tile != null) {
				tile.release();
				tile = null;
			}
		}

		public void save(Path dir) throws IOException {
			if (tile != null) {
				tile.save(dir);
			}
		}

		public void update(int caveY, boolean atSurface, class_1937 level) {
			if (tile != null) {
				tile.update(caveY, atSurface, level);
			}
		}
	}

	private record Tile(int tilePosX, int tilePosY, class_2960 rl, class_1043 tex,
						LevelChunkStorage.Entry chunk) {
		public static final String FILE_EXTENSION = ".bin";

		public void release() {
			class_310.method_1551().method_1531().method_4615(rl);
		}

		public void save(Path dir) throws IOException {
			var out = dir.resolve("%d_%d%s".formatted(tilePosX, tilePosY, FILE_EXTENSION));
			chunk.write(out);
		}

		public static Tile create(int x, int y, class_2791 chunk) {
			return create(x, y, new LevelChunkStorage.Entry(chunk));
		}

		public static Tile create(int x, int y, LevelChunkStorage.Entry chunk) {
			var rl = getTileRl(x, y);
			var tex = new class_1043(new class_1011(TILE_SIZE, TILE_SIZE, false));
			tex.method_4525().method_4326(0, 0, TILE_SIZE, TILE_SIZE, Colors.BLACK.toInt());
			class_310.method_1551().method_1531().method_4616(rl, tex);
			return new Tile(x, y, rl, tex, chunk);
		}

		public static Tile read(Path p, class_1937 level) throws IOException {
			var name = p.getFileName().toString();
			name = name.substring(0, name.indexOf("."));
			var coords = name.split("_");
			int x = Integer.parseInt(coords[0]);
			int y = Integer.parseInt(coords[1]);
			return create(x, y, LevelChunkStorage.Entry.read(p, level));
		}

		public void update(int caveY, boolean atSurface, class_1937 level) {
			if (chunk.chunk() instanceof class_2818 levelChunk && levelChunk.method_12223()) {
				tex.method_4525().method_4326(0, 0, TILE_SIZE, TILE_SIZE, Colors.BLACK.toInt());
				tex.method_4524();
				return;
			}
			int levelMinY = level.method_31607();
			int centerX = (tilePosX * TILE_SIZE) + TILE_SIZE / 2;
			int centerZ = (tilePosY * TILE_SIZE) + TILE_SIZE / 2;

			var pixels = tex.method_4525();
			int size = pixels.method_4307();
			int texHalfWidth = size / 2;

			class_2338.class_2339 mutableBlockPos = new class_2338.class_2339();
			class_2338.class_2339 mutableBlockPos2 = new class_2338.class_2339();
			boolean updated = false;
			for (int x = 0; x < size; x++) {
				double d = 0;
				for (int z = -1; z < size; z++) {
					int chunkX = (centerX + x - texHalfWidth);
					int chunkZ = (centerZ + z - texHalfWidth);

					int fluidDepth = 0;
					double e = 0.0;
					mutableBlockPos.method_10103(chunkX, 0, chunkZ);
					class_2791 levelChunk;
					int y;
					if (z < 0 || z >= TILE_SIZE) {
						mutableBlockPos.method_33099(chunkZ + 1);
					}
					levelChunk = chunk.chunk();
					y = levelChunk.method_12005(class_2902.class_2903.field_13202, mutableBlockPos.method_10263(), mutableBlockPos.method_10260()) + 1;
					if (!atSurface) {
						y = Math.min(y, caveY);
					}
					class_2680 blockState;
					if (y <= levelMinY) {
						blockState = class_2246.field_10124.method_9564();
					} else {
						do {
							mutableBlockPos.method_33098(--y);
							blockState = levelChunk.method_8320(mutableBlockPos);
						} while (blockState.method_26205(level, mutableBlockPos) == class_3620.field_16008 && y > levelMinY);

						if (y > levelMinY && !blockState.method_26227().method_15769()) {
							int highestFullBlockY = y - 1;
							mutableBlockPos2.method_10101(mutableBlockPos);

							class_2680 blockState2;
							do {
								mutableBlockPos2.method_33098(highestFullBlockY--);
								blockState2 = levelChunk.method_8320(mutableBlockPos2);
								fluidDepth++;
							} while (highestFullBlockY > levelMinY && !blockState2.method_26227().method_15769());

							class_3610 fluidState = blockState.method_26227();
							blockState = !fluidState.method_15769() && !blockState.method_26206(level, mutableBlockPos, class_2350.field_11036) ? fluidState.method_15759() : blockState;
						}
					}

					e += y;
					var mapColor = blockState.method_26205(level, mutableBlockPos);

					int color;
					if (mapColor == class_3620.field_16019) {
						var floorBlock = levelChunk.method_8320(mutableBlockPos2);
						var floorColor = floorBlock.method_26205(level, mutableBlockPos2).field_16011;
						int biomeColor = class_1163.method_4961(level, mutableBlockPos);
						float shade = level.method_24852(class_2350.field_11036, true);
						int waterColor = biomeColor;
						waterColor = ARGB.colorFromFloat(1f, ARGB.redFloat(waterColor) * shade, ARGB.greenFloat(waterColor) * shade, ARGB.blueFloat(waterColor) * shade);
						waterColor = ARGB.average(waterColor, ARGB.scaleRGB(floorColor, 1f - fluidDepth / 15f));
						color = waterColor;
					} else {
						double f = (e - d) * 4.0 / (1 + 4) + ((x + z & 1) - 0.5) * 0.4;
						class_3620.class_6594 brightness;
						if (f > 0.6) {
							brightness = class_3620.class_6594.field_34761;
						} else if (f < -0.6) {
							brightness = class_3620.class_6594.field_34759;
						} else {
							brightness = class_3620.class_6594.field_34760;
						}
						color = mapColor.method_15820(brightness);
					}

					d = e;

					if (z >= 0 && Integer.rotateRight(pixels.method_4315(x, z), 4) != color) {
						pixels.method_4305(x, z, ARGB.opaque(color));
						updated = true;
					}
				}
			}

			if (updated) {
				class_310.method_1551().method_20493(tex::method_4524);
			}
		}
	}
}
