/*
 * Copyright © 2025 moehreag <moehreag@gmail.com> & Contributors
 *
 * This file is part of AxolotlClient (Waypoints Mod).
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * For more information, see the LICENSE file.
 */

package io.github.axolotlclient.waypoints.waypoints;

import java.lang.Math;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.class_1921;
import net.minecraft.class_243;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_4184;
import net.minecraft.class_4587;
import net.minecraft.class_4597;
import com.mojang.blaze3d.systems.RenderSystem;
import io.github.axolotlclient.AxolotlClientConfig.api.util.Colors;
import io.github.axolotlclient.waypoints.AxolotlClientWaypoints;
import io.github.axolotlclient.waypoints.mixin.GameRendererAccessor;
import org.jetbrains.annotations.Nullable;
import org.joml.*;

public class WaypointRenderer {

	private static final double CUTOFF_DIST = 12;
	private final class_310 minecraft = class_310.method_1551();
	private final Matrix4f view = new Matrix4f();
	private final Vector4f viewProj = new Vector4f();

	public void render(class_4587 stack, class_4597.class_4598 source, float tick) {
		var profiler = class_310.method_1551().method_16011();
		profiler.method_15405("waypoints");
		stack.method_22903();
		var cam = minecraft.field_1773.method_19418();
		var camPos = cam.method_19326();
		minecraft.field_1773.method_22709(minecraft.field_1773.method_22973(((GameRendererAccessor) minecraft.field_1773).invokeGetFov(cam, 0, false)));
		RenderSystem.enableBlend();
		float fov = (float) ((GameRendererAccessor) minecraft.field_1773).invokeGetFov(cam, tick, true);

		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			if (waypoint.closerToThan(camPos.method_10216(), camPos.method_10214(), camPos.method_10215(), CUTOFF_DIST / minecraft.method_22683().method_4495())) {
				profiler.method_15396(waypoint.name());
				renderWaypoint(waypoint, stack, camPos, cam, source, fov);
				profiler.method_15407();
			}
		}

		stack.method_22909();
		if (!stack.method_22911()) {
			throw new IllegalStateException("Pose stack not empty");
		}
	}

	private void renderWaypoint(Waypoint waypoint, class_4587 stack, class_243 camPos, class_4184 cam, class_4597.class_4598 bufferSource, float fov) {
		int textWidth = minecraft.field_1772.method_1727(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.field_1772.field_2000;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		var displayStart = projectToScreen(cam, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(-(width / 2f * 0.04f), (height / 2f * 0.04f)));
		if (displayStart == null) return;
		var displayEnd = projectToScreen(cam, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(width / 2f * 0.04f, -(height / 2f * 0.04f)));
		if (displayEnd == null) return;
		float projWidth = Math.abs(displayEnd.x() - displayStart.x());
		float projHeight = Math.abs(displayEnd.y() - displayStart.y());
		if (projWidth < width && projHeight < height) {
			return;
		}

		stack.method_22903();
		stack.method_22904(waypoint.x() - camPos.method_10216(), waypoint.y() - camPos.method_10214(), waypoint.z() - camPos.method_10215());
		stack.method_22907(cam.method_23767());
		float scale = 0.04F;
		stack.method_22905(-scale, -scale, scale);
		//fillRect(stack, bufferSource, -width / 2f, -height / 2f, 0f, width / 2f, height / 2f, waypoint.color().toInt());
		drawFontBatch(waypoint.display(), -textWidth / 2f, -textHeight / 2f, stack.method_23760().method_23761(), bufferSource, waypoint.color().toInt());
		stack.method_22909();
	}

	private void fillRect(class_4587 stack, class_4597.class_4598 source, float x, float y, float z, float x2, float y2, int color) {
		var buf = source.getBuffer(class_1921.method_49046());
		var matrix = stack.method_23760().method_23761();
		buf.method_22918(matrix, x, y, z).method_39415(color).method_22916(0xF000F0).method_1344();
		buf.method_22918(matrix, x, y2, z).method_39415(color).method_22916(0xF000F0).method_1344();
		buf.method_22918(matrix, x2, y2, z).method_39415(color).method_22916(0xF000F0).method_1344();
		buf.method_22918(matrix, x2, y, z).method_39415(color).method_22916(0xF000F0).method_1344();
	}

	private void drawFontBatch(String text, float x, float y, Matrix4f matrix, class_4597 bufferSource, int bgColor) {
		minecraft.field_1772.method_27521(text, x, y, -1, false, matrix, bufferSource, class_327.class_6415.field_33993, bgColor, 0xF000F0);
	}

	public void renderWaypoints(class_332 graphics, float delta) {
		if (!AxolotlClientWaypoints.renderWaypoints.get()) return;
		if (!AxolotlClientWaypoints.renderWaypointsInWorld.get()) return;
		var profiler = class_310.method_1551().method_16011();
		var cam = minecraft.field_1773.method_19418();
		profiler.method_15396("waypoints");

		graphics.method_51448().method_22903();
		graphics.method_51448().method_46416(0.0F, 0.0F, -100.0F);
		var positionDrawer = new AtomicReference<Runnable>();
		for (Waypoint waypoint : AxolotlClientWaypoints.getCurrentWaypoints()) {
			graphics.method_51448().method_22903();
			renderWaypoint(waypoint, graphics, delta, cam, positionDrawer);
			graphics.method_51448().method_22909();
		}
		if (positionDrawer.get() != null) {
			positionDrawer.get().run();
		}
		graphics.method_51448().method_22909();
		profiler.method_15407();
	}

	private void renderWaypoint(Waypoint waypoint, class_332 graphics, float tick, class_4184 camera, AtomicReference<Runnable> positionDrawn) {
		var fov = ((GameRendererAccessor) minecraft.field_1773).invokeGetFov(camera, tick, false);
		var pose = graphics.method_51448();

		var textWidth = minecraft.field_1772.method_1727(waypoint.display());
		int width = textWidth + Waypoint.displayXOffset() * 2;
		int textHeight = minecraft.field_1772.field_2000;
		int height = textHeight + Waypoint.displayYOffset() * 2;
		var camPos = camera.method_19326();

		var displayStart = projectToScreen(camera, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f(-(width / 2f * 0.04f), (height / 2f * 0.04f)));
		var displayEnd = projectToScreen(camera, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), new Vector2f((width / 2f * 0.04f), -(height / 2f * 0.04f)));
		Result result = projectToScreen(camera, fov, width, height, waypoint.x(), waypoint.y(), waypoint.z(), null);
		if (result == null) return;
		float projWidth;
		float projHeight;
		if (displayStart != null && displayEnd != null) {
			projWidth = Math.abs(displayEnd.x() - displayStart.x());
			projHeight = Math.abs(displayEnd.y() - displayStart.y());
		} else {
			projWidth = 0;
			projHeight = 0;
		}

		pose.method_46416(result.x(), result.y(), 0);
		boolean outOfView = result.x() < -width / 2f || result.x() > graphics.method_51421() + width / 2f || result.y() < -height / 2f || result.y() > graphics.method_51443() + height / 2f;
		if (!AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get() && outOfView) {
			return;
		}

		boolean _3dOnScreen;
		if (displayEnd != null && displayStart != null) {
			float minX = displayStart.x();
			float minY = displayStart.y();
			float maxX = displayEnd.x();
			float maxY = displayEnd.y();
			int guiWidth = graphics.method_51421();
			int guiHeight = graphics.method_51443();
			_3dOnScreen = minX > 0 && minY > 0 && minX < guiWidth && minY < guiHeight ||
				minX > 0 && maxY > 0 && minX < guiWidth && maxY < guiHeight ||
				maxX > 0 && maxY > 0 && maxX < guiWidth && maxY < guiHeight ||
				maxX > 0 && minY > 0 && maxX < guiWidth && minY < guiHeight;
		} else {
			_3dOnScreen = false;
		}
		if (positionDrawn.get() == null && Math.abs(result.x() - graphics.method_51421() / 2f) < (_3dOnScreen ? Math.max(projWidth, width) : width) / 2f && Math.abs(result.y() - graphics.method_51443() / 2f) < (_3dOnScreen ? Math.max(height, projHeight) : height) / 2f) {
			pose.method_22903();
			pose.method_46416(0, Math.max(height, projHeight + 4) / 2f + 4, 0);
			var pos = pose.method_23760().method_23761().transformPosition(new Vector3f());
			if ((projWidth >= width || projHeight >= height) && _3dOnScreen) {
				pos.y = Math.min(pos.y, displayEnd.y()+6);
			}
			positionDrawn.set(() -> {
				var line1 = waypoint.name();
				pose.method_22903();
				pose.method_23760().method_23761().translate(pos);
				int line1W = minecraft.field_1772.method_1727(line1);
				graphics.method_25294(-line1W / 2 - 2, -2, line1W / 2 + 2, minecraft.field_1772.field_2000 + 2, Colors.GRAY.withAlpha(100).toInt());
				graphics.method_49601(-line1W / 2 - 2, -2, line1W + 4, minecraft.field_1772.field_2000 + 4, Colors.GRAY.toInt());
				graphics.method_51433(minecraft.field_1772, line1, -line1W / 2, 0, -1, true);
				if (!waypoint.closerToThan(camPos.method_10216(), camPos.method_10214(), camPos.method_10215(), CUTOFF_DIST)) {
					pose.method_46416(0, minecraft.field_1772.field_2000 + 4, 0);
					var line2 = AxolotlClientWaypoints.tr("distance", "%.2f".formatted(waypoint.distTo(camPos.method_10216(), camPos.method_10214(), camPos.method_10215())));
					graphics.method_51439(minecraft.field_1772, line2, -minecraft.field_1772.method_27525(line2) / 2, 0, -1, false);
				}
				pose.method_22909();
			});
			pose.method_22909();
		}

		if ((projWidth >= width || projHeight >= height) && _3dOnScreen) {
			return;
		}

		graphics.method_25294(-width / 2, -height / 2, width / 2, height / 2, waypoint.color().toInt());
		graphics.method_51433(minecraft.field_1772, waypoint.display(), -textWidth / 2, -textHeight / 2, -1, false);
	}

	private @Nullable Result projectToScreen(class_4184 camera, double fov, int width, int height, double x, double y, double z, Vector2f orthoOffset) {
		viewProj.set(x, y, z, 1);
		if (orthoOffset != null) {
			var vec = new Matrix4f();
			var camRot = camera.method_23767().rotateY((float) -(Math.PI), new Quaternionf()).invert();
			vec.rotate(camRot.invert(new Quaternionf()));
			vec.translate(orthoOffset.x(), orthoOffset.y(), 0);
			vec.rotate(camRot);
			vec.transform(viewProj);
		}
		view.rotation(camera.method_23767().rotateY((float) -(Math.PI), new Quaternionf()).invert()).translate(camera.method_19326().method_46409().negate());

		Matrix4f projection = minecraft.field_1773.method_22973(fov);
		projection.mul(view);
		viewProj.mul(projection);


		if (orthoOffset == null && AxolotlClientWaypoints.renderOutOfViewWaypointsOnScreenEdge.get()) {
			viewProj.w = Math.max(Math.abs(viewProj.x()), Math.max(Math.abs(viewProj.y()), viewProj.w()));
		}

		if (viewProj.w() <= 0) {
			return null;
		}
		viewProj.div(viewProj.w());

		float projX = viewProj.x();
		float projY = viewProj.y();

		//float x = (graphics.guiWidth()/2f) + ((graphics.guiWidth() - width) * (viewProj.x() / 2f));
		float resultX = 0.5f * (minecraft.method_22683().method_4486() * (projX + 1) - width * projX);
		//float y = graphics.guiHeight() - (graphics.guiHeight()/2f + (graphics.guiHeight()-height) * (viewProj.y() / 2f));
		float resultY = minecraft.method_22683().method_4486() * (0.5f - projY / 2) + (height * projY) / 2f;
		return new Result(resultX, resultY);
	}

	private record Result(float x, float y) {
	}
}
