/*
 * Decompiled with CFR 0.152.
 */
package eu.avalanche7.paradigm.utils;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import eu.avalanche7.paradigm.utils.DebugLogger;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JsonValidator {
    private static final Logger LOGGER = LoggerFactory.getLogger(JsonValidator.class);
    private static final Gson gson = new GsonBuilder().setPrettyPrinting().disableHtmlEscaping().create();
    private final DebugLogger debugLogger;

    public JsonValidator(DebugLogger debugLogger) {
        this.debugLogger = debugLogger;
    }

    public ValidationResult validateAndFix(String jsonString) {
        if (jsonString == null || jsonString.trim().isEmpty()) {
            return new ValidationResult(false, "JSON string is null or empty", jsonString != null ? jsonString : "", List.of("Input is null or empty"));
        }
        ArrayList<String> issues = new ArrayList<String>();
        String originalJson = jsonString.trim();
        try {
            JsonElement element = JsonParser.parseString((String)originalJson);
            return new ValidationResult(true, "JSON is valid", originalJson, issues);
        }
        catch (JsonParseException e) {
            LOGGER.warn("[Paradigm] JSON syntax error detected: {}", (Object)e.getMessage());
            issues.add("Syntax error: " + e.getMessage());
            String fixedJson = this.attemptJsonFixes(originalJson, issues);
            try {
                JsonParser.parseString((String)fixedJson);
                LOGGER.info("[Paradigm] JSON syntax has been automatically fixed");
                return new ValidationResult(true, "JSON syntax fixed", fixedJson, issues);
            }
            catch (JsonParseException fixError) {
                LOGGER.error("[Paradigm] Unable to automatically fix JSON syntax errors: {}", (Object)fixError.getMessage());
                issues.add("Failed to fix: " + fixError.getMessage());
                return new ValidationResult(false, "Unable to fix JSON syntax errors", originalJson, issues);
            }
        }
    }

    private String attemptJsonFixes(String json, List<String> issues) {
        String result = json;
        result = this.removeComments(result, issues);
        result = this.fixQuotes(result, issues);
        result = this.fixUnquotedProperties(result, issues);
        result = this.fixTrailingCommas(result, issues);
        result = this.fixMissingCommas(result, issues);
        result = this.fixMissingBrackets(result, issues);
        return result;
    }

    private String removeComments(String json, List<String> issues) {
        if (!json.contains("//") && !json.contains("/*")) {
            return json;
        }
        StringBuilder result = new StringBuilder();
        boolean inString = false;
        boolean escaped = false;
        for (int i = 0; i < json.length(); ++i) {
            char next;
            char c = json.charAt(i);
            char c2 = next = i + 1 < json.length() ? json.charAt(i + 1) : (char)'\u0000';
            if (escaped) {
                result.append(c);
                escaped = false;
                continue;
            }
            if (c == '\\' && inString) {
                result.append(c);
                escaped = true;
                continue;
            }
            if (c == '\"') {
                inString = !inString;
                result.append(c);
                continue;
            }
            if (!inString) {
                if (c == '/' && next == '/') {
                    while (i < json.length() && json.charAt(i) != '\n') {
                        ++i;
                    }
                    if (i < json.length()) {
                        result.append('\n');
                    }
                    issues.add("Removed single-line comments");
                    continue;
                }
                if (c == '/' && next == '*') {
                    i += 2;
                    while (i + 1 < json.length() && (json.charAt(i) != '*' || json.charAt(i + 1) != '/')) {
                        ++i;
                    }
                    if (i + 1 < json.length()) {
                        i += 2;
                    }
                    issues.add("Removed multi-line comments");
                    continue;
                }
            }
            result.append(c);
        }
        return result.toString();
    }

    private String fixQuotes(String json, List<String> issues) {
        StringBuilder result = new StringBuilder();
        boolean inDoubleString = false;
        boolean escaped = false;
        boolean foundSingleQuotes = false;
        for (int i = 0; i < json.length(); ++i) {
            char c = json.charAt(i);
            if (escaped) {
                result.append(c);
                escaped = false;
                continue;
            }
            if (c == '\\') {
                result.append(c);
                escaped = true;
                continue;
            }
            if (c == '\"') {
                inDoubleString = !inDoubleString;
                result.append(c);
                continue;
            }
            if (c == '\'' && !inDoubleString) {
                result.append('\"');
                foundSingleQuotes = true;
                continue;
            }
            result.append(c);
        }
        if (foundSingleQuotes) {
            issues.add("Fixed single quotes to double quotes");
        }
        return result.toString();
    }

    private String fixUnquotedProperties(String json, List<String> issues) {
        StringBuilder result = new StringBuilder();
        String[] lines = json.split("\n");
        boolean foundIssues = false;
        for (String line : lines) {
            String trimmed = line.trim();
            if (trimmed.matches("^\\s*[a-zA-Z_][a-zA-Z0-9_]*\\s*:.*") && !trimmed.matches("^\\s*\".*\"\\s*:.*")) {
                int colonIndex = trimmed.indexOf(58);
                if (colonIndex > 0) {
                    String propertyName = trimmed.substring(0, colonIndex).trim();
                    String rest = trimmed.substring(colonIndex);
                    String indent = line.substring(0, line.indexOf(line.trim()));
                    result.append(indent).append('\"').append(propertyName).append('\"').append(rest).append('\n');
                    foundIssues = true;
                    continue;
                }
                result.append(line).append('\n');
                continue;
            }
            result.append(line).append('\n');
        }
        if (foundIssues) {
            issues.add("Added quotes around property names");
        }
        return result.toString().trim();
    }

    private String fixTrailingCommas(String json, List<String> issues) {
        String[] lines = json.split("\n");
        StringBuilder result = new StringBuilder();
        boolean foundIssues = false;
        for (int i = 0; i < lines.length; ++i) {
            String nextLine;
            String line = lines[i];
            String trimmed = line.trim();
            String string = nextLine = i + 1 < lines.length ? lines[i + 1].trim() : "";
            if (trimmed.endsWith(",") && (nextLine.startsWith("}") || nextLine.startsWith("]"))) {
                int commaIndex = line.lastIndexOf(44);
                result.append(line.substring(0, commaIndex)).append('\n');
                foundIssues = true;
                continue;
            }
            result.append(line).append('\n');
        }
        if (foundIssues) {
            issues.add("Removed trailing commas");
        }
        return result.toString().trim();
    }

    private String fixMissingCommas(String json, List<String> issues) {
        String[] lines = json.split("\n");
        StringBuilder result = new StringBuilder();
        boolean foundIssues = false;
        for (int i = 0; i < lines.length; ++i) {
            String nextLine;
            String line = lines[i];
            String trimmed = line.trim();
            String string = nextLine = i + 1 < lines.length ? lines[i + 1].trim() : "";
            if (trimmed.endsWith("\"") && !trimmed.contains(":") && nextLine.startsWith("\"") && !nextLine.isEmpty() && !trimmed.endsWith("\",")) {
                result.append(line).append(',').append('\n');
                foundIssues = true;
                continue;
            }
            if (trimmed.endsWith("}") && nextLine.startsWith("\"") && !trimmed.endsWith("},")) {
                result.append(line).append(',').append('\n');
                foundIssues = true;
                continue;
            }
            result.append(line).append('\n');
        }
        if (foundIssues) {
            issues.add("Added missing commas");
        }
        return result.toString().trim();
    }

    private String fixMissingBrackets(String json, List<String> issues) {
        int i;
        int missing;
        int openBraces = 0;
        int closeBraces = 0;
        int openBrackets = 0;
        int closeBrackets = 0;
        boolean inString = false;
        boolean escaped = false;
        block6: for (int i2 = 0; i2 < json.length(); ++i2) {
            char c = json.charAt(i2);
            if (escaped) {
                escaped = false;
                continue;
            }
            if (c == '\\') {
                escaped = true;
                continue;
            }
            if (c == '\"') {
                inString = !inString;
                continue;
            }
            if (inString) continue;
            switch (c) {
                case '{': {
                    ++openBraces;
                    continue block6;
                }
                case '}': {
                    ++closeBraces;
                    continue block6;
                }
                case '[': {
                    ++openBrackets;
                    continue block6;
                }
                case ']': {
                    ++closeBrackets;
                }
            }
        }
        StringBuilder result = new StringBuilder(json);
        boolean foundIssues = false;
        if (openBraces > closeBraces) {
            missing = openBraces - closeBraces;
            for (i = 0; i < missing; ++i) {
                result.append('\n').append('}');
            }
            issues.add("Added " + missing + " missing closing brace(s)");
            foundIssues = true;
        }
        if (openBrackets > closeBrackets) {
            missing = openBrackets - closeBrackets;
            for (i = 0; i < missing; ++i) {
                result.append('\n').append(']');
            }
            issues.add("Added " + missing + " missing closing bracket(s)");
            foundIssues = true;
        }
        return result.toString();
    }

    public boolean isValidJson(String jsonString) {
        try {
            JsonParser.parseString((String)jsonString);
            return true;
        }
        catch (JsonParseException e) {
            return false;
        }
    }

    public String formatJson(String jsonString) {
        try {
            JsonElement element = JsonParser.parseString((String)jsonString);
            return gson.toJson(element);
        }
        catch (JsonParseException e) {
            this.debugLogger.debugLog("Failed to format JSON: " + e.getMessage());
            return jsonString;
        }
    }

    public static class ValidationResult {
        private final boolean valid;
        private final String message;
        private final String fixedJson;
        private final List<String> issues;

        public ValidationResult(boolean valid, String message, String fixedJson, List<String> issues) {
            this.valid = valid;
            this.message = message;
            this.fixedJson = fixedJson;
            this.issues = new ArrayList<String>(issues);
        }

        public boolean isValid() {
            return this.valid;
        }

        public String getMessage() {
            return this.message;
        }

        public String getFixedJson() {
            return this.fixedJson;
        }

        public List<String> getIssues() {
            return new ArrayList<String>(this.issues);
        }

        public boolean hasIssues() {
            return !this.issues.isEmpty();
        }

        public String getIssuesSummary() {
            if (this.issues.isEmpty()) {
                return "No issues found";
            }
            return String.join((CharSequence)", ", this.issues);
        }
    }
}

