/*
 * Decompiled with CFR 0.152.
 */
package eu.avalanche7.paradigm.utils;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import eu.avalanche7.paradigm.configs.MainConfigHandler;
import eu.avalanche7.paradigm.platform.Interfaces.IComponent;
import eu.avalanche7.paradigm.platform.Interfaces.IPlatformAdapter;
import eu.avalanche7.paradigm.utils.DebugLogger;
import eu.avalanche7.paradigm.utils.MessageParser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import net.minecraft.server.MinecraftServer;
import net.neoforged.fml.loading.FMLPaths;
import org.slf4j.Logger;

public class Lang {
    private final Logger logger;
    private final Path langFolder = FMLPaths.GAMEDIR.get().resolve("config/paradigm/lang");
    private final Map<String, String> translations = new HashMap<String, String>();
    private String currentLanguage;
    private final MainConfigHandler.Config mainConfig;
    private final MessageParser messageParser;
    private final IPlatformAdapter platformAdapter;

    public Lang(Logger logger, MainConfigHandler.Config mainConfig, MessageParser messageParser, IPlatformAdapter platformAdapter) {
        this.logger = logger;
        this.mainConfig = mainConfig;
        this.messageParser = messageParser;
        this.platformAdapter = platformAdapter;
        try {
            this.ensureDefaultLangFiles();
        }
        catch (Exception e) {
            if (this.logger != null) {
                this.logger.error("Failed to initialize Lang class", (Throwable)e);
            }
            System.err.println("Failed to initialize Lang class and logger is null: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void initializeLanguage() {
        if (this.mainConfig == null || this.mainConfig.defaultLanguage == null) {
            if (this.logger != null) {
                this.logger.error("Lang: MainConfig or defaultLanguage setting is null. Cannot initialize language.");
            }
            return;
        }
        String language = this.mainConfig.defaultLanguage.get();
        if (this.logger != null) {
            this.logger.info("Paradigm: Loaded language setting: {}", (Object)language);
        }
        this.loadLanguage(language);
    }

    public void loadLanguage(String language) {
        block14: {
            if (this.logger != null) {
                this.logger.info("Paradigm: Attempting to load language: {}", (Object)language);
            }
            Gson gson = new Gson();
            Type type = new TypeToken<Map<String, Object>>(this){}.getType();
            Path langFile = this.langFolder.resolve(language + ".json");
            this.translations.clear();
            this.loadFromResource("en", gson, type);
            if (!"en".equalsIgnoreCase(language)) {
                this.loadFromResource(language, gson, type);
            }
            if (Files.exists(langFile, new LinkOption[0])) {
                try (BufferedReader reader = Files.newBufferedReader(langFile, StandardCharsets.UTF_8);){
                    Map rawMap = (Map)gson.fromJson((Reader)reader, type);
                    this.flattenMap("", rawMap);
                    if (this.logger != null) {
                        this.logger.info("Paradigm: Successfully loaded language from config: {}", (Object)language);
                    }
                    break block14;
                }
                catch (Exception e) {
                    if (this.logger != null) {
                        this.logger.error("Paradigm: Failed to load language file from config: " + language, (Throwable)e);
                    }
                    break block14;
                }
            }
            if (this.logger != null) {
                this.logger.warn("Language file not found in config: {}.json. Using defaults from resources.", (Object)language);
            }
        }
        this.currentLanguage = language;
    }

    private void loadFromResource(String langCode, Gson gson, Type type) {
        block16: {
            try (InputStream in = this.getClass().getResourceAsStream("/lang/" + langCode + ".json");){
                if (in == null) {
                    if (this.logger != null) {
                        this.logger.warn("Resource language file missing: /lang/{}.json", (Object)langCode);
                    }
                    return;
                }
                try (InputStreamReader reader = new InputStreamReader(in, StandardCharsets.UTF_8);){
                    Map rawMap = (Map)gson.fromJson((Reader)reader, type);
                    this.flattenMap("", rawMap);
                }
            }
            catch (Exception e) {
                if (this.logger == null) break block16;
                this.logger.warn("Failed to load resource language file for: {}", (Object)langCode, (Object)e);
            }
        }
    }

    private void flattenMap(String prefix, Map<String, Object> map) {
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            String key = prefix.isEmpty() ? entry.getKey() : prefix + "." + entry.getKey();
            Object value = entry.getValue();
            if (value instanceof String) {
                this.translations.put(key, (String)value);
                continue;
            }
            if (value instanceof Map) {
                Map castedMap = (Map)value;
                this.flattenMap(key, castedMap);
                continue;
            }
            if (value == null) continue;
            this.translations.put(key, value.toString());
        }
    }

    public IComponent translate(String key) {
        String translatedText = this.translations.getOrDefault(key, key);
        translatedText = translatedText.replace("&", "\u00a7");
        if (this.messageParser == null) {
            if (this.logger != null) {
                this.logger.warn("Lang.translate: MessageParser is null for key '{}'. Returning literal text.", (Object)key);
            }
            return this.platformAdapter.createLiteralComponent(translatedText);
        }
        return this.messageParser.parseMessage(translatedText, null);
    }

    private void ensureDefaultLangFiles() throws IOException {
        if (!Files.exists(this.langFolder, new LinkOption[0])) {
            Files.createDirectories(this.langFolder, new FileAttribute[0]);
        }
        List<String> availableLanguages = List.of("en", "cs", "ru");
        for (String langCode : availableLanguages) {
            Path langFile = this.langFolder.resolve(langCode + ".json");
            if (Files.exists(langFile, new LinkOption[0])) continue;
            if (this.logger != null) {
                this.logger.warn("Language file missing: {}.json. Attempting to copy from resources.", (Object)langCode);
            }
            try {
                InputStream in = this.getClass().getResourceAsStream("/lang/" + langCode + ".json");
                try {
                    if (in == null) {
                        if (this.logger == null) continue;
                        this.logger.error("Default language file /lang/{}.json not found in JAR resources.", (Object)langCode);
                        continue;
                    }
                    Files.copy(in, langFile, StandardCopyOption.REPLACE_EXISTING);
                    if (this.logger == null) continue;
                    this.logger.info("Copied default language file for: {}", (Object)langCode);
                }
                finally {
                    if (in == null) continue;
                    in.close();
                }
            }
            catch (Exception e) {
                if (this.logger == null) continue;
                this.logger.warn("Failed to copy default language file for: " + langCode, (Throwable)e);
            }
        }
    }

    public static class TaskScheduler {
        private ScheduledExecutorService executorService;
        private final AtomicReference<MinecraftServer> serverRef = new AtomicReference<Object>(null);
        private final DebugLogger debugLogger;

        public TaskScheduler(DebugLogger debugLogger) {
            this.debugLogger = debugLogger;
        }

        public void initialize(MinecraftServer serverInstance) {
            if (this.executorService == null || this.executorService.isShutdown()) {
                this.executorService = Executors.newScheduledThreadPool(1);
                this.debugLogger.debugLog("TaskScheduler: Executor service created.");
            }
            this.serverRef.set(serverInstance);
            if (serverInstance != null) {
                this.debugLogger.debugLog("TaskScheduler: Initialized with server instance.");
            } else {
                this.debugLogger.debugLog("TaskScheduler: Initialized with null server instance (server might not be ready).");
            }
        }

        public void scheduleAtFixedRate(Runnable task, long initialDelay, long period, TimeUnit unit) {
            if (this.executorService == null || this.executorService.isShutdown()) {
                this.debugLogger.debugLog("TaskScheduler: Cannot schedule task, executor service is not running.");
                return;
            }
            this.executorService.scheduleAtFixedRate(() -> this.syncExecute(task), initialDelay, period, unit);
        }

        public ScheduledFuture<?> schedule(Runnable task, long delay, TimeUnit unit) {
            if (this.executorService == null || this.executorService.isShutdown()) {
                this.debugLogger.debugLog("TaskScheduler: Cannot schedule task, executor service is not running.");
                return null;
            }
            return this.executorService.schedule(() -> this.syncExecute(task), delay, unit);
        }

        private void syncExecute(Runnable task) {
            MinecraftServer currentServer = this.serverRef.get();
            if (currentServer != null && !currentServer.isStopped()) {
                currentServer.execute(task);
            } else if (currentServer != null) {
                this.debugLogger.debugLog("TaskScheduler: Server instance is stopped, unable to execute task synchronously.");
            }
        }

        public void onServerStopping() {
            if (this.executorService == null) {
                this.debugLogger.debugLog("TaskScheduler: Executor service was null, nothing to shut down.");
                return;
            }
            if (this.executorService.isShutdown()) {
                this.debugLogger.debugLog("TaskScheduler: Executor service already shut down.");
                return;
            }
            this.debugLogger.debugLog("TaskScheduler: Server is stopping, shutting down scheduler...");
            this.executorService.shutdown();
            try {
                if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.executorService.shutdownNow();
                    this.debugLogger.debugLog("TaskScheduler: Executor service forcefully shut down.");
                } else {
                    this.debugLogger.debugLog("TaskScheduler: Executor service shut down gracefully.");
                }
            }
            catch (InterruptedException ex) {
                this.executorService.shutdownNow();
                this.debugLogger.debugLog("TaskScheduler: Executor service shutdown interrupted.");
                Thread.currentThread().interrupt();
            }
        }

        public boolean isServerAvailable() {
            MinecraftServer currentServer = this.serverRef.get();
            return currentServer != null && !currentServer.isStopped();
        }
    }
}

