/*
 * Decompiled with CFR 0.152.
 */
package com.chestlocklite.managers;

import com.chestlocklite.ChestLockLitePlugin;
import com.chestlocklite.utils.PasswordHasher;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.logging.Level;
import org.bukkit.Location;

public class DatabaseManager {
    private final ChestLockLitePlugin plugin;
    private Connection connection;
    private String databasePath;

    public DatabaseManager(ChestLockLitePlugin plugin) {
        this.plugin = plugin;
        this.databasePath = plugin.getDataFolder().getAbsolutePath() + "/" + plugin.getConfigManager().getDatabaseFilename();
    }

    public void initialize() throws SQLException {
        if (!this.plugin.getDataFolder().exists()) {
            this.plugin.getDataFolder().mkdirs();
        }
        String url = "jdbc:sqlite:" + this.databasePath;
        this.connection = DriverManager.getConnection(url);
        this.createTables();
        this.plugin.getLogger().info("Database connected: " + this.databasePath);
    }

    private void createTables() throws SQLException {
        String createLocksTable = "CREATE TABLE IF NOT EXISTS chest_locks (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    world TEXT NOT NULL,\n    x INTEGER NOT NULL,\n    y INTEGER NOT NULL,\n    z INTEGER NOT NULL,\n    owner_uuid TEXT NOT NULL,\n    owner_name TEXT NOT NULL,\n    password TEXT,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    UNIQUE(world, x, y, z)\n)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createLocksTable);
        }
        String createIndex = "CREATE INDEX IF NOT EXISTS idx_location ON chest_locks(world, x, y, z)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createIndex);
        }
        String createTrustedPlayersTable = "CREATE TABLE IF NOT EXISTS trusted_players (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    world TEXT NOT NULL,\n    x INTEGER NOT NULL,\n    y INTEGER NOT NULL,\n    z INTEGER NOT NULL,\n    trusted_uuid TEXT NOT NULL,\n    trusted_name TEXT NOT NULL,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    UNIQUE(world, x, y, z, trusted_uuid)\n)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createTrustedPlayersTable);
        }
        String createTrustedIndex = "CREATE INDEX IF NOT EXISTS idx_trusted_location ON trusted_players(world, x, y, z)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createTrustedIndex);
        }
        if (this.plugin.getConfigManager().isDebug()) {
            this.plugin.getLogger().info("Database tables created/verified successfully");
        }
    }

    public void addLock(Location location, UUID ownerUUID, String ownerName) throws SQLException {
        String sql = "INSERT OR REPLACE INTO chest_locks (world, x, y, z, owner_uuid, owner_name)\nVALUES (?, ?, ?, ?, ?, ?)\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            pstmt.setString(5, ownerUUID.toString());
            pstmt.setString(6, ownerName);
            pstmt.executeUpdate();
        }
        if (this.plugin.getConfigManager().isLogActions()) {
            this.plugin.getLogger().info("Lock added: " + ownerName + " at " + location.getWorld().getName() + " " + location.getBlockX() + "," + location.getBlockY() + "," + location.getBlockZ());
        }
    }

    public void removeLock(Location location) throws SQLException {
        String sql = "DELETE FROM chest_locks WHERE world = ? AND x = ? AND y = ? AND z = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            int rows = pstmt.executeUpdate();
            if (rows > 0 && this.plugin.getConfigManager().isLogActions()) {
                this.plugin.getLogger().info("Lock removed at " + location.getWorld().getName() + " " + location.getBlockX() + "," + location.getBlockY() + "," + location.getBlockZ());
            }
        }
        this.removeAllTrustedPlayers(location);
    }

    public LockData getLock(Location location) throws SQLException {
        String sql = "SELECT owner_uuid, owner_name, password FROM chest_locks\nWHERE world = ? AND x = ? AND y = ? AND z = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            ResultSet rs = pstmt.executeQuery();
            if (rs.next()) {
                LockData lockData = new LockData(UUID.fromString(rs.getString("owner_uuid")), rs.getString("owner_name"), rs.getString("password"));
                return lockData;
            }
        }
        return null;
    }

    public void setPassword(Location location, String password) throws SQLException {
        String sql = "UPDATE chest_locks SET password = ? WHERE world = ? AND x = ? AND y = ? AND z = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            String hashedPassword = PasswordHasher.hashPassword(password);
            pstmt.setString(1, hashedPassword);
            pstmt.setString(2, location.getWorld().getName());
            pstmt.setInt(3, location.getBlockX());
            pstmt.setInt(4, location.getBlockY());
            pstmt.setInt(5, location.getBlockZ());
            pstmt.executeUpdate();
        }
    }

    public void removePassword(Location location) throws SQLException {
        String sql = "UPDATE chest_locks SET password = NULL WHERE world = ? AND x = ? AND y = ? AND z = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            pstmt.executeUpdate();
        }
    }

    public int getChestCount(UUID playerUUID) throws SQLException {
        String sql = "SELECT COUNT(*) FROM chest_locks WHERE owner_uuid = ?";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, playerUUID.toString());
            ResultSet rs = pstmt.executeQuery();
            if (rs.next()) {
                int n = rs.getInt(1);
                return n;
            }
        }
        return 0;
    }

    public int clearAllLocksByPlayer(UUID playerUUID) throws SQLException {
        String sql = "DELETE FROM chest_locks WHERE owner_uuid = ?";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, playerUUID.toString());
            int rows = pstmt.executeUpdate();
            if (rows > 0 && this.plugin.getConfigManager().isLogActions()) {
                this.plugin.getLogger().info("Cleared " + rows + " lock(s) for player UUID: " + String.valueOf(playerUUID));
            }
            int n = rows;
            return n;
        }
    }

    public int clearAllLocksInArea(Location center, int radius) throws SQLException {
        String sql = "DELETE FROM chest_locks\nWHERE world = ?\nAND x BETWEEN ? AND ?\nAND y BETWEEN ? AND ?\nAND z BETWEEN ? AND ?\n";
        int minX = center.getBlockX() - radius;
        int maxX = center.getBlockX() + radius;
        int minY = center.getBlockY() - radius;
        int maxY = center.getBlockY() + radius;
        int minZ = center.getBlockZ() - radius;
        int maxZ = center.getBlockZ() + radius;
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, center.getWorld().getName());
            pstmt.setInt(2, minX);
            pstmt.setInt(3, maxX);
            pstmt.setInt(4, minY);
            pstmt.setInt(5, maxY);
            pstmt.setInt(6, minZ);
            pstmt.setInt(7, maxZ);
            int rows = pstmt.executeUpdate();
            if (rows > 0 && this.plugin.getConfigManager().isLogActions()) {
                this.plugin.getLogger().info("Cleared " + rows + " lock(s) in area around " + center.getWorld().getName() + " " + center.getBlockX() + "," + center.getBlockY() + "," + center.getBlockZ() + " (radius: " + radius + ")");
            }
            int n = rows;
            return n;
        }
    }

    public String getOwnerName(UUID playerUUID) throws SQLException {
        String sql = "SELECT DISTINCT owner_name FROM chest_locks WHERE owner_uuid = ? LIMIT 1";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, playerUUID.toString());
            ResultSet rs = pstmt.executeQuery();
            if (rs.next()) {
                String string = rs.getString("owner_name");
                return string;
            }
        }
        return null;
    }

    public void addTrustedPlayer(Location location, UUID trustedUUID, String trustedName) throws SQLException {
        String sql = "INSERT OR REPLACE INTO trusted_players (world, x, y, z, trusted_uuid, trusted_name)\nVALUES (?, ?, ?, ?, ?, ?)\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            pstmt.setString(5, trustedUUID.toString());
            pstmt.setString(6, trustedName);
            pstmt.executeUpdate();
        }
        if (this.plugin.getConfigManager().isLogActions()) {
            this.plugin.getLogger().info("Added trusted player: " + trustedName + " for chest at " + location.getWorld().getName() + " " + location.getBlockX() + "," + location.getBlockY() + "," + location.getBlockZ());
        }
    }

    public void removeTrustedPlayer(Location location, UUID trustedUUID) throws SQLException {
        String sql = "DELETE FROM trusted_players\nWHERE world = ? AND x = ? AND y = ? AND z = ? AND trusted_uuid = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            pstmt.setString(5, trustedUUID.toString());
            pstmt.executeUpdate();
        }
    }

    public void removeAllTrustedPlayers(Location location) throws SQLException {
        String sql = "DELETE FROM trusted_players\nWHERE world = ? AND x = ? AND y = ? AND z = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            pstmt.executeUpdate();
        }
    }

    public boolean isTrustedPlayer(Location location, UUID playerUUID) throws SQLException {
        String sql = "SELECT COUNT(*) FROM trusted_players\nWHERE world = ? AND x = ? AND y = ? AND z = ? AND trusted_uuid = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            pstmt.setString(5, playerUUID.toString());
            ResultSet rs = pstmt.executeQuery();
            if (rs.next()) {
                boolean bl = rs.getInt(1) > 0;
                return bl;
            }
        }
        return false;
    }

    public List<TrustedPlayerData> getTrustedPlayers(Location location) throws SQLException {
        ArrayList<TrustedPlayerData> trustedPlayers = new ArrayList<TrustedPlayerData>();
        String sql = "SELECT trusted_uuid, trusted_name FROM trusted_players\nWHERE world = ? AND x = ? AND y = ? AND z = ?\n";
        try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
            pstmt.setString(1, location.getWorld().getName());
            pstmt.setInt(2, location.getBlockX());
            pstmt.setInt(3, location.getBlockY());
            pstmt.setInt(4, location.getBlockZ());
            ResultSet rs = pstmt.executeQuery();
            while (rs.next()) {
                trustedPlayers.add(new TrustedPlayerData(UUID.fromString(rs.getString("trusted_uuid")), rs.getString("trusted_name")));
            }
        }
        return trustedPlayers;
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
                this.plugin.getLogger().info("Database connection closed");
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Error closing database connection", e);
        }
    }

    public Connection getConnection() {
        return this.connection;
    }

    public static class LockData {
        private final UUID ownerUUID;
        private final String ownerName;
        private final String password;

        public LockData(UUID ownerUUID, String ownerName, String password) {
            this.ownerUUID = ownerUUID;
            this.ownerName = ownerName;
            this.password = password;
        }

        public UUID getOwnerUUID() {
            return this.ownerUUID;
        }

        public String getOwnerName() {
            return this.ownerName;
        }

        public String getPassword() {
            return this.password;
        }

        public boolean hasPassword() {
            return this.password != null && !this.password.isEmpty();
        }
    }

    public static class TrustedPlayerData {
        private final UUID uuid;
        private final String name;

        public TrustedPlayerData(UUID uuid, String name) {
            this.uuid = uuid;
            this.name = name;
        }

        public UUID getUUID() {
            return this.uuid;
        }

        public String getName() {
            return this.name;
        }
    }
}

