package com.swacky.ohmega.api;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.swacky.ohmega.api.event.AccessoryEquipCallback;
import com.swacky.ohmega.api.event.AccessoryOverrideTypesCallback;
import com.swacky.ohmega.common.OhmegaCommon;
import com.swacky.ohmega.common.accessorytype.AccessoryType;
import com.swacky.ohmega.common.accessorytype.AccessoryTypeManager;
import com.swacky.ohmega.common.Ohmega;
import com.swacky.ohmega.common.init.OhmegaBinds;
import com.swacky.ohmega.common.init.OhmegaDataAttachments;
import com.swacky.ohmega.common.init.OhmegaDataComponents;
import com.swacky.ohmega.common.init.OhmegaTags;
import com.swacky.ohmega.common.datacomponent.AccessoryItemDataComponent;
import com.swacky.ohmega.common.inv.AccessoryContainer;
import com.swacky.ohmega.config.OhmegaConfig;
import com.swacky.ohmega.common.dataattachment.AccessoryInvDataAttachment;
import com.swacky.ohmega.event.OhmegaHooks;
import com.swacky.ohmega.network.S2C.SyncAccessorySlotsPacket;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_124;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1322;
import net.minecraft.class_1324;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_304;
import net.minecraft.class_3222;
import net.minecraft.class_5250;
import net.minecraft.class_7417;
import net.minecraft.class_8828;
import net.minecraft.class_9285;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

@SuppressWarnings("unused")
public class AccessoryHelper {
    /**
     * This is an internal function; it is in the {@code api} package as you can use it, however it is discouraged
     * @param stack an {@link class_1799} instance to retrieve the {@link AccessoryItemDataComponent} from
     * @return the {@link AccessoryItemDataComponent} bound to the {@link class_1799}
     */
    public static AccessoryItemDataComponent _getInternalData(class_1799 stack) {
        if (stack.method_57826(OhmegaDataComponents.ACCESSORY_ITEM)) {
            return stack.method_58694(OhmegaDataComponents.ACCESSORY_ITEM);
        }
        return stack.method_57379(OhmegaDataComponents.ACCESSORY_ITEM, new AccessoryItemDataComponent(-1, false, ModifierHolder.EMPTY));
    }

    /**
     * @param player a {@link class_1657} instance to retrieve data from
     * @return the {@link AccessoryContainer} bound to the {@link class_1657} if present, else creates one and stores for later use
     */
    public static AccessoryContainer getContainer(class_1657 player) {
        return new AccessoryContainer(player, player.getAttachedOrCreate(OhmegaDataAttachments.ACCESSORY_HANDLER));
    }

    /**
     * Makes any (including vanilla and other mods') {@link class_1792}(s) into an accessory by binding an {@link IAccessory} instance to it.
     * @param item the registered {@link class_1792} target
     * @param binding an {@link IAccessory} instance to bind the target to
     * @return true if it can be bound, false if it is already bound
     */
    public static boolean bindAccessory(class_1792 item, IAccessory binding) {
        return OhmegaCommon.bindAccessory(item, binding);
    }

    /**
     * Checks if an {@link class_1792} is in any way an accessory, through either implementing {@link IAccessory} or binding with {@link #bindAccessory(class_1792, IAccessory)}
     * @param item the {@link class_1792} to test
     * @return  true if an accessory
     */
    public static boolean isItemAccessoryBound(class_1792 item) {
        return OhmegaCommon.isItemAccessoryBound(item);
    }

    /**
     * Retrieves the {@link IAccessory} instance bound to the passed {@link class_1792} through implementing {@link IAccessory} or binding with {@link #bindAccessory(class_1792, IAccessory)}
     * @param item the {@link class_1792} to retrieve the bound {@link IAccessory} instance
     * @return if found, the {@link class_1792}'s bound {@link IAccessory} instance, else null
     */
    public static IAccessory getBoundAccessory(class_1792 item) {
        return OhmegaCommon.getBoundAccessory(item);
    }

    /**
     * Finds the slot that a given {@link IAccessory} is found
     * @param player the {@link class_1657} to check for
     * @param acc the {@link IAccessory} to find the slot of
     * @return the slot that the passed {@link IAccessory} was found. If not found, returns -1
     */
    public static int getSlotFor(class_1657 player, IAccessory acc) {
        ArrayList<class_1799> list = getStacks(player);
        for (int i = 0; i < list.size(); i++) {
            if (acc == getBoundAccessory(list.get(i).method_7909())) {
                return i;
            }
        }
        return -1;
    }

    /**
     * A shortcut method to {@link #getSlotFor(class_1657, IAccessory)}
     * @param player the {@link class_1657} to check for
     * @param item the accessory to find the slot of
     * @return the slot that the passed accessory was found. If not found, returns -1
     */
    public static int getSlotFor(class_1657 player, class_1792 item) {
        IAccessory acc = getBoundAccessory(item);
        if (acc != null) {
            return getSlotFor(player, acc);
        }
        return -1;
    }

    /**
     * Sets the slot of an accessory {@link class_1799}
     * <p>
     * Does not place the {@link class_1799} into an accessory slot, but instead adds slot index data to the {@link class_1799}
     * @param stack an {@link class_1799} of an accessory
     * @param slot the index of the slot to set in
     */
    public static void setSlot(class_1799 stack, int slot) {
        AccessoryItemDataComponent data = _getInternalData(stack);
        if (data != null) {
            data.setSlot(slot);
        }
    }

    /**
     * Gets the item's slot in the player's inventory from an {@link class_1799}'s tag
     * @param stack the {@link class_1799} to test against
     * @return the slot of the item if the tag is present, -1 otherwise
     * -1 is returned if not present because otherwise it would return 0, the first slot, which messes things up. Using -1 makes it an outlier
     */
    public static int getSlot(class_1799 stack) {
        AccessoryItemDataComponent data = _getInternalData(stack);
        if (data != null) {
            return data.getSlot();
        }
        return -1;
    }

    /**
     * Checks if the player has an accessory
     * @param player the {@link class_1657} to check for
     * @param acc the {@link IAccessory} to check if the player has it
     * @return true if the {@link class_1657} has the passed {@link IAccessory} equipped, otherwise false
     */
    public static boolean hasAccessory(class_1657 player, IAccessory acc) {
        return getSlotFor(player, acc) != -1;
    }

    /**
     * A shortcut method to {@link #hasAccessory(class_1657, IAccessory)}
     * @param player the {@link class_1657} to check for
     * @param item the accessory to check if the player has it
     * @return true if the {@link class_1657} has the passed accessory equipped, otherwise false
     */
    public static boolean hasAccessory(class_1657 player, class_1792 item) {
        IAccessory acc = getBoundAccessory(item);
        if (acc != null) {
            return hasAccessory(player, acc);
        }
        return false;
    }

    /**
     * Returns the {@link class_1799} corresponding to the slot index of the accessory inventory
     * @param player the {@link class_1657} to get the stacks from
     * @param slot the index of slot, 5 would be the special slot as an example
     * @return the {@link class_1799} in the slot provided
     */
    public static class_1799 getStackInSlot(class_1657 player, int slot) {
        return getContainer(player).getStackInSlot(slot);
    }

    /**
     * An inner class allowing functions to be called with a {@link class_1657} and an {@link class_1799} argument
     */
    @FunctionalInterface
    public interface Inner {
        void apply(class_1657 player, class_1799 stack);
    }

    /**
     * Checks if a player has an accessory and allows for the calling of a function upon it
     * @param player the {@link class_1657} to check for
     * @param acc checks if the player has this {@link IAccessory} instance
     * @param holder a holder class for a functional interface, allows users to run methods if the accessory is present
     * @return true if the {@link class_1657} has the passed accessory, false otherwise
     */
    public static boolean runIfPresent(class_1657 player, IAccessory acc, Inner holder) {
        int slot = getSlotFor(player, acc);
        if (slot != -1) {
            holder.apply(player, getStackInSlot(player, slot));
            return true;
        }
        return false;
    }

    /**
     * Checks if a player has an accessory, if true, the accessory will update
     * @param player the player to check for
     * @param acc the {@link IAccessory} to check if the player has it
     * @return true if the player has the passed accessory, false otherwise
     */
    public static boolean updateIfPresent(class_1657 player, IAccessory acc) {
        return runIfPresent(player, acc, acc::update);
    }

    /**
     * Gets all the {@link AccessoryType}s that this accessory is marked as.
     * Consider instead using {@link #getType(class_1792)}
     * @param item the {@link class_1792} of the accessory
     * @return a non-duplicate {@link ImmutableList} of all {@link AccessoryType}s bound to the given accessory
     */
    @SuppressWarnings("deprecation")
    public static ImmutableList<AccessoryType> getTypes(class_1792 item) {
        ImmutableSet.Builder<AccessoryType> builder = new ImmutableSet.Builder<>();

        if (OhmegaConfig.CONFIG_SERVER.noAccessoryTypes.get()) {
            builder.add(AccessoryType.GENERIC.get());
        }

        for (OhmegaTags.TagHolder holder : OhmegaTags.getTags()) {
            if (item.method_40131().method_40220(holder.getTag())) {
                builder.add(holder.getType());
            }
        }

        ImmutableSet<AccessoryType> set = builder.build();
        if (set.isEmpty()) {
            return ImmutableList.of(AccessoryType.NORMAL.get());
        }
        return ImmutableList.copyOf(set);
    }

    /**
     * A shortcut method to {@link #getTypes(class_1792)}.
     * Consider instead using {@link #getType(class_1799)}
     * @param stack the {@link class_1799} instance of the given accessory
     * @return a non-duplicate {@link ImmutableList} of all {@link AccessoryType}s bound to the given accessory
     */
    public static ImmutableList<AccessoryType> getTypes(class_1799 stack) {
        return getTypes(stack.method_7909());
    }

    /**
     * Retrieves the accessory's effective {@link AccessoryType} (lowest priority index), or if overridden by the {@link AccessoryOverrideTypesCallback}, then that one
     * @param item the {@link class_1792} of the accessory
     * @return the {@link AccessoryType} of lowest priority index bound to the given accessory
     */
    @SuppressWarnings("deprecation")
    public static AccessoryType getType(class_1792 item) {
        if (OhmegaConfig.CONFIG_SERVER.noAccessoryTypes.get()) {
            return AccessoryType.GENERIC.get();
        }

        {
            ImmutableMap<class_1792, AccessoryType> map = Ohmega.getAccessoryTypeOverrides();
            if (map.containsKey(item)) {
                return map.get(item);
            }
        }

        AccessoryType type = AccessoryType.NORMAL.get();

        for (OhmegaTags.TagHolder holder : OhmegaTags.getTags()) {
            if (item.method_40131().method_40220(holder.getTag())) {
                AccessoryType holderType = holder.getType();
                if (holderType.getPriority() < type.getPriority()) {
                    type = holderType;
                }
            }
        }

        return type;
    }

    /**
     * A shortcut method to {@link #getTypes(class_1792)}.
     * @return the {@link AccessoryType} of lowest priority index bound to the given accessory
     */
    public static AccessoryType getType(class_1799 stack) {
        return getType(stack.method_7909());
    }

    /**
     * Turns config value for slot types into usable data
     * @return an {@link ImmutableList} of {@link AccessoryType}s from the config value
     */
    public static ImmutableList<AccessoryType> getSlotTypes() {
        if (OhmegaConfig.CONFIG_SERVER.noAccessoryTypes.get()) {
            int size = OhmegaConfig.CONFIG_SERVER.slotTypes.get().size();
            ImmutableList.Builder<AccessoryType> builder = ImmutableList.builderWithExpectedSize(size);
            for (int i = 0; i < size; i++) {
                builder.add(AccessoryType.GENERIC.get());
            }
            return builder.build();
        }

        ImmutableList.Builder<AccessoryType> builder = new ImmutableList.Builder<>();
        for (String str : OhmegaConfig.CONFIG_SERVER.slotTypes.get()) {
            builder.add(AccessoryTypeManager.getInstance().get(class_2960.method_60654(str)));
        }
        return builder.build();
    }

    /**
     * @return an {@link ImmutableList} of {@link String}s of accessory slots' types
     */
    public static ImmutableList<String> getSlotTypesStr() {
        if (OhmegaConfig.CONFIG_SERVER.noAccessoryTypes.get()) {
            int size = OhmegaConfig.CONFIG_SERVER.slotTypes.get().size();
            ImmutableList.Builder<String> builder = ImmutableList.builderWithExpectedSize(size);
            for (int i = 0; i < size; i++) {
                builder.add(AccessoryType.GENERIC.get().getId().toString());
            }
            return builder.build();
        }

        return ImmutableList.copyOf(OhmegaConfig.CONFIG_SERVER.slotTypes.get());
    }

    /**
     * Turns config value for keybound slot types into usable data, filtering out duplicates
     * @return an {@link ImmutableList} of {@link AccessoryType}s from the config value
     */
    public static ImmutableList<AccessoryType> getKeyboundSlotTypes() {
        if (OhmegaConfig.CONFIG_SERVER.noAccessoryTypes.get()) {
            return ImmutableList.of(AccessoryType.GENERIC.get());
        }

        ImmutableList.Builder<AccessoryType> builder = new ImmutableList.Builder<>();
        for (String str : OhmegaConfig.CONFIG_SERVER.keyboundSlotTypes.get()) {
            builder.add(AccessoryTypeManager.getInstance().get(class_2960.method_60654(str)));
        }
        return ImmutableSet.copyOf(builder.build()).asList();
    }

    /**
     * @return an {@link ImmutableList} of {@link String}s of key-bound accessory slots' types
     */
    @SuppressWarnings("unchecked")
    public static ImmutableList<String> getKeyboundSlotTypesStr() {
        if (OhmegaConfig.CONFIG_SERVER.noAccessoryTypes.get()) {
            return ImmutableList.of(AccessoryType.GENERIC.get().getId().toString());
        }

        return (ImmutableList<String>) ImmutableSet.copyOf(OhmegaConfig.CONFIG_SERVER.keyboundSlotTypes.get()).asList();
    }

    /**
     * A utility method to get a description for a key-bind activated {@link IAccessory}.
     * @param bindDescription what will be displayed for the bind to do / activate. Use "&lt;BIND&gt;" to show the bind letter
     * @param stack a {@link class_1799} instance of the given accessory
     * @param other the "other" tooltip, for displaying when {@link IAccessory} is not in an accessory slot (such sas in the normal inventory)
     * @return if the slot is a normal category slot: The "other" {@link class_2561}
     * <p>
     * An example: {@code "Press B to activate invisibility"} is produced by {@code "Press <BIND> to activate invisibility"} when in a slot with key-binding of key B
     */
    public static class_5250 getBindTooltip(class_7417 bindDescription, class_1799 stack, class_7417 other) {
        ImmutableList<AccessoryType> slotTypes = getSlotTypes();

        int slot = getSlot(stack);
        AccessoryType type;
        if (slot < 0 || slot >= slotTypes.size()) {
            type = null;
        } else {
            type = slotTypes.get(slot);
        }

        // Starts at -1 to align properly
        int typeIndex = -1;
        if (type != null) {
            for (int i = 0; i < slot + 1; i++) {
                if (slotTypes.get(i) == type) {
                    typeIndex++;
                }
            }
        }

        boolean flag = false;
        IAccessory acc = getBoundAccessory(stack.method_7909());
        if (acc != null) {
            for (AccessoryType type0 : getKeyboundSlotTypes()) {
                if (slotTypes.contains(type0)) {
                    flag = true;
                    break;
                }
            }
        }

        class_304 mapping = OhmegaBinds.Generated.getMapping(type, typeIndex);

        if (slot < 0 || !flag || mapping == null) {
            return class_5250.method_43477(other).method_27692(class_124.field_1080);
        } else {
            return class_5250.method_43477(new class_8828.class_2585(class_5250.method_43477(bindDescription).getString().replace("<BIND>", mapping.method_16007().getString()))).method_27692(class_124.field_1080);
        }
    }

    /**
     * A utility method to create a {@link class_5250} of the translated {@link AccessoryType} bound to an accessory
     * @param item the {@link class_1792} of an accessory to get the type from
     * @return a {@link class_5250} instance of "Accessory type: TYPE"
     */
    @Nullable
    public static class_5250 getTypeTooltip(class_1792 item) {
        AccessoryType type = getType(item);
        if (type.displayHoverText()) {
            return class_2561.method_43469("accessory_type", type.getTranslation().getString()).method_27692(class_124.field_1063);
        }
        return null;
    }

    /**
     * Utility function to add {@link class_1322}s to a {@link class_1657} from an {@link ModifierHolder.Builder}
     * @param player add/remove modifiers to/from this {@link class_1657}
     * @param map a {@link Multimap} which contains the modifiers to add or remove
     * @param add if true, will add the {@link class_1322}(s) to the {@link class_1657}, otherwise existing ones will be removed
     */
    public static void changeModifiers(class_1657 player, class_9285 map, boolean add) {
        for (class_9285.class_9287 entry : map.comp_2393()) {
            class_1324 attribute0 = player.method_5996(entry.comp_2395());
            if (attribute0 != null) {
                if (add) {
                    if (!attribute0.method_6196(entry.comp_2396().comp_2447())) {
                        attribute0.method_26835(entry.comp_2396());
                    }
                } else {
                    attribute0.method_6202(entry.comp_2396());
                }
            }
        }
    }

    /**
     * Retrieves the {@link ModifierHolder} from an {@link class_1799} instance
     * @param stack an {@link class_1799} of an accessory
     * @return the bound {@link ModifierHolder}
     */
    public static ModifierHolder getModifiers(class_1799 stack) {
        AccessoryItemDataComponent data = _getInternalData(stack);
        if (data != null) {
            return data.getModifiers();
        }
        return ModifierHolder.EMPTY;
    }

    /**
     * Removes both passive and active only {@link class_1322}s
     * @param player to remove modifiers from
     * @param holder contains the modifiers to be removed
     */
    public static void removeAllModifiers(class_1657 player, ModifierHolder holder) {
        changeModifiers(player, holder.getPassive(), false);
        changeModifiers(player, holder.getActive(), false);
    }

    /**
     * Adds a tag to show the active state of an {@link IAccessory} item
     * @param player used to add/remove active only attribute modifiers to
     * @param stack the {@link IAccessory} to add the tag to
     * @param value the value of the active state
     * This is handled internally but for whatever reason you want to change the value of this, you can
     */
    public static void setActive(class_1657 player, class_1799 stack, boolean value) {
        if (isItemAccessoryBound(stack.method_7909())) {
            AccessoryItemDataComponent data = _getInternalData(stack);
            if (data != null) {
                data.setActive(value);
            }
        }
        changeModifiers(player, getModifiers(stack).getActive(), value);
    }

    /**
     * Checks the active state of an {@link IAccessory} item by tag
     * @param stack the {@link class_1799} to check the active state of
     * @return true if active, false if inactive or not an {{@link IAccessory} item
     */
    public static boolean isActive(class_1799 stack) {
        if (isItemAccessoryBound(stack.method_7909())) {
            AccessoryItemDataComponent data = _getInternalData(stack);
            if (data != null) {
                return data.isActive();
            }
        }
        return false;
    }

    /**
     * Activates an {@link IAccessory} item in a slot
     * @param player used to add/remove active only attribute modifiers to
     * @param stack the {@link class_1799} to activate an accessory on
     */
    public static void activate(class_1657 player, class_1799 stack) {
        setActive(player, stack, true);
    }

    /**
     * Deactivates an {@link IAccessory} item in a slot
     * @param player used to add/remove active only attribute modifiers to
     * @param stack the {@link class_1799} to deactivate an accessory on
     */
    public static void deactivate(class_1657 player, class_1799 stack) {
        setActive(player, stack, false);
    }

    /**
     * Toggles an {@link IAccessory}'s active state
     * @param player used to add/remove active only attribute modifiers to
     * @param stack the {@link class_1799} to toggle an accessory's active state
     */
    public static void toggle(class_1657 player, class_1799 stack) {
        if (isActive(stack)) {
            deactivate(player, stack);
        } else {
            activate(player, stack);
        }
    }

    /**
     * Returns the first open accessory slot for an {@link AccessoryType} or {@link AccessoryType}s
     * @param player the {@link class_1657} to test their slots against
     * @param type the {@link AccessoryType} to check for the slots
     * @return the first open slot of the type, else if none is found then -1
     */
    public static int getFirstOpenSlot(class_1657 player, AccessoryType type) {
        AccessoryContainer a = getContainer(player);
        ImmutableList<AccessoryType> slotTypes = getSlotTypes();
        for (int i = 0; i < a.getSlots(); i++) {
            if (slotTypes.get(i) == type && a.getStackInSlot(i).method_7960()) {
                return i;
            }
        }
        return -1;
    }

    /**
     * A utility method to allow for the equipping of accessories by a right-click
     * <p>
     * This is automatically called by Ohmega for accessory items, you can disable this through the use of {@link AccessoryEquipCallback}, checking for {@link AccessoryEquipCallback.Context#RIGHT_CLICK_HELD_ITEM}
     * @param player the {@link class_1657} to put the {@link IAccessory} on
     * @param hand the {@link class_1268} to get the accessory held
     * @return an interaction result of success if the item is equipped, else a pass
     */
    public static class_1269 tryEquip(class_1657 player, class_1268 hand) {
        class_1799 stack = player.method_5998(hand);
        class_1792 item = stack.method_7909();
        IAccessory acc = getBoundAccessory(item);
        if (acc != null) {
            int slot = getFirstOpenSlot(player, getType(item));
            if (slot >= 0) {
                class_1799 stack0 = stack.method_46651(1);
                if (getContainer(player).setStackInSlot(slot, stack0)) {
                    changeModifiers(player, getModifiers(stack).getPassive(), true);

                    stack.method_7934(1);
                    setSlot(stack, slot);

                    if (!OhmegaHooks.accessoryEquipEvent(player, stack0, AccessoryEquipCallback.Context.RIGHT_CLICK_HELD_ITEM)) {
                        acc.onEquip(player, stack0);
                    }
                    if (acc.getEquipSound() != null) {
                        player.method_5783(acc.getEquipSound().comp_349(), 1, 1);
                    }
                    return class_1269.field_5812;
                }
            }
        }
        return class_1269.field_5811;
    }

    /**
     * Returns all the stacks from the player's accessory inventory
     * @param player the {@link class_1657} to get the stacks from
     * @return an {@link ArrayList} of all accessory slot stacks
     */
    public static ArrayList<class_1799> getStacks(class_1657 player) {
        ArrayList<class_1799> stacks = new ArrayList<>();
        AccessoryContainer a = getContainer(player);
        for (int i = 0; i < a.getSlots(); i++) {
            stacks.add(a.getStackInSlot(i));
        }
        return stacks;
    }

    /**
     * Returns all the accessories from the player's accessory inventory
     * @param player the {@link class_1657} to get the stacks from
     * @return an {@link ArrayList} of all accessory slot accessories
     */
    public static ArrayList<IAccessory> getAccessories(class_1657 player) {
        ArrayList<IAccessory> accessories = new ArrayList<>();
        for (class_1799 stack : getStacks(player)) {
            IAccessory acc = getBoundAccessory(stack.method_7909());
            if (acc != null) {
                accessories.add(acc);
            }
        }
        return accessories;
    }

    /**
     * Checks if two {@link IAccessory} instances are compatible with each other
     * @param acc first {@link IAccessory} instance
     * @param acc0 second {@link IAccessory} instance
     * @return true if both {@link IAccessory} instances are compatible with one another
     */
    public static boolean compatibleWith(IAccessory acc, IAccessory acc0) {
        return acc.checkCompatibility(acc0) && acc0.checkCompatibility(acc);
    }

    /**
     * A shortcut method to {@link #compatibleWith(IAccessory, IAccessory)} with one {@link class_1792} and one {@link IAccessory} parameter
     * @param item first {@link class_1792} accessory instance
     * @param acc second {@link IAccessory} instance
     * @return true if both accessories are compatible with one another
     */
    public static boolean compatibleWith(class_1792 item, IAccessory acc) {
        IAccessory acc0 = getBoundAccessory(item);
        if (acc0 != null) {
            return compatibleWith(acc, acc0);
        }
        return false;
    }

    /**
     * A shortcut method to {@link #compatibleWith(IAccessory, IAccessory)} with two {@link class_1792} parameters
     * @param item first {@link class_1792} accessory instance
     * @param item0 second {@link class_1792} accessory instance
     * @return true if both accessories are compatible with one another
     */
    public static boolean compatibleWith(class_1792 item, class_1792 item0) {
        IAccessory acc = getBoundAccessory(item);
        IAccessory acc0 = getBoundAccessory(item0);
        if (acc != null && acc0 != null) {
            return compatibleWith(acc, acc0);
        }
        return false;
    }

    /**
     * Checks compatibility of the {@link IAccessory} provided with all other equipped accessories
     * @param player the {@link class_1657} to get the accessory inventory of
     * @param acc the {@link IAccessory} to test against
     * @return true if compatible, false if not
     */
    public static boolean compatibleWith(class_1657 player, IAccessory acc) {
        boolean[] out = {true};
        for (IAccessory acc0 : getAccessories(player)) {
            if (!compatibleWith(acc, acc0)) {
                out[0] = false;
            }
        }
        return out[0];
    }

    /**
     * A shortcut method to {@link #compatibleWith(class_1657, IAccessory)}
     * @param player the {@link class_1657} to get the accessory inventory of
     * @param item the accessory to test against
     * @return true if compatible, false if not
     */
    public static boolean compatibleWith(class_1657 player, class_1792 item) {
        IAccessory acc = getBoundAccessory(item);
        if (acc != null) {
            return compatibleWith(player, acc);
        }
        return false;
    }

    /**
     * Forcefully notifies the game that a slot has been changed
     * <p>
     * You may need to use this when an {@link class_1799}'s data has been changed, but {@link AccessoryInvDataAttachment#setStackInSlot(int, class_1799)} (or similar) has not been called
     * @param player {@link class_1657} to update the {@link AccessoryInvDataAttachment} of
     * @param slot the index of which to notify changes of
     */
    public static void setSlotChanged(class_1657 player, int slot) {
        getContainer(player).onContentsChanged(slot);
    }

    /**
     * Synchronises slots from server data with provided clients
     * @param player a {@link class_3222} to retrieve and send data from
     * @param slots the indexes of the slots to sync
     * @param stacks {@link class_1799}s from the server, data used to sync
     * @param receivers {@link class_1657}(s)/client(s) to have server data sent to
     */
    public static void syncSlots(class_3222 player, int[] slots, List<class_1799> stacks, Collection<class_3222> receivers) {
        SyncAccessorySlotsPacket packet = new SyncAccessorySlotsPacket(player.method_5628(), slots, stacks);
        for (class_3222 receiver : receivers) {
            ServerPlayNetworking.send(receiver, packet);
        }
    }

    /**
     * Synchronises all slots from server data with provided clients
     * @param player a {@link class_3222} to retrieve and send data from
     * @param receivers {@link class_1657}(s)/client(s) to have server data sent to
     */
    public static void syncAllSlots(class_3222 player, Collection<class_3222> receivers) {
        ArrayList<class_1799> stacks = getStacks(player);
        int size = stacks.size();
        int[] slots = new int[size];
        for (int i = 0; i < size; i++) {
            slots[i] = i;
        }
        syncSlots(player, slots, stacks, receivers);
    }
}