package com.swacky.ohmega.common.dataattachment;

import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Booleans;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.swacky.ohmega.api.AccessoryHelper;
import com.swacky.ohmega.api.ModifierHolder;
import com.swacky.ohmega.common.accessorytype.AccessoryType;
import com.swacky.ohmega.api.IAccessory;
import com.swacky.ohmega.config.OhmegaConfig;
import com.swacky.ohmega.event.OhmegaHooks;
import org.apache.commons.lang3.ArrayUtils;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import net.minecraft.class_1262;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1928;
import net.minecraft.class_2371;
import net.minecraft.class_3222;

public class AccessoryInvDataAttachment {
    public static final Codec<AccessoryInvDataAttachment> CODEC = RecordCodecBuilder.create(builder -> builder.group(
            Codec.list(class_1799.field_49266).fieldOf("stacks").forGetter(inst -> inst.stacks),
            Codec.list(class_1799.field_49266).fieldOf("previous").forGetter(inst -> inst.previous),
            Codec.list(Codec.BOOL).fieldOf("changed").forGetter(inst -> Booleans.asList(inst.changed))
    ).apply(builder, AccessoryInvDataAttachment::new));

    private class_2371<class_1799> stacks;
    private class_2371<class_1799> previous;
    private boolean[] changed;

    private AccessoryInvDataAttachment(List<class_1799> stacks, List<class_1799> previous, boolean[] changed) {
        this.stacks = class_2371.method_10212(class_1799.field_8037, stacks.toArray(new class_1799[0]));
        this.previous = class_2371.method_10212(class_1799.field_8037, previous.toArray(new class_1799[0]));
        this.changed = changed;
    }

    private AccessoryInvDataAttachment(List<class_1799> stacks, List<class_1799> previous, List<Boolean> changed) {
        this(stacks, previous, Booleans.toArray(changed));
    }

    public AccessoryInvDataAttachment() {
        int size = AccessoryHelper.getSlotTypes().size();
        this.stacks = class_2371.method_10213(size, class_1799.field_8037);
        this.previous = class_2371.method_10213(size, class_1799.field_8037);
        this.changed = new boolean[size];
    }

    public void initialise(class_1657 player) {
        for (int i = 0; i < this.getSlots(); i++) {
            class_1799 stack = this.getStackInSlot(i);

            if (!stack.method_7960()) {
                ModifierHolder modifiers = AccessoryHelper.getModifiers(stack);
                AccessoryHelper.changeModifiers(player, modifiers.getPassive(), true);
                if (AccessoryHelper.isActive(stack)) {
                    AccessoryHelper.changeModifiers(player, modifiers.getActive(), true);
                }
            }
        }
    }

    public void validateSlotIndex(int slot) {
        if (slot < 0 || slot >= this.stacks.size()) {
            throw new RuntimeException("Slot " + slot + " not in valid range - [0," + this.stacks.size() + ")");
        }
    }

    public int getSlots() {
        return this.stacks.size();
    }

    public void onContentsChanged(int index) {
        this.changed[index] = true;
    }

    public class_1799 getStackInSlot(int index) {
        this.validateSlotIndex(index);
        return this.stacks.get(index);
    }

    public void setStackInSlot(int index, @NotNull class_1799 stack) {
        this.validateSlotIndex(index);
        this.previous.set(index, this.getStackInSlot(index));
        this.stacks.set(index, stack);
        this.onContentsChanged(index);
    }

    public class_1799 removeItem(int index, int amount) {
        class_1799 stack = class_1262.method_5430(this.stacks, index, amount);
        if (!stack.method_7960()) {
            this.onContentsChanged(index);
        }

        return stack;
    }

    public void sync(class_1657 player) {
        if (player instanceof class_3222 svr) {
            List<class_3222> receivers = new ArrayList<>(svr.method_51469().method_18766((svr0) -> true));
            receivers.add(svr);

            List<Integer> slots = new ArrayList<>();
            List<class_1799> stacks = new ArrayList<>();
            for (int i = 0; i < getSlots(); i++) {
                class_1799 stack = getStackInSlot(i);
                boolean autoSync;
                IAccessory acc = AccessoryHelper.getBoundAccessory(stack.method_7909());
                if (acc != null) {
                    autoSync = acc.autoSync(svr);
                } else {
                    autoSync = false;
                }
                if (autoSync && !class_1799.method_31577(stack, this.previous.get(i)) || this.changed[i]) {
                    slots.add(i);
                    stacks.add(stack);
                    this.changed[i] = false;
                    this.previous.set(i, stack.method_7972());
                }
            }
            if (!slots.isEmpty()) {
                AccessoryHelper.syncSlots(svr, slots.stream().mapToInt(Integer::intValue).toArray(), stacks, receivers);
            }
        }
    }

    public void invalidate(class_1657 player) {
        boolean flag = switch (OhmegaConfig.CONFIG_SERVER.keepAccessories.get()) { // Inverse
            case ON -> false;
            case OFF -> true;
            case DEFAULT -> player.method_5682() == null || !player.method_5682().method_3767().method_8355(class_1928.field_19389);
        };

        if (flag) {
            for (int i = 0; i < this.getSlots(); i++) {
                class_1799 stack = this.getStackInSlot(i);
                if (!stack.method_7960()) {
                    this.setStackInSlot(i, class_1799.field_8037);
                    IAccessory acc = AccessoryHelper.getBoundAccessory(stack.method_7909());
                    if (acc != null) {
                        if (!OhmegaHooks.accessoryUnequipEvent(player, stack)) {
                            acc.onUnequip(player, stack);
                        }
                        AccessoryHelper.setSlot(stack, -1);
                        player.method_7329(stack, true, false);
                    }
                }
            }
        }
    }

    public void reloadCfg(class_1657 player) {
        int oldSize = this.changed.length;
        int newSize = AccessoryHelper.getSlotTypes().size();

        if (newSize > oldSize) {
            // Grow data
            class_1799[] newStacks = new class_1799[newSize - oldSize];
            Arrays.fill(newStacks, class_1799.field_8037);
            this.stacks = class_2371.method_10212(class_1799.field_8037, ArrayUtils.addAll(this.stacks.toArray(new class_1799[0]), newStacks));
            this.changed = ArrayUtils.addAll(this.changed, new boolean[newSize - oldSize]);
            class_1799[] newPrevious = new class_1799[newSize - oldSize];
            Arrays.fill(newPrevious, class_1799.field_8037);
            this.previous = class_2371.method_10212(class_1799.field_8037, ArrayUtils.addAll(this.previous.toArray(new class_1799[0]), newPrevious));
        } else if (newSize < oldSize) {
            // Drop stacks outside of range
            for (class_1799 stack : Arrays.copyOfRange(this.stacks.toArray(new class_1799[0]), newSize, oldSize)) {
                if (!stack.method_7960()) {
                    IAccessory acc = AccessoryHelper.getBoundAccessory(stack.method_7909());
                    if (acc != null) {
                        if (!OhmegaHooks.accessoryUnequipEvent(player, stack)) {
                            acc.onUnequip(player, stack);
                        }
                        AccessoryHelper.setSlot(stack, -1);
                    }

                    if (!player.method_7270(stack)) {
                        player.method_7328(stack, false);
                    }
                }
            }

            // Shrink data
            this.stacks = class_2371.method_10212(class_1799.field_8037, Arrays.copyOfRange(this.stacks.toArray(new class_1799[0]), 0, newSize));
            this.changed = Arrays.copyOfRange(this.changed, 0, newSize);
            this.previous = class_2371.method_10212(class_1799.field_8037, Arrays.copyOfRange(this.previous.toArray(new class_1799[0]), 0, newSize));
        }

        // Drop invalid stacks (mismatched accessory types and non-accessory items)
        ImmutableList<AccessoryType> slotTypes = AccessoryHelper.getSlotTypes();
        for (int i = 0; i < stacks.size(); i++) {
            class_1799 stack = stacks.get(i);
            if (!stack.method_7960()) {
                class_1792 item = stack.method_7909();
                IAccessory acc = AccessoryHelper.getBoundAccessory(item);

                if (slotTypes.get(i) != AccessoryHelper.getType(item)) {
                    if (acc != null) {
                        if (!OhmegaHooks.accessoryUnequipEvent(player, stack)) {
                            acc.onUnequip(player, stack);
                        }
                        AccessoryHelper.setSlot(stack, -1);
                    }

                    if (!player.method_7270(stack)) {
                        player.method_7328(stack, false);
                    }
                }
            }
        }
    }
}
