/*
 * Decompiled with CFR 0.152.
 */
package dev.deftu.filestream.download;

import dev.deftu.filestream.api.Downloader;
import dev.deftu.filestream.api.Rewriter;
import dev.deftu.filestream.api.Store;
import dev.deftu.filestream.download.DownloadImpl;
import dev.deftu.filestream.download.Networking;
import dev.deftu.filestream.util.HashingHelper;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class DownloaderImpl
implements Downloader {
    private static final boolean TRACE_BYTES = Boolean.getBoolean("filestream.debug.traceDownloadedBytes");
    private static final Logger logger = LogManager.getLogger();
    private final Store downloadStore;

    public DownloaderImpl(Store downloadStore) {
        this.downloadStore = downloadStore;
    }

    @Override
    public Downloader.Download<URL> download(@NotNull URL url, @Nullable Path target, @Nullable Downloader.HashProvider hashProvider, @Nullable Downloader.DownloadCallback callback2) {
        return new DownloadImpl(url, CompletableFuture.supplyAsync(() -> {
            boolean needsLinking;
            logger.trace("Starting download of {}", url);
            Path downloadStoreObject = this.downloadStore.getObject(url.toString());
            logger.trace("Download store object is {}", downloadStoreObject);
            boolean bl = needsLinking = target != null;
            if (!this.isValid(downloadStoreObject, hashProvider)) {
                logger.trace("Invalid local object, downloading {} to {}", url, downloadStoreObject);
                boolean success = this.downloadFile(url, downloadStoreObject, callback2 != null ? callback2 : Downloader.DownloadCallback.NOOP);
                logger.trace("Download of {} to {} was {}", url, downloadStoreObject, success ? "successful" : "unsuccessful");
            }
            if (!needsLinking) {
                logger.trace("No linking required, returning {}", downloadStoreObject);
                return downloadStoreObject;
            }
            return Rewriter.DEFAULT.rewrite(downloadStoreObject, target);
        }));
    }

    @Override
    public Downloader.Download<URL> download(@NotNull URL url, @NotNull Store store, @Nullable Downloader.HashProvider hashProvider, @Nullable Downloader.DownloadCallback callback2) {
        String targetName = HashingHelper.hash(url.toString(), HashingHelper.SHA256);
        Path target = store.getStoreRoot().resolve(targetName);
        return this.download(url, target, hashProvider, callback2);
    }

    private boolean downloadFile(URL url, Path storeObject, @NotNull Downloader.DownloadCallback callback2) {
        HttpURLConnection httpURLConnection;
        logger.trace("Opening connection to {}", url);
        try {
            httpURLConnection = (HttpURLConnection)Networking.createConnection(url);
        }
        catch (IOException e) {
            throw new RuntimeException("Error while opening connection to " + url, e);
        }
        long total = httpURLConnection.getContentLengthLong();
        logger.trace("Connection opened, total size is {}", total);
        logger.trace("Creating download store object");
        try {
            Files.createDirectories(storeObject.getParent(), new FileAttribute[0]);
            if (Files.exists(storeObject, new LinkOption[0])) {
                logger.trace("Download store object already exists, deleting...");
                Files.delete(storeObject);
            }
            Files.createFile(storeObject, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Error while creating download store object " + storeObject, e);
        }
        logger.trace("Downloading {} to {}", url, storeObject);
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        try (InputStream inputStream2 = httpURLConnection.getInputStream();){
            int read;
            byte[] buffer = new byte[1024];
            long totalRead = 0L;
            while ((read = inputStream2.read(buffer, 0, buffer.length)) != -1) {
                if (TRACE_BYTES) {
                    logger.trace("Read {}/{} total bytes", read, totalRead);
                    StringBuilder stringBuilder = new StringBuilder();
                    for (int i2 = 0; i2 < read; ++i2) {
                        stringBuilder.append(String.format("%02X", buffer[i2]));
                    }
                    logger.trace("buffer={}", stringBuilder.toString());
                }
                byteArrayOutputStream.write(buffer, 0, read);
                callback2.updateProgress(totalRead += (long)read, total);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Error while downloading " + url + " to " + storeObject, e);
        }
        finally {
            httpURLConnection.disconnect();
        }
        try {
            byte[] bytes = byteArrayOutputStream.toByteArray();
            if (TRACE_BYTES) {
                logger.trace("Writing {} bytes to {}", bytes.length, storeObject);
                StringBuilder stringBuilder = new StringBuilder();
                for (byte b : bytes) {
                    stringBuilder.append(String.format("%02X", b));
                }
                logger.trace("buffer={}", stringBuilder.toString());
            }
            Files.write(storeObject, bytes, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new RuntimeException("Error while writing to " + storeObject, e);
        }
        return true;
    }

    private boolean isValid(@NotNull Path target, @Nullable Downloader.HashProvider hashProvider) {
        String computedHash;
        logger.trace("Checking if {} is valid", target);
        if (!Files.exists(target, new LinkOption[0])) {
            logger.trace("{} does not exist, invalid", target);
            return false;
        }
        if (hashProvider == null) {
            logger.trace("No hash provider, assuming invalid");
            return false;
        }
        String hash = hashProvider.getHash();
        Supplier<MessageDigest> hashingFunction = hashProvider.getHashingFunction();
        if (hash == null || hashingFunction == null) {
            logger.trace("No hash or hashing function, assuming invalid");
            return false;
        }
        logger.trace("Hash provider for {} returned {}", target, hash);
        try {
            MessageDigest messageDigest = hashingFunction.get();
            computedHash = HashingHelper.hash(target, messageDigest);
        }
        catch (IOException e) {
            throw new RuntimeException("Error while computing hash of " + target, e);
        }
        logger.trace("Computed hash of {} is {}", target, computedHash);
        boolean valid = hash.equals(computedHash);
        logger.trace("Hash is {}valid", valid ? "" : "in");
        return valid;
    }
}

