package me.emafire003.dev.coloredglowlib;

import me.emafire003.dev.coloredglowlib.component.ColorComponent;
import me.emafire003.dev.coloredglowlib.component.GlobalColorComponent;
import me.emafire003.dev.coloredglowlib.custom_data_animations.CustomColorAnimation;
import me.emafire003.dev.coloredglowlib.util.ColorUtils;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1657;
import net.minecraft.class_269;

import static me.emafire003.dev.coloredglowlib.component.ColorComponent.COLOR_COMPONENT;

@SuppressWarnings("unused")
public class ColoredGlowLibAPI {


	//Needed to access server-wide settings such as default color and entitytype's colors
	private final GlobalColorComponent globalColorComponent;

	/**Create an instance of the ColoredGlowLib API
	 *
	 * @param scoreboard An instance of the server/worlds scoreboard that will be used to get
	 *                      server-wide settings such as defaultColor the EntityType's colors and so on*/
	public ColoredGlowLibAPI(class_269 scoreboard){
		this.globalColorComponent = GlobalColorComponent.GLOBAL_COLOR_COMPONENT.get(scoreboard);

	}


	/**Set this to true to override the default minecraft team colors
	 * even of the entity is in a team.
	 * <p>
	 * By default this option is set to <i>false</i>
	 *
	 * @param b The value to assing to overrideTeamColors*/
	public void setOverrideTeamColors(boolean b){
		globalColorComponent.setOverrideTeamColors(b);
	}

	/**Gets the overrideTeamColors option.
	 *
	 * @return Returns true if the default minecraft team colors are being overridden by the mod's ones.
	 * */
	public  boolean getOverrideTeamColors(){
		return globalColorComponent.getOverrideTeamColors();
	}

	/** Makes the Default color override a potential Entity-specific or EntityType-specific color.
	 * <p>
	 * By default this option is <i>false</i>
	 *
	 * @param b Set to true to enable overriding, set false to disable it.
	 */
	public void setDefaultOverridesAll(boolean b){
		globalColorComponent.setDefaultOverridesAll(b);
	}

	/**
	 * Gets the current override status of Default color over others
	 *
	 * @return Returns true if the Default color overrides the Entity/EntityType-specific one
	 * */
	public boolean getDefaultOverridesAll(){
		return globalColorComponent.getDefaultOverridesAll();
	}

	/**
	 * Makes the EntityType-specific color override a potential
	 * Entity-specific color.
	 * <p>
	 * By default this option is <i>false</i>
	 *
	 * @param b Set to true to enable overriding, set false to disable it.
	 * */
	public void setEntityTypeColorOverridesEntityColor(boolean b){
		globalColorComponent.setTypeOverridesEntityColor(b);
	}

	/**
	 * Gets the current override status of EntityType color over Entity Color
	 *
	 * @return Returns true if the EntityType-specific color overrides the Entity-specific one
	 * */
	public boolean getEntityTypeColorOverridesEntityColor(){
		return globalColorComponent.getEntityTypeOverridesEntityColor();
	}
	
	// ========= END OF THE SETTINGS PART =========
	
	
	// ========= START OF THE SET COLOR PART =========


	/** Sets a default global color. The one used by default is <b>"#ffffff</b>, same as vanilla Minecraft.
	 * <p>
	 * If an entity has another color assigned to its EntityType, or itself it will glow that color
	 * instead of this one, unless {@link #setDefaultOverridesAll(boolean)} method is used.
	 *
	 * @param color An hexadecimal color value String, like <b>"#RRGGBB"</b>, or <b>"rainbow"</b> to make the rainbow color,
	 *                 or <b>"random"</b> to make a random color each tick.
	 *                 It can also be a custom color animation name added by a datapack
	 */
	public void setGlobalColor(String color){
		globalColorComponent.setDefaultColor(color);
	}

	/** Makes the default color to be "rainbow", so every entity will by default
	 * glow in rainbow colors, kinda like a _jeb sheep.
	 * <p>
	 * If an entity has another color assigned to its EntityType, or itself it will glow that color
	 * instead of this one, unless {@link #setDefaultOverridesAll(boolean)} method is used.
	 */
	public void setGlobalRainbow(){
		setGlobalColor("rainbow");
	}

	/** Makes the default color to be "random", so every entity will by default
	 * glow in a random color each tick.
	 * <p>
	 * If an entity has another color assigned to its EntityType, or itself it will glow that color
	 * instead of this one, unless {@link #setDefaultOverridesAll(boolean)} method is used.
	 */
	public void setGlobalRandom(){
		setGlobalColor("random");
	}

	/**An alias of {@link #setGlobalColor(String)}*/
	public void setDefaultColor(String color){
		setGlobalColor(color);
	}

	/**Resets the global default color to #ffffff, which is the default color
	 * used by vanilla Minecraft.*/
	public void clearGlobalColor(){
		globalColorComponent.setDefaultColor("#ffffff");
	}

	/**
	 * Sets a custom glow color for an Entity.
	 * The entity will now glow the specified color instead of vanilla minecraft's one.
	 * <p>
	 * This glow color can be overridden by other methods, such as:
	 * {@link #setDefaultOverridesAll(boolean)} and {@link #setEntityTypeColorOverridesEntityColor(boolean)}
	 *
	 * If no color is specified, but an EntityType or default color is, the entity will glow that color.
	 *
	 * @param target The Entity that will glow the specified color
	 * @param color An hexadecimal color value String, like <b>"#RRGGBB"</b>, or <b>"rainbow"</b> to make the rainbow color,
	 *                 or <b>"random"</b> to make a random color each tick.
	 *                 It can also be a custom color animation name added by a datapack
	 */
	public void setColor(class_1297 target, String color){
		ColorComponent component = COLOR_COMPONENT.get(target);
		component.setColor(color);
	}

	/**
	 * Sets a custom glow color for an EntityType.
	 * All entities of the specified EntityType will now glow the specified color instead of vanilla minecraft's one.
	 * <p>
	 * This glow color can be overridden by other methods, such as:
	 * {@link #setDefaultOverridesAll(boolean)}
	 *
	 * If an Entity has a color which is different from its EntityType's one it will glow that color unless
	 * {@link #setEntityTypeColorOverridesEntityColor(boolean)} is enabled
	 *<p>
	 * If no color is specified, but a default color is, the entities will glow that color.
	 *
	 * @param target The EntityType that will glow the specified color
	 * @param color An hexadecimal color value String, like <b>"#RRGGBB"</b>, or <b>"rainbow"</b> to make the rainbow color,
	 *                 or <b>"random"</b> to make a random color each tick.
	 *                 It can also be a custom color animation name added by a datapack
	 */
	public void setColor(class_1299<?> target, String color){
		globalColorComponent.setEntityTypeColor(target, color);
	}

	/**
	 * Sets the custom glow color of an Entity to rainbow.
	 * This will make the entity glow every color periodically like a _jeb sheep
	 *<p>
	 * See {@link #setColor(class_1297, String)} for more information
	 *
	 * @param target The Entity that will glow the specified color
	 * */
	public void setRainbowColor(class_1297 target){
		setColor(target, "rainbow");
	}

	/**
	 * Sets the custom glow color of an Entity to a random, the entity will glow a different color each tick.
	 *<p>
	 * See {@link #setColor(class_1297, String)} for more information
	 *
	 * @param target The Entity that will glow the specified color
	 * */
	public void setRandomColor(class_1297 target){
		setColor(target, "random");
	}

	/**
	 * Sets the custom glow color of an EntityType to rainbow.
	 * This will make the entities of that type glow every color periodically like a _jeb sheep
	 *<p>
	 * See {@link #setColor(class_1299, String)} for more information
	 *
	 * @param target The EntityType that will glow the specified color
	 * */
	public void setRainbowColor(class_1299<?> target){
		setColor(target, "rainbow");
	}

	/**
	 * Sets the custom glow color of an Entity to a random, the entity will glow a different color each tick.
	 *<p>
	 * See {@link #setColor(class_1299, String)} for more information
	 *
	 * @param target The EntityType that will glow the specified color
	 * */
	public void setRandomColor(class_1299<?> target){
		setColor(target, "random");
	}


	/**Removes the custom color from an entity. It will be set back to "#fffff",
	 * unless <b>useDefaultColorInstead</b> is true, in which case the default color
	 * you specified will be used. The default color is the same one that would be applied globally
	 * if {@link #setDefaultOverridesAll(boolean)} is used
	 *
	 * @param entity The entity that will be cleared from the color
	 * @param useDefaultColorInstead Weather or not to use the default color or #ffffff
	 * */
	public void clearColor(class_1297 entity, boolean useDefaultColorInstead){
		ColorComponent component = COLOR_COMPONENT.get(entity);
		if(useDefaultColorInstead){
			component.setColor(globalColorComponent.getDefaultColor());
			return;
		}
		component.clear();
	}

	/**Removes the custom color from an EntityType. It will be set back to "#fffff",
	 * unless <b>useDefaultColorInstead</b> is true, in which case the default color
	 * you specified will be used. The default color is the same one that would be applied globally
	 * if {@link #setDefaultOverridesAll(boolean)} is used.
	 *<p>
	 * This will also clear the rainbow/random/custom color!
	 *
	 * @param entityType The EntityType that will be cleared from the color
	 * @param useDefaultColorInstead Weather or not to use the default color or #ffffff
	 * */
	public void clearColor(class_1299<?> entityType, boolean useDefaultColorInstead){
		if(useDefaultColorInstead){
			globalColorComponent.setEntityTypeColor(entityType, globalColorComponent.getDefaultColor());
			return;
		}
		globalColorComponent.clearEntityTypeColor(entityType);
	}


	/**
	 * Gets the custom glow color of an Entity.
	 *<p>
	 * The result could be "#ffffff" meaning it does not have a custom color and is using
	 * the vanilla one, or "rainbow" meaning its glowing rainbow,
	 * or "random" meaning its glowing a random color each tick,
	 * or another hexadecimal string color.
	 *<p>
	 * It can also return a custom animation name if they are added by a datapack
	 *<p>
	 * If you need a color value instead you can use {@link me.emafire003.dev.coloredglowlib.util.ColorUtils} to manipulate it
 	 *
	 * @param target The Entity to check the color for
	 *
	 * @return The color string associated to that entity
	 * */
	public String getColor(class_1297 target){
		return COLOR_COMPONENT.get(target).getColor();
	}

	/**
	 * Gets the custom glow color of an EntityType.
	 *<p>
	 * The result could be "#ffffff" meaning it does not have a custom color and is using
	 * the vanilla one, or "rainbow" meaning its glowing rainbow,
	 * or "random" meaning its glowing a random color each tick,
	 * or another hexadecimal string color.
	 *<p>
	 * It can also return a custom animation name if they are added by a datapack
	 *<p>
	 * If you need a color value instead you can use {@link me.emafire003.dev.coloredglowlib.util.ColorUtils} to manipulate it
	 *
	 * @param target The EntityType to check the color for
	 *
	 * @return The color string associated to that EntityType
	 * */
	public String getColor(class_1299<?> target){
		return globalColorComponent.getEntityTypeColor(target);
	}


	/**
	 * Gets the global/default custom glow color
	 *<p>
	 * The result could be "#ffffff" meaning it does not have a custom color and is using
	 * the vanilla one, or "rainbow" meaning its glowing rainbow,
	 * or "random" meaning its glowing a random color each tick,
	 * or another hexadecimal string color.
	 *<p>
	 * It can also return a custom animation name if they are added by a datapack
	 *<p>
	 * If you need a color value instead you can use {@link me.emafire003.dev.coloredglowlib.util.ColorUtils} to manipulate it
	 *
	 * @return The color string associated to the global color
	 * */
	public String getGlobalColor(){
		return globalColorComponent.getDefaultColor();
	}

	/**
	 * An alias of {@link #getGlobalColor()}
	 *
	 * @return The color string associated to the global color
	 * */
	public String getDefaultColor(){
		return globalColorComponent.getDefaultColor();
	}

	/**
	 * Checks if an EntityType has a custom glow color or not.
	 * This is done by checking if its color (stored in the global component) is <i>"#ffffff"</i> or not.
	 *<p>
	 * <b>WARNING! This doesn't mean necessarily mean it has a custom color added by a datapack
	 * but that it has a color that is different from the default value of white!</b>
	 *<p>
	 * Warning! If you used {@link #clearColor(class_1299, boolean)} with <i>useDefaultColorInstead</i> to true,
	 * you may want to use: {@link #hasCustomOrDefaultColor(class_1299)}
	 *
	 * @param target The EntityType to check the color for
	 *
	 * @return Returns true if the EntityType has a custom glow color associated to it.
	 */
	public boolean hasCustomColor(class_1299<?> target){
		return !ColorUtils.checkDefault(globalColorComponent.getEntityTypeColor(target));
	}

	/**
	 * Checks if an EntityType has a custom glow color or not.
	 * This is done by checking if its color is <i>"#ffffff"</i> or if it's the defaultColor specified using {@link #setDefaultColor(String)}
	 *
	 * <b>WARNING! This doesn't mean necessarily mean it has a custom color added by a datapack
	 * but that it has a color that is different from the default value of white!</b>
	 *
	 * @param target The EntityType to check the color for
	 *
	 * @return Returns true if the EntityType has a custom glow color associated to it that differs from the defaultColor.
	 */
	public boolean hasCustomOrDefaultColor(class_1299<?> target){
		return !(ColorUtils.checkDefault(globalColorComponent.getEntityTypeColor(target))
				|| ColorUtils.checkSameColor(globalColorComponent.getEntityTypeColor(target), globalColorComponent.getDefaultColor())) ;
	}

	/**
	 * Checks if an Entity has a custom glow color or not.
	 * This is done by checking if its color (stored in the component) is <i>"#ffffff"</i> or not.
	 *<p>
	 * <b>WARNING! This doesn't mean necessarily mean it has a custom color added by a datapack
	 * but that it has a color that is different from the default value of white!</b>
	 *<p>
	 * Warning! If you used {@link #clearColor(class_1297, boolean)} with <i>useDefaultColorInstead</i> to true,
	 * you may want to use: {@link #hasCustomOrDefaultColor(class_1297)}
	 *<p>
	 * @param target The Entity to check the color for
	 *
	 * @return Returns true if the Entity has a custom glow color associated to it.
	 */
	public boolean hasCustomColor(class_1297 target){
		return !ColorUtils.checkDefault(COLOR_COMPONENT.get(target).getColor());
	}

	/**
	 * Checks if an Entity has a custom glow color or not.
	 * This is done by checking if its color is <i>"#ffffff"</i> or if it's the defaultColor specified using {@link #setDefaultColor(String)}
	 *
	 * <b>WARNING! This doesn't mean necessarily mean it has a custom color added by a datapack
	 * but that it has a color that is different from the default value of white!</b>
	 *
	 * @param target The Entity to check the color for
	 *
	 * @return Returns true if the Entity has a custom glow color associated to it that differs from the defaultColor.
	 */
	public  boolean hasCustomOrDefaultColor(class_1297 target){
		return !(ColorUtils.checkDefault(COLOR_COMPONENT.get(target).getColor())
				|| ColorUtils.checkSameColor(COLOR_COMPONENT.get(target).getColor(), globalColorComponent.getDefaultColor())) ;
	}


	/**
	 * Checks is the custom color of EntityType is rainbow
	 *
	 * @param target The EntityType to check the rainbow color for
	 *
	 * @return Returns true if the color associated to that EntityType is rainbow
	 * */
	public boolean hasRainbowColor(class_1299<?> target){
		return globalColorComponent.getEntityTypeColor(target).equalsIgnoreCase("rainbow");
	}

	/**
	 * Checks is the custom color of EntityType is random
	 *
	 * @param target The EntityType to check the random color for
	 *
	 * @return Returns true if the color associated to that EntityType is random
	 * */
	public boolean hasRandomColor(class_1299<?> target){
		return globalColorComponent.getEntityTypeColor(target).equalsIgnoreCase("random");
	}

	/**
	 * Checks is the custom color of Entity is rainbow
	 *
	 * @param target The Entity to check the rainbow color for
	 *
	 * @return Returns true if the color associated to that Entity is rainbow
	 * */
	public boolean hasRainbowColor(class_1297 target){
		return COLOR_COMPONENT.get(target).getColor().equalsIgnoreCase("rainbow");
	}

	/**
	 * Checks is the custom color of Entity is random
	 *
	 * @param target The Entity to check the random color for
	 *
	 * @return Returns true if the color associated to that Entity is random
	 * */
	public boolean hasRandomColor(class_1297 target){
		return COLOR_COMPONENT.get(target).getColor().equalsIgnoreCase("random");
	}

	/**
	 * Checks is the custom color of EntityType is a custom animation added through a datapack
	 *
	 * @param target The EntityType to check the rainbow color for
	 *
	 * @return Returns true if the color associated to that EntityType is added by a datapack
	 * */
	public boolean hasCustomColorAnimation(class_1299<?> target){
		String target_color = globalColorComponent.getEntityTypeColor(target);
		for(CustomColorAnimation animation : ColoredGlowLibMod.getCustomColorAnimations()){
			if(target_color.equalsIgnoreCase(animation.getName())){
				return true;
			}
		}
		return false;
	}

	/**
	 * Checks is the custom color of EntityType is a custom animation added through a datapack
	 *
	 * @param target The Entity to check the random color for
	 *
	 * @return Returns true if the color associated to that Entity is added by a datapack
	 * */
	public boolean hasCustomColorAnimation(class_1297 target){
		String color = COLOR_COMPONENT.get(target).getColor();
		for(CustomColorAnimation customColorAnimation : ColoredGlowLibMod.getCustomColorAnimations()){
			if(color.equalsIgnoreCase(customColorAnimation.getName())){
				return true;
			}
		}
		return false;
	}


	//new stuff

	/**
	 * Sets a custom glow color for an Entity, visible only to another player.
	 * The entity will now glow the specified color instead of vanilla minecraft's one.
	 * <p>
	 * This glow color can be overridden by other methods, such as:
	 * {@link #setDefaultOverridesAll(boolean)} and {@link #setEntityTypeColorOverridesEntityColor(boolean)}
	 *
	 * If no color is specified, but an EntityType or default color is, the entity will glow that color.
	 *
	 * @param target The Entity that will glow the specified color
	 * @param color An hexadecimal color value String, like <b>"#RRGGBB"</b>, or <b>"rainbow"</b> to make the rainbow color,
	 *                 or <b>"random"</b> to make a random color each tick.
	 *                 It can also be a custom color animation name added by a datapack
	 * @param colorViewer The only Player who will see the specified color
	 */
	public void setExclusiveColorFor(class_1297 target, String color, class_1657 colorViewer){
		ColorComponent component = COLOR_COMPONENT.get(target);
		component.setExclusiveColorFor(colorViewer.method_5667(), color);
	}

	/**Removes the custom color from an entity visible to a certain player. It will be set back to "#fffff",
	 * unless <b>useDefaultColorInstead</b> is true, in which case the default color
	 * you specified will be used. The default color is the same one that would be applied globally
	 * if {@link #setDefaultOverridesAll(boolean)} is used
	 * <p>
	 * <b>IT WILL ONLY CLEAR THE COLOR SET BY {@link #setExclusiveColorFor(class_1297, String, class_1657)} </b>
	 *
	 * @param entity The entity that will be cleared from the color
	 * @param colorViewer The player that is currently seeing the custom exclusive color
	 * @param useDefaultColorInstead Weather or not to use the default color or #ffffff
	 * */
	public void clearExclusiveColorFor(class_1297 entity, class_1657 colorViewer, boolean useDefaultColorInstead){
		ColorComponent component = COLOR_COMPONENT.get(entity);
		if(useDefaultColorInstead){
			component.setExclusiveColorFor(colorViewer.method_5667(), globalColorComponent.getDefaultColor());
			return;
		}
		component.clearExclusiveColorFor(colorViewer.method_5667());
	}


	/**
	 * Gets the custom glow color of an Entity that only the specified player sees.
	 *<p>
	 * The result could be "#ffffff" meaning it does not have a custom color and is using
	 * the vanilla one, or "rainbow" meaning its glowing rainbow,
	 * or "random" meaning its glowing a random color each tick,
	 * or another hexadecimal string color.
	 *<p>
	 * It can also return a custom animation name if they are added by a datapack
	 *<p>
	 * If you need a color value instead you can use {@link me.emafire003.dev.coloredglowlib.util.ColorUtils} to manipulate it
	 *
	 * @param target The Entity to check the color for
	 * @param colorViewer The player that is viewing the custom exclusive color
	 *
	 * @return The color string associated to that entity
	 * */
	public String getExclusiveColorFor(class_1297 target, class_1657 colorViewer){
		return COLOR_COMPONENT.get(target).getExclusiveColorFor(colorViewer.method_5667());
	}

	/**
	 * Checks if an Entity has a custom glow color visible only to another player or not.
	 *<p>
	 * <b>WARNING! This doesn't mean necessarily mean it has a custom color added by a datapack
	 * but that it has a color that is different from the default value of white!</b>
	 *<p>
	 * Warning! If you used {@link #clearColor(class_1299, boolean)} with <i>useDefaultColorInstead</i> to true,
	 * you may want to use: {@link #hasExclusiveCustomOrDefaultColorFor(class_1297, class_1657)}
	 *
	 * @param target The EntityType to check the color for
	 * @param colorViewer The player that could be seeing the custom exclusive color
	 *
	 * @return Returns true if the Entity has a custom glow color associated to the player.
	 */
	public boolean hasExclusiveCustomColorFor(class_1297 target, class_1657 colorViewer){
		return !ColorUtils.checkDefault(COLOR_COMPONENT.get(target).getExclusiveColorFor(colorViewer.method_5667()));
	}

	/**
	 * Checks if an Entity has a custom glow color visible only to the specified player or not.
	 * <p>
	 * <b>WARNING! This doesn't mean necessarily mean it has a custom color added by a datapack
	 * but that it has a color that is different from the default value of white!</b>
	 *
	 * @param target The Entity to check the color for
	 * @param colorViewer The player that could be seeing the custom exclusive color
	 *
	 * @return Returns true if the Entity has a custom glow color associated to it that differs from the defaultColor.
	 */
	public  boolean hasExclusiveCustomOrDefaultColorFor(class_1297 target, class_1657 colorViewer){
		return !(ColorUtils.checkDefault(COLOR_COMPONENT.get(target).getExclusiveColorFor(colorViewer.method_5667()))
				|| ColorUtils.checkSameColor(COLOR_COMPONENT.get(target).getExclusiveColorFor(colorViewer.method_5667()), globalColorComponent.getDefaultColor())) ;
	}
}
