/*
 * Decompiled with CFR 0.152.
 */
package net.momirealms.craftengine.core.world.chunk.storage;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.IntBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import net.momirealms.craftengine.core.plugin.CraftEngine;
import net.momirealms.craftengine.core.plugin.logger.PluginLogger;
import net.momirealms.craftengine.core.world.ChunkPos;
import net.momirealms.craftengine.core.world.chunk.storage.CompressionMethod;
import net.momirealms.craftengine.core.world.chunk.storage.RegionBitmap;
import net.momirealms.craftengine.libraries.nbt.CompoundTag;
import net.momirealms.craftengine.libraries.nbt.NBT;
import org.jetbrains.annotations.Nullable;

public class RegionFile
implements AutoCloseable {
    private static final PluginLogger LOGGER = CraftEngine.instance().logger();
    private static final byte FORMAT_VERSION = 1;
    public static final int SECTOR_BYTES = 4096;
    public static final int CHUNK_HEADER_SIZE = 5;
    public static final int EXTERNAL_STREAM_FLAG = 128;
    public static final int EXTERNAL_CHUNK_THRESHOLD = 256;
    public static final int MAX_CHUNK_SIZE = 524288000;
    public static final int INFO_NOT_PRESENT = 0;
    public static final String EXTERNAL_FILE_SUFFIX = ".mcc";
    public static final String EXTERNAL_FILE_PREFIX = "c.";
    private static final ByteBuffer PADDING_BUFFER = ByteBuffer.allocateDirect(1);
    private final FileChannel fileChannel;
    private final Path directory;
    private final CompressionMethod compression;
    private final ByteBuffer header;
    private final IntBuffer sectorInfo;
    private final IntBuffer timestamps;
    private final RegionBitmap usedSectors;
    public final ReentrantLock fileLock = new ReentrantLock(true);
    public final Path regionFile;
    private static final List<Function<DataInputStream, DataInputStream>> FORMAT_UPDATER = List.of(old -> {
        try {
            CompoundTag tag = NBT.readCompound(new DataInputStream((InputStream)old), true);
            return new DataInputStream(new ByteArrayInputStream(NBT.toBytes(tag)));
        }
        catch (IOException e) {
            CraftEngine.instance().logger().warn("Failed to migrate data from version 0 -> 1", e);
            return null;
        }
    });

    public RegionFile(Path path, Path directory, CompressionMethod compressionMethod) throws IOException {
        this.header = ByteBuffer.allocateDirect(8192);
        this.regionFile = path;
        this.usedSectors = new RegionBitmap();
        this.compression = compressionMethod;
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            throw new IllegalArgumentException("Expected directory, got " + String.valueOf(directory.toAbsolutePath()));
        }
        this.directory = directory;
        this.sectorInfo = this.header.asIntBuffer();
        this.sectorInfo.limit(1024);
        this.header.position(4096);
        this.timestamps = this.header.asIntBuffer();
        this.fileChannel = FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        this.usedSectors.allocate(0, 2);
        this.header.position(0);
        int byteAmount = this.fileChannel.read(this.header, 0L);
        if (byteAmount == -1) {
            return;
        }
        if (byteAmount != 8192) {
            LOGGER.warn(String.format("Region file %s has truncated header: %s", path, byteAmount));
        }
        long regionSize = Files.size(path);
        for (int chunkLocation = 0; chunkLocation < 1024; ++chunkLocation) {
            int sectorInfo = this.sectorInfo.get(chunkLocation);
            if (sectorInfo == 0) continue;
            int offset = RegionFile.unpackSectorOffset(sectorInfo);
            int size = RegionFile.unpackSectorSize(sectorInfo);
            if (offset < 2) {
                LOGGER.warn(String.format("Region file %s has invalid sector at index: %s; sector %s overlaps with header", path, chunkLocation, offset));
                this.sectorInfo.put(chunkLocation, 0);
                continue;
            }
            if (size == 0) {
                LOGGER.warn(String.format("Region file %s has an invalid sector at index: %s; size has to be > 0", path, chunkLocation));
                this.sectorInfo.put(chunkLocation, 0);
                continue;
            }
            if ((long)offset * 4096L > regionSize) {
                LOGGER.warn(String.format("Region file %s has an invalid sector at index: %s; sector %s is out of bounds", path, chunkLocation, offset));
                this.sectorInfo.put(chunkLocation, 0);
                continue;
            }
            this.usedSectors.allocate(offset, size);
        }
    }

    @Nullable
    public synchronized DataInputStream getChunkDataInputStream(ChunkPos pos) throws IOException {
        int sectorInfo = this.getSectorInfo(pos);
        if (sectorInfo == 0) {
            return null;
        }
        int sectorOffset = RegionFile.unpackSectorOffset(sectorInfo);
        int sectorSize = RegionFile.unpackSectorSize(sectorInfo);
        int totalSize = sectorSize * 4096;
        ByteBuffer bytebuffer = ByteBuffer.allocate(totalSize);
        this.fileChannel.read(bytebuffer, (long)sectorOffset * 4096L);
        ((Buffer)bytebuffer).flip();
        if (bytebuffer.remaining() < 5) {
            LOGGER.severe(String.format("Chunk %s header is truncated: expected %s but read %s", pos, totalSize, bytebuffer.remaining()));
            return null;
        }
        int size = bytebuffer.getInt();
        byte flags = bytebuffer.get();
        byte compressionScheme = (byte)(flags & 7);
        int version = (flags & 0x78) >>> 3;
        if (size == 0) {
            LOGGER.warn(String.format("Chunk %s is allocated, but stream is missing", pos));
            return null;
        }
        int actualSize = size - 1;
        if (RegionFile.isExternalStreamChunk(flags)) {
            if (actualSize != 0) {
                LOGGER.warn("Chunk has both internal and external streams");
            }
            if (version == 1) {
                return this.createExternalChunkInputStream(pos, RegionFile.getExternalChunkVersion(compressionScheme));
            }
            DataInputStream inputStream = this.createExternalChunkInputStream(pos, RegionFile.getExternalChunkVersion(compressionScheme));
            for (int currentVersion = version; currentVersion < 1 && (inputStream = FORMAT_UPDATER.get(currentVersion).apply(inputStream)) != null; ++currentVersion) {
            }
            return inputStream;
        }
        if (actualSize > bytebuffer.remaining()) {
            LOGGER.severe(String.format("Chunk %s stream is truncated: expected %s but read %s", pos, actualSize, bytebuffer.remaining()));
            return null;
        }
        if (actualSize < 0) {
            LOGGER.severe(String.format("Declared size %s of chunk %s is negative", size, pos));
            return null;
        }
        if (version == 1) {
            return this.createChunkInputStream(pos, compressionScheme, RegionFile.createInputStream(bytebuffer, actualSize));
        }
        DataInputStream inputStream = this.createChunkInputStream(pos, compressionScheme, RegionFile.createInputStream(bytebuffer, actualSize));
        for (int currentVersion = version; currentVersion < 1 && (inputStream = FORMAT_UPDATER.get(currentVersion).apply(inputStream)) != null; ++currentVersion) {
        }
        return inputStream;
    }

    public static byte encodeFlag(byte compressionScheme, byte version, boolean external) {
        if (compressionScheme <= 0 || compressionScheme > 7) {
            throw new IllegalArgumentException("compression method can only be a number between 1 and 7");
        }
        if (version < 0 || version > 15) {
            throw new IllegalArgumentException("Version number can only be a number between 0 and 15");
        }
        return (byte)((external ? 128 : 0) | (version & 0xF) << 3 | compressionScheme & 7);
    }

    private static int getTimestamp() {
        return (int)(Instant.now().toEpochMilli() / 1000L);
    }

    private static boolean isExternalStreamChunk(byte flags) {
        return (flags & 0x80) != 0;
    }

    private static byte getExternalChunkVersion(byte flags) {
        return (byte)(flags & 0xFFFFFF7F);
    }

    @Nullable
    private DataInputStream createChunkInputStream(ChunkPos pos, byte flags, InputStream stream) throws IOException {
        CompressionMethod compressionMethod = CompressionMethod.fromId(flags);
        if (compressionMethod == null) {
            LOGGER.severe(String.format("Chunk %s has invalid chunk stream version %s", pos, flags));
            return null;
        }
        return new DataInputStream(compressionMethod.wrap(stream));
    }

    @Nullable
    private DataInputStream createExternalChunkInputStream(ChunkPos pos, byte flags) throws IOException {
        Path path = this.getExternalChunkPath(pos);
        if (!Files.isRegularFile(path, new LinkOption[0])) {
            LOGGER.severe(String.format("External chunk path %s is not file", path));
            return null;
        }
        return this.createChunkInputStream(pos, flags, Files.newInputStream(path, new OpenOption[0]));
    }

    private static ByteArrayInputStream createInputStream(ByteBuffer buffer, int length) {
        return new ByteArrayInputStream(buffer.array(), buffer.position(), length);
    }

    private int packSectorOffset(int offset, int size) {
        return offset << 8 | size;
    }

    private static int unpackSectorSize(int sectorData) {
        return sectorData & 0xFF;
    }

    private static int unpackSectorOffset(int sectorData) {
        return sectorData >> 8 & 0xFFFFFF;
    }

    private static int sizeToSectors(int byteCount) {
        return (byteCount + 4096 - 1) / 4096;
    }

    public synchronized boolean doesChunkExist(ChunkPos pos) {
        int sectorInfo = this.getSectorInfo(pos);
        if (sectorInfo == 0) {
            return false;
        }
        int sectorOffset = RegionFile.unpackSectorOffset(sectorInfo);
        int sectorSize = RegionFile.unpackSectorSize(sectorInfo);
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        try {
            this.fileChannel.read(bytebuffer, (long)sectorOffset * 4096L);
            ((Buffer)bytebuffer).flip();
            if (bytebuffer.remaining() != 5) {
                return false;
            }
            int size = bytebuffer.getInt();
            byte type = bytebuffer.get();
            if (RegionFile.isExternalStreamChunk(type)) {
                if (!CompressionMethod.isValid(RegionFile.getExternalChunkVersion(type))) {
                    return false;
                }
                return Files.isRegularFile(this.getExternalChunkPath(pos), new LinkOption[0]);
            }
            if (!CompressionMethod.isValid(type)) {
                return false;
            }
            if (size == 0) {
                return false;
            }
            int actualSize = size - 1;
            return actualSize >= 0 && actualSize <= 4096 * sectorSize;
        }
        catch (IOException ioexception) {
            return false;
        }
    }

    public DataOutputStream getChunkDataOutputStream(ChunkPos pos) throws IOException {
        return new DataOutputStream(this.compression.wrap(new ChunkBuffer(pos)));
    }

    public void flush() throws IOException {
        this.fileChannel.force(true);
    }

    public void clear(ChunkPos pos) throws IOException {
        int chunkLocation = RegionFile.getChunkLocation(pos);
        int sectorInfo = this.sectorInfo.get(chunkLocation);
        if (sectorInfo != 0) {
            this.sectorInfo.put(chunkLocation, 0);
            this.timestamps.put(chunkLocation, RegionFile.getTimestamp());
            this.writeHeader();
            Files.deleteIfExists(this.getExternalChunkPath(pos));
            this.usedSectors.free(RegionFile.unpackSectorOffset(sectorInfo), RegionFile.unpackSectorSize(sectorInfo));
        }
    }

    protected synchronized void write(ChunkPos pos, ByteBuffer buf) throws IOException {
        CommitOp regionFileOperation;
        int sectorStartPosition;
        int offsetIndex = RegionFile.getChunkLocation(pos);
        int previousSectorInfo = this.sectorInfo.get(offsetIndex);
        int previousSectorOffset = RegionFile.unpackSectorOffset(previousSectorInfo);
        int previousSectorSize = RegionFile.unpackSectorSize(previousSectorInfo);
        int sizeToWrite = buf.remaining();
        int sectorsToWrite = RegionFile.sizeToSectors(sizeToWrite);
        if (sectorsToWrite >= 256) {
            Path path = this.getExternalChunkPath(pos);
            LOGGER.warn(String.format("Saving oversized chunk %s (%s bytes) to external file %s", pos.x() + "," + pos.z(), sizeToWrite, path));
            sectorsToWrite = 1;
            sectorStartPosition = this.usedSectors.allocate(sectorsToWrite);
            regionFileOperation = this.writeToExternalFileSafely(path, buf);
            ByteBuffer externalBuf = this.createExternalHeader();
            this.fileChannel.write(externalBuf, (long)sectorStartPosition * 4096L);
        } else {
            sectorStartPosition = this.usedSectors.allocate(sectorsToWrite);
            regionFileOperation = () -> Files.deleteIfExists(this.getExternalChunkPath(pos));
            this.fileChannel.write(buf, (long)sectorStartPosition * 4096L);
        }
        this.sectorInfo.put(offsetIndex, this.packSectorOffset(sectorStartPosition, sectorsToWrite));
        this.timestamps.put(offsetIndex, RegionFile.getTimestamp());
        this.writeHeader();
        regionFileOperation.run();
        if (previousSectorOffset != 0) {
            this.usedSectors.free(previousSectorOffset, previousSectorSize);
        }
    }

    private ByteBuffer createExternalHeader() {
        return this.createExternalHeader(this.compression);
    }

    private ByteBuffer createExternalHeader(CompressionMethod compression) {
        ByteBuffer bytebuffer = ByteBuffer.allocate(5);
        bytebuffer.putInt(1);
        bytebuffer.put(RegionFile.encodeFlag((byte)compression.getId(), (byte)1, true));
        ((Buffer)bytebuffer).flip();
        return bytebuffer;
    }

    private CommitOp writeToExternalFileSafely(Path destination, ByteBuffer buf) throws IOException {
        Path tempFile = Files.createTempFile(this.directory, "tmp", null, new FileAttribute[0]);
        FileChannel filechannel = FileChannel.open(tempFile, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
        try {
            ((Buffer)buf).position(5);
            filechannel.write(buf);
        }
        catch (Throwable t1) {
            if (filechannel != null) {
                try {
                    filechannel.close();
                }
                catch (Throwable t2) {
                    t1.addSuppressed(t2);
                }
            }
            throw t1;
        }
        filechannel.close();
        return () -> Files.move(tempFile, destination, StandardCopyOption.REPLACE_EXISTING);
    }

    private void writeHeader() throws IOException {
        this.header.position(0);
        this.fileChannel.write(this.header, 0L);
    }

    private int getSectorInfo(ChunkPos pos) {
        return this.sectorInfo.get(RegionFile.getChunkLocation(pos));
    }

    public boolean hasChunk(ChunkPos pos) {
        return this.getSectorInfo(pos) != 0;
    }

    private static int getChunkLocation(ChunkPos pos) {
        return pos.regionLocalX() + pos.regionLocalZ() * 32;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        this.fileLock.lock();
        RegionFile regionFile = this;
        synchronized (regionFile) {
            try {
                try {
                    this.padToFullSector();
                }
                finally {
                    try {
                        this.fileChannel.force(true);
                    }
                    finally {
                        this.fileChannel.close();
                    }
                }
            }
            finally {
                this.fileLock.unlock();
            }
        }
    }

    private void padToFullSector() throws IOException {
        int expectedSize;
        int currentSize = (int)this.fileChannel.size();
        if (currentSize != (expectedSize = RegionFile.sizeToSectors(currentSize) * 4096)) {
            ByteBuffer bytebuffer = PADDING_BUFFER.duplicate();
            ((Buffer)bytebuffer).position(0);
            this.fileChannel.write(bytebuffer, expectedSize - 1);
        }
    }

    private static int getChunkIndex(int x, int z) {
        return (x & 0x1F) + (z & 0x1F) * 32;
    }

    private Path getExternalChunkPath(ChunkPos chunkPos) {
        String s = EXTERNAL_FILE_PREFIX + chunkPos.x + "." + chunkPos.z + EXTERNAL_FILE_SUFFIX;
        return this.directory.resolve(s);
    }

    private class ChunkBuffer
    extends ByteArrayOutputStream {
        private final ChunkPos pos;

        public ChunkBuffer(ChunkPos pos) {
            super(8096);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(0);
            super.write(RegionFile.encodeFlag((byte)RegionFile.this.compression.getId(), (byte)1, false));
            this.pos = pos;
        }

        @Override
        public void write(int b) {
            if (this.count > 524288000) {
                throw new RegionFileSizeException("Region file too large: " + this.count);
            }
            super.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) {
            if (this.count + len > 524288000) {
                throw new RegionFileSizeException("Region file too large: " + (this.count + len));
            }
            super.write(b, off, len);
        }

        @Override
        public void close() throws IOException {
            ByteBuffer bytebuffer = ByteBuffer.wrap(this.buf, 0, this.count);
            bytebuffer.putInt(0, this.count - 5 + 1);
            RegionFile.this.write(this.pos, bytebuffer);
        }
    }

    private static interface CommitOp {
        public void run() throws IOException;
    }

    public static final class RegionFileSizeException
    extends RuntimeException {
        public RegionFileSizeException(String message) {
            super(message);
        }
    }
}

