package io.github.gaming32.bingo.client;

import com.mojang.blaze3d.platform.InputConstants;
import com.mojang.blaze3d.platform.Window;
import io.github.gaming32.bingo.Bingo;
import io.github.gaming32.bingo.client.config.BingoClientConfig;
import io.github.gaming32.bingo.client.icons.DefaultIconRenderers;
import io.github.gaming32.bingo.client.icons.IconRenderer;
import io.github.gaming32.bingo.client.icons.IconRenderers;
import io.github.gaming32.bingo.client.icons.pip.BlockPictureInPictureRenderState;
import io.github.gaming32.bingo.client.icons.pip.BlockPictureInPictureRenderer;
import io.github.gaming32.bingo.client.recipeviewer.RecipeViewerPlugin;
import io.github.gaming32.bingo.data.icons.GoalIcon;
import io.github.gaming32.bingo.game.BingoBoard;
import io.github.gaming32.bingo.game.GoalProgress;
import io.github.gaming32.bingo.game.mode.BingoGameMode;
import io.github.gaming32.bingo.network.ClientPayloadHandler;
import io.github.gaming32.bingo.network.messages.both.ManualHighlightPayload;
import io.github.gaming32.bingo.platform.BingoPlatform;
import io.github.gaming32.bingo.platform.event.ClientEvents;
import io.github.gaming32.bingo.platform.registrar.KeyMappingBuilder;
import io.github.gaming32.bingo.util.ResourceLocations;
import io.github.gaming32.bingo.util.Vec2i;
import net.minecraft.ChatFormatting;
import net.minecraft.client.KeyMapping;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.AbstractButton;
import net.minecraft.client.gui.screens.ChatScreen;
import net.minecraft.client.input.KeyEvent;
import net.minecraft.client.multiplayer.ClientLevel;
import net.minecraft.client.multiplayer.ClientPacketListener;
import net.minecraft.client.multiplayer.PlayerInfo;
import net.minecraft.client.renderer.RenderPipelines;
import net.minecraft.locale.Language;
import net.minecraft.network.chat.CommonComponents;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.FormattedText;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.ARGB;
import net.minecraft.util.FormattedCharSequence;
import net.minecraft.util.Mth;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.scores.PlayerTeam;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Objects;

public class BingoClient {
    private static final ResourceLocation BOARD_TEXTURE = ResourceLocations.bingo("board");
    private static final ResourceLocation BOARD_CELL_TEXTURE = ResourceLocations.bingo("board_cell");
    private static final ResourceLocation SLOT_HIGHLIGHT_BACK_SPRITE = ResourceLocations.minecraft("container/slot_highlight_back");
    private static final ResourceLocation SLOT_HIGHLIGHT_FRONT_SPRITE = ResourceLocations.minecraft("container/slot_highlight_front");
    public static final Component BOARD_TITLE = Component.translatable("bingo.board.title");
    public static final Component BOARD_TITLE_SHORT = Component.translatable("bingo.board.title.short");

    public static final int BOARD_OFFSET = 3;

    public static KeyMapping manualHighlightKeyMapping;

    public static BingoBoard.Teams clientTeam = BingoBoard.Teams.NONE;
    public static BingoBoard.Teams receivedClientTeam = BingoBoard.Teams.NONE;
    public static ClientGame clientGame;

    public static final BingoClientConfig CONFIG = new BingoClientConfig(
        BingoPlatform.platform.getConfigDir().resolve("bingo-client.toml")
    );
    private static RecipeViewerPlugin recipeViewerPlugin;

    public static void init() {
        CONFIG.load();
        CONFIG.save();

        registerEventHandlers();

        DefaultIconRenderers.setup();

        BingoPlatform.platform.registerKeyMappings(builder -> {
            KeyMapping.Category category = builder.registerCategory(ResourceLocations.bingo("category"));
            builder
                .name("bingo.key.board")
                .category(category)
                .keyCode(InputConstants.KEY_B)
                .conflictContext(KeyMappingBuilder.ConflictContext.IN_GAME)
                .register(minecraft -> {
                    if (clientGame != null) {
                        minecraft.setScreen(new BoardScreen());
                    }
                });
            manualHighlightKeyMapping = builder
                .name("bingo.key.manual_highlight")
                .category(category)
                .keyType(InputConstants.Type.MOUSE)
                .keyCode(InputConstants.MOUSE_BUTTON_LEFT)
                .conflictContext(KeyMappingBuilder.ConflictContext.UNIVERSAL) // TODO: better conflict context
                .register(minecraft -> {})
                .mapping();
        });

        BingoPlatform.platform.registerClientTooltips(registrar -> registrar.register(IconTooltip.class, ClientIconTooltip::new));
        BingoPlatform.platform.registerPictureInPictureRenderers(registrar ->
            registrar.register(BlockPictureInPictureRenderState.class, BlockPictureInPictureRenderer::new)
        );

        ClientPayloadHandler.init(new ClientPayloadHandlerImpl());

        Bingo.LOGGER.info("Bongo");
    }

    private static void registerEventHandlers() {
        ClientEvents.KEY_RELEASED_PRE.register((screen, event) -> {
            if (clientGame == null || !(screen instanceof ChatScreen)) {
                return false;
            }
            return detectPress(event, getBoardPosition());
        });

        ClientEvents.MOUSE_RELEASED_PRE.register((screen, event) -> {
            if (clientGame == null || !(screen instanceof ChatScreen)) {
                return false;
            }
            return detectClick(event.button(), getBoardPosition());
        });

        ClientEvents.PLAYER_QUIT.register(player -> {
            clientTeam = receivedClientTeam = BingoBoard.Teams.NONE;
            clientGame = null;
        });

        ClientEvents.CLIENT_TICK_END.register(minecraft -> {
            if (minecraft.player == null || !minecraft.player.isSpectator()) {
                clientTeam = receivedClientTeam;
            } else if (minecraft.player.isSpectator() && !clientTeam.any()) {
                clientTeam = BingoBoard.Teams.TEAM1;
            }
        });
    }

    public static PositionAndScale getBoardPosition() {
        final Window window = Minecraft.getInstance().getWindow();
        final float scale = CONFIG.getBoardScale();
        final float x = CONFIG.getBoardCorner().getX(window.getGuiScaledWidth(), scale);
        final float y = CONFIG.getBoardCorner().getY(window.getGuiScaledHeight(), scale);
        return new PositionAndScale(x, y, scale);
    }

    public static RecipeViewerPlugin getRecipeViewerPlugin() {
        if (recipeViewerPlugin == null) {
            recipeViewerPlugin = RecipeViewerPlugin.detect();
        }
        return recipeViewerPlugin;
    }

    public static void renderBoardOnHud(Minecraft minecraft, GuiGraphics graphics) {
        if (clientGame == null) {
            return;
        }
        if (minecraft.getDebugOverlay().showDebugScreen() && !BingoClient.CONFIG.showBoardOnF3Screen()) {
            return;
        }
        if (minecraft.screen instanceof BoardScreen) {
            return;
        }

        final PositionAndScale pos = getBoardPosition();
        renderBingo(graphics, minecraft.screen instanceof ChatScreen, pos);

        if (CONFIG.isShowScoreCounter() && clientGame.renderMode() == BingoGameMode.RenderMode.ALL_TEAMS) {
            class TeamValue {
                final BingoBoard.Teams team;
                int score;

                TeamValue(BingoBoard.Teams team) {
                    this.team = team;
                }
            }

            final TeamValue[] teams = new TeamValue[clientGame.teams().length];
            for (int i = 0; i < teams.length; i++) {
                teams[i] = new TeamValue(BingoBoard.Teams.fromOne(i));
            }

            int totalScore = 0;
            for (final BingoBoard.Teams state : clientGame.states()) {
                if (state.any()) {
                    totalScore++;
                    teams[state.getFirstIndex()].score++;
                }
            }

            Arrays.sort(teams, Comparator.comparing(v -> -v.score)); // Sort in reverse

            final Font font = minecraft.font;
            final int scoreX = (int)(pos.x() * pos.scale() + getBoardWidth() * pos.scale() / 2);
            int scoreY;
            if (CONFIG.getBoardCorner().isOnBottom) {
                scoreY = (int)((pos.y() - BOARD_OFFSET) * pos.scale() - font.lineHeight);
            } else {
                scoreY = (int)(pos.y() * pos.scale() + (getBoardHeight() + BOARD_OFFSET) * pos.scale());
            }
            final int shift = CONFIG.getBoardCorner().isOnBottom ? -12 : 12;
            for (final TeamValue teamValue : teams) {
                if (teamValue.score == 0) break;
                final PlayerTeam team = clientGame.teams()[teamValue.team.getFirstIndex()];
                final MutableComponent leftText = getDisplayName(team).copy();
                final MutableComponent rightText = Component.literal(" - " + teamValue.score);
                if (team.getColor() != ChatFormatting.RESET) {
                    leftText.withStyle(team.getColor());
                    rightText.withStyle(team.getColor());
                }
                graphics.drawString(font, leftText, scoreX - font.width(leftText), scoreY, 0xffffffff);
                graphics.drawString(font, rightText, scoreX, scoreY, 0xffffffff);
                scoreY += shift;
            }

            final MutableComponent leftText = Component.translatable("bingo.unclaimed");
            final MutableComponent rightText = Component.literal(" - " + (clientGame.states().length - totalScore));
            graphics.drawString(font, leftText, scoreX - font.width(leftText), scoreY, 0xffffffff);
            graphics.drawString(font, rightText, scoreX, scoreY, 0xffffffff);
        }
    }

    public static int getBoardWidth() {
        return 14 + 18 * clientGame.shape().getVisualSize(clientGame.size()).x();
    }

    public static int getBoardHeight() {
        return 24 + 18 * clientGame.shape().getVisualSize(clientGame.size()).y();
    }

    public static void renderBingo(GuiGraphics graphics, boolean mouseHover, PositionAndScale pos) {
        if (clientGame == null) {
            Bingo.LOGGER.warn("BingoClient.renderBingo() called when Bingo.clientGame == null!");
            return;
        }
        final Minecraft minecraft = Minecraft.getInstance();

        graphics.pose().pushMatrix();
        graphics.pose().scale(pos.scale(), pos.scale());
        // get rid of the fractional part to avoid funky matrix values causing rendering artifacts
        graphics.pose().translate((int) pos.x(), (int) pos.y());

        final BingoMousePos mousePos = mouseHover ? BingoMousePos.getPos(minecraft, clientGame.shape(), clientGame.size(), pos) : null;
        final Vec2i visualSize = clientGame.shape().getVisualSize(clientGame.size());
        final int goalCount = clientGame.shape().getGoalCount(clientGame.size());

        graphics.blitSprite(
            RenderPipelines.GUI_TEXTURED,
            BOARD_TEXTURE, 0, 0,
            7 + 18 * visualSize.x() + 7,
            17 + 18 * visualSize.y() + 7
        );

        for (int goalIndex = 0; goalIndex < goalCount; goalIndex++) {
            final Vec2i slotPos = clientGame.shape().getCoords(clientGame.size(), goalIndex);
            final int slotX = slotPos.x() * 18 + 7;
            final int slotY = slotPos.y() * 18 + 17;
            graphics.blitSprite(RenderPipelines.GUI_TEXTURED, BOARD_CELL_TEXTURE, slotX, slotY, 18, 18);
        }

        renderBoardTitle(graphics, minecraft.font);

        if (BingoMousePos.hasSlotPos(mousePos)) {
            graphics.pose().pushMatrix();
            final Vec2i slotPos = mousePos.getSlotPos(clientGame);
            final int slotX = slotPos.x() * 18 + 8;
            final int slotY = slotPos.y() * 18 + 18;
            graphics.blitSprite(RenderPipelines.GUI_TEXTURED, SLOT_HIGHLIGHT_BACK_SPRITE, slotX - 4, slotY - 4, 24, 24);
            graphics.pose().popMatrix();
        }

        final boolean spectator = minecraft.player != null && minecraft.player.isSpectator();
        for (int goalIndex = 0; goalIndex < goalCount; goalIndex++) {
            final Vec2i slotPos = clientGame.shape().getCoords(clientGame.size(), goalIndex);
            final var goal = clientGame.goals()[goalIndex];
            final BingoBoard.Teams state = clientGame.states()[goalIndex];
            final boolean isGoalCompleted = state.and(clientTeam);
            final int slotX = slotPos.x() * 18 + 8;
            final int slotY = slotPos.y() * 18 + 18;

            final Integer color = switch (clientGame.renderMode()) {
                case FANCY -> isGoalCompleted ? Integer.valueOf(0x55ff55) : goal.specialType().incompleteColor;
                case ALL_TEAMS -> {
                    if (!state.any()) {
                        yield null;
                    }
                    final BingoBoard.Teams team = isGoalCompleted ? clientTeam : state;
                    final Integer maybeColor = clientGame.teams()[team.getFirstIndex()].getColor().getColor();
                    yield maybeColor != null ? maybeColor : 0x55ff55;
                }
            };
            if (color != null) {
                graphics.fill(slotX, slotY, slotX + 16, slotY + 16, 0xA0000000 | color);
            }

            Integer manualHighlight = clientGame.manualHighlights()[goalIndex];
            if (manualHighlight != null) {
                int highlightColor = ARGB.color(0xff, BingoClient.CONFIG.getManualHighlightColor(manualHighlight));
                graphics.hLine(slotX, slotX + 15, slotY, highlightColor);
                graphics.hLine(slotX, slotX + 15, slotY + 15, highlightColor);
                graphics.vLine(slotX, slotY, slotY + 15, highlightColor);
                graphics.vLine(slotX + 15, slotY, slotY + 15, highlightColor);
            }

            final GoalIcon icon = goal.icon();
            final IconRenderer<? super GoalIcon> renderer = IconRenderers.getRenderer(icon);
            renderer.render(icon, graphics, slotX, slotY);
            renderer.renderDecorations(icon, minecraft.font, graphics, slotX, slotY);

            GoalProgress progress = clientGame.progress()[goalIndex];
            if (progress != null && !isGoalCompleted && progress.progress() > 0 && !spectator) {
                final int pWidth = Math.round(progress.progress() * 13f / progress.maxProgress());
                final int pColor = Mth.hsvToRgb((float) progress.progress() / progress.maxProgress() / 3f, 1f, 1f);
                final int pX = slotX + 2;
                final int pY = slotY + 13;
                graphics.fill(pX, pY, pX + 13, pY + 2, 0xff000000);
                graphics.fill(pX, pY, pX + pWidth, pY + 1, pColor | 0xff000000);
            }
        }

        if (BingoMousePos.hasSlotPos(mousePos)) {
            final Vec2i slotPos = mousePos.getSlotPos(clientGame);
            final int slotX = slotPos.x() * 18 + 8;
            final int slotY = slotPos.y() * 18 + 18;
            graphics.blitSprite(RenderPipelines.GUI_TEXTURED, SLOT_HIGHLIGHT_FRONT_SPRITE, slotX - 4, slotY - 4, 24, 24);
        }

        if (!clientGame.nerfedTeams().and(clientTeam)) {
            for (int goalIndex = 0; goalIndex < goalCount; goalIndex++) {
                if (!clientGame.shape().isNerfCell(clientGame.size(), goalIndex)) {
                    continue;
                }

                final Vec2i slotPos = clientGame.shape().getCoords(clientGame.size(), goalIndex);
                final int slotX = slotPos.x() * 18 + 8;
                final int slotY = slotPos.y() * 18 + 18;

                graphics.fill(slotX, slotY, slotX + 16, slotY + 16, 0x80000000);
            }
        }

        graphics.pose().popMatrix();
        if (BingoMousePos.hasSlotPos(mousePos)) {
            final var goal = clientGame.goals()[mousePos.goalIndex()];
            final GoalProgress progress = clientGame.progress()[mousePos.goalIndex()];
            final TooltipBuilder tooltip = new TooltipBuilder();
            tooltip.add(goal.name());
            if (progress != null && (progress.maxProgress() > 1 || minecraft.options.advancedItemTooltips)) {
                tooltip.add(Component.translatable("bingo.progress", progress.progress(), progress.maxProgress()));
            }
            if (minecraft.options.advancedItemTooltips) {
                tooltip.add(Component.literal(goal.id().toString()).withStyle(ChatFormatting.DARK_GRAY));
            }
            goal.tooltip().ifPresent(component -> {
                final int width = Math.max(300, minecraft.font.width(goal.name()));
                tooltip.add(FormattedCharSequence.EMPTY);
                minecraft.font.split(component, width).forEach(tooltip::add);
            });
            goal.tooltipIcon().map(IconTooltip::new).ifPresent(tooltip::add);
            tooltip.draw(minecraft.font, graphics, (int) mousePos.mouseX(), (int) mousePos.mouseY());
        }
    }

    private static void renderBoardTitle(GuiGraphics graphics, Font font) {
        final int maxWidth = getBoardWidth() - 16;
        final FormattedCharSequence title = font.width(BOARD_TITLE) > maxWidth
            ? getVisualOrderWithEllipses(BOARD_TITLE_SHORT, font, maxWidth)
            : BOARD_TITLE.getVisualOrderText();
        graphics.drawString(font, title, 8, 6, 0xff404040, false);
    }

    public static FormattedCharSequence getVisualOrderWithEllipses(Component text, Font font, int maxWidth) {
        final int textWidth = font.width(text);
        if (textWidth <= maxWidth) {
            return text.getVisualOrderText();
        }
        final FormattedText shortText = font.substrByWidth(text, maxWidth - font.width(CommonComponents.ELLIPSIS));
        final FormattedText combinedText = FormattedText.composite(shortText, CommonComponents.ELLIPSIS);
        return Language.getInstance().getVisualOrder(combinedText);
    }

    public static boolean detectClick(int button, PositionAndScale boardPos) {
        return detectClickOrPress(InputConstants.Type.MOUSE.getOrCreate(button), boardPos);
    }

    public static boolean detectPress(KeyEvent event, PositionAndScale boardPos) {
        return detectClickOrPress(InputConstants.getKey(event), boardPos);
    }

    public static boolean detectClickOrPress(InputConstants.Key key, PositionAndScale boardPos) {
        if (clientGame == null) {
            return false;
        }

        final BingoMousePos mousePos = BingoMousePos.getPos(Minecraft.getInstance(), clientGame.shape(), clientGame.size(), boardPos);
        if (!mousePos.hasSlotPos()) {
            return false;
        }

        if (key.equals(manualHighlightKeyMapping.key)) {
            Integer manualHighlight = clientGame.manualHighlights()[mousePos.goalIndex()];
            Integer nextHighlight = switch (manualHighlight) {
                case null -> 0;
                case BingoBoard.NUM_MANUAL_HIGHLIGHT_COLORS - 1 -> null;
                default -> manualHighlight + 1;
            };
            clientGame.manualHighlights()[mousePos.goalIndex()] = nextHighlight;
            new ManualHighlightPayload(mousePos.goalIndex(), nextHighlight == null ? 0 : nextHighlight + 1, clientGame.manualHighlightModCount().getValue())
                .sendToServer();
            clientGame.manualHighlightModCount().increment();
            AbstractButton.playButtonClickSound(Minecraft.getInstance().getSoundManager());
        }

        final var goal = clientGame.goals()[mousePos.goalIndex()];

        final RecipeViewerPlugin plugin = getRecipeViewerPlugin();
        if (plugin.isViewRecipe(key)) {
            plugin.showRecipe(IconRenderers.getRenderer(goal.icon()).getIconItem(goal.icon()));
            return true;
        }
        if (plugin.isViewUsages(key)) {
            plugin.showUsages(IconRenderers.getRenderer(goal.icon()).getIconItem(goal.icon()));
            return true;
        }
        return false;
    }

    public static Component getDisplayName(PlayerTeam team) {
        final ClientPacketListener connection = Minecraft.getInstance().getConnection();
        if (connection != null) {
            final Iterator<PlayerInfo> players = team.getPlayers()
                .stream()
                .map(connection::getPlayerInfo)
                .filter(Objects::nonNull)
                .iterator();
            if (players.hasNext()) {
                final PlayerInfo playerInfo = players.next();
                if (!players.hasNext()) {
                    final ClientLevel level = Minecraft.getInstance().level;
                    if (level != null) {
                        final Player player = level.getPlayerByUUID(playerInfo.getProfile().id());
                        if (player != null) {
                            return player.getName();
                        }
                    }
                    return Component.literal(playerInfo.getProfile().name());
                }
            }
        }
        return team.getDisplayName();
    }
}
