#version 150

/*
 * Volumetric Cloud Fragment Shader
 * Raymarches through cloud layers for realistic sky
 * 
 * Optimizations:
 * - Adaptive step size based on density
 * - Early ray termination
 * - LOD based on distance
 * - Half-resolution rendering support
 */

uniform vec3 uCameraPos;
uniform mat4 uInvViewProjMat;
uniform float uTime;
uniform vec3 uSunDirection;
uniform float uSunIntensity;
uniform vec2 uScreenSize;
uniform float uCloudCoverage;       // 0-1, how much of sky is covered
uniform float uCloudAltitude;       // Base altitude of clouds
uniform float uCloudThickness;      // Vertical thickness of cloud layer
uniform vec3 uWindDirection;        // Wind movement
uniform float uWindSpeed;

// Storm cloud uniforms (when tornado is active)
uniform int uStormActive;
uniform vec3 uStormCenter;          // Tornado position
uniform float uStormRadius;         // Storm cloud radius around tornado

in vec2 texCoord;
out vec4 fragColor;

// ============== NOISE FUNCTIONS ==============

vec3 hash33(vec3 p) {
    p = vec3(dot(p, vec3(127.1, 311.7, 74.7)),
             dot(p, vec3(269.5, 183.3, 246.1)),
             dot(p, vec3(113.5, 271.9, 124.6)));
    return fract(sin(p) * 43758.5453123);
}

// Worley noise for fluffy cloud shapes
float worley(vec3 p) {
    vec3 n = floor(p);
    vec3 f = fract(p);
    float minDist = 1.0;
    
    for (int x = -1; x <= 1; x++) {
        for (int y = -1; y <= 1; y++) {
            for (int z = -1; z <= 1; z++) {
                vec3 neighbor = vec3(float(x), float(y), float(z));
                vec3 point = hash33(n + neighbor);
                vec3 diff = neighbor + point - f;
                minDist = min(minDist, dot(diff, diff));
            }
        }
    }
    return sqrt(minDist);
}

// Inverted worley for fluffy appearance
float worleyFBM(vec3 p) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;
    
    for (int i = 0; i < 4; i++) {
        value += (1.0 - worley(p * frequency)) * amplitude;
        amplitude *= 0.5;
        frequency *= 2.0;
    }
    return value;
}

// Simplex-like noise for variation
vec4 permute(vec4 x) { return mod(((x*34.0)+1.0)*x, 289.0); }
vec4 taylorInvSqrt(vec4 r) { return 1.79284291400159 - 0.85373472095314 * r; }

float snoise(vec3 v) {
    const vec2 C = vec2(1.0/6.0, 1.0/3.0);
    vec3 i = floor(v + dot(v, C.yyy));
    vec3 x0 = v - i + dot(i, C.xxx);
    vec3 g = step(x0.yzx, x0.xyz);
    vec3 l = 1.0 - g;
    vec3 i1 = min(g.xyz, l.zxy);
    vec3 i2 = max(g.xyz, l.zxy);
    vec3 x1 = x0 - i1 + C.xxx;
    vec3 x2 = x0 - i2 + C.yyy;
    vec3 x3 = x0 - 0.5;
    i = mod(i, 289.0);
    vec4 p = permute(permute(permute(
              i.z + vec4(0.0, i1.z, i2.z, 1.0))
            + i.y + vec4(0.0, i1.y, i2.y, 1.0))
            + i.x + vec4(0.0, i1.x, i2.x, 1.0));
    float n_ = 0.142857142857;
    vec3 ns = n_ * vec3(2.0, 0.5, 1.0) - vec3(0.0, 1.0, 0.0);
    vec4 j = p - 49.0 * floor(p * ns.z * ns.z);
    vec4 x_ = floor(j * ns.z);
    vec4 y_ = floor(j - 7.0 * x_);
    vec4 x = x_ * ns.x + ns.yyyy;
    vec4 y = y_ * ns.x + ns.yyyy;
    vec4 h = 1.0 - abs(x) - abs(y);
    vec4 b0 = vec4(x.xy, y.xy);
    vec4 b1 = vec4(x.zw, y.zw);
    vec4 s0 = floor(b0) * 2.0 + 1.0;
    vec4 s1 = floor(b1) * 2.0 + 1.0;
    vec4 sh = -step(h, vec4(0.0));
    vec4 a0 = b0.xzyw + s0.xzyw * sh.xxyy;
    vec4 a1 = b1.xzyw + s1.xzyw * sh.zzww;
    vec3 p0 = vec3(a0.xy, h.x);
    vec3 p1 = vec3(a0.zw, h.y);
    vec3 p2 = vec3(a1.xy, h.z);
    vec3 p3 = vec3(a1.zw, h.w);
    vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2,p2), dot(p3,p3)));
    p0 *= norm.x; p1 *= norm.y; p2 *= norm.z; p3 *= norm.w;
    vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);
    m = m * m;
    return 42.0 * dot(m*m, vec4(dot(p0,x0), dot(p1,x1), dot(p2,x2), dot(p3,x3)));
}

// ============== CLOUD DENSITY ==============

float getCloudDensity(vec3 worldPos) {
    // Cloud layer bounds
    float cloudBase = uCloudAltitude;
    float cloudTop = uCloudAltitude + uCloudThickness;
    
    // Outside cloud layer
    if (worldPos.y < cloudBase || worldPos.y > cloudTop) {
        return 0.0;
    }
    
    // Height within cloud layer (0 at base, 1 at top)
    float heightFraction = (worldPos.y - cloudBase) / uCloudThickness;
    
    // Vertical density profile - denser in middle, fade at edges
    float heightDensity = smoothstep(0.0, 0.2, heightFraction) * 
                          smoothstep(1.0, 0.7, heightFraction);
    
    // Animate with wind
    vec3 windOffset = uWindDirection * uWindSpeed * uTime * 0.01;
    vec3 samplePos = worldPos * 0.008 + windOffset;
    
    // Multi-scale noise for cloud shape
    float worleyNoise = worleyFBM(samplePos * 2.0);
    float detailNoise = snoise(samplePos * 8.0) * 0.3 + 0.5;
    
    // Combine noises
    float cloudShape = worleyNoise * detailNoise;
    
    // Apply coverage threshold
    float coverageThreshold = 1.0 - uCloudCoverage;
    cloudShape = smoothstep(coverageThreshold, coverageThreshold + 0.3, cloudShape);
    
    // Final density
    float density = cloudShape * heightDensity;
    
    // Storm clouds - darker and denser near tornado
    if (uStormActive > 0) {
        float distToStorm = length(worldPos.xz - uStormCenter.xz);
        float stormInfluence = 1.0 - smoothstep(0.0, uStormRadius, distToStorm);
        
        if (stormInfluence > 0.0) {
            // Add extra density and lower the cloud base near storm
            float stormDensity = stormInfluence * 0.5;
            
            // Storm clouds extend lower
            float stormBase = cloudBase - stormInfluence * 30.0;
            if (worldPos.y > stormBase && worldPos.y < cloudBase) {
                float stormHeight = (worldPos.y - stormBase) / (cloudBase - stormBase);
                stormDensity *= smoothstep(0.0, 0.5, stormHeight);
            }
            
            density = max(density, stormDensity);
            
            // Make storm clouds more turbulent
            float turbulence = snoise(samplePos * 4.0 + vec3(uTime * 0.1)) * stormInfluence;
            density += turbulence * 0.2;
        }
    }
    
    return clamp(density, 0.0, 1.0);
}

// ============== LIGHTING ==============

vec3 getCloudLighting(vec3 pos, float density, vec3 rayDir) {
    // Base cloud color
    vec3 brightColor = vec3(1.0, 1.0, 1.0);
    vec3 darkColor = vec3(0.4, 0.45, 0.5);
    
    // Sun lighting
    float sunDot = max(dot(uSunDirection, vec3(0, 1, 0)), 0.0);
    float sunLight = sunDot * uSunIntensity;
    
    // Height-based lighting (higher = brighter)
    float heightFactor = (pos.y - uCloudAltitude) / uCloudThickness;
    heightFactor = clamp(heightFactor, 0.0, 1.0);
    
    // Mix colors based on density and height
    vec3 cloudColor = mix(brightColor, darkColor, density * 0.5);
    cloudColor = mix(darkColor * 0.7, cloudColor, heightFactor);
    
    // Sun tint
    vec3 sunColor = vec3(1.0, 0.95, 0.8);
    cloudColor = mix(cloudColor, cloudColor * sunColor, sunLight * 0.3);
    
    // Storm darkening
    if (uStormActive > 0) {
        float distToStorm = length(pos.xz - uStormCenter.xz);
        float stormDark = 1.0 - smoothstep(0.0, uStormRadius * 1.5, distToStorm);
        cloudColor = mix(cloudColor, vec3(0.2, 0.22, 0.25), stormDark * 0.7);
    }
    
    return cloudColor;
}

// ============== RAYMARCHING ==============

vec4 raymarchClouds(vec3 rayOrigin, vec3 rayDir) {
    // Cloud layer intersection
    float cloudBase = uCloudAltitude;
    float cloudTop = uCloudAltitude + uCloudThickness;
    
    // Storm extends cloud base lower
    if (uStormActive > 0) {
        cloudBase -= 30.0;
    }
    
    // Calculate intersection with cloud layer
    float tBase = (cloudBase - rayOrigin.y) / rayDir.y;
    float tTop = (cloudTop - rayOrigin.y) / rayDir.y;
    
    float tNear = min(tBase, tTop);
    float tFar = max(tBase, tTop);
    
    // Skip if behind camera or looking away
    if (tFar < 0.0) return vec4(0.0);
    tNear = max(tNear, 0.0);
    
    // Limit ray length for performance
    tFar = min(tFar, 5000.0);
    
    if (tNear > tFar) return vec4(0.0);
    
    // Adaptive step count based on distance
    float rayLength = tFar - tNear;
    int maxSteps = 32;
    
    // Reduce steps for distant clouds
    float distFactor = 1.0 - smoothstep(500.0, 2000.0, tNear);
    maxSteps = int(float(maxSteps) * max(distFactor, 0.3));
    
    float stepSize = rayLength / float(maxSteps);
    stepSize = max(stepSize, 5.0); // Minimum step size
    
    vec4 accumulated = vec4(0.0);
    float t = tNear;
    
    for (int i = 0; i < 32; i++) {
        if (i >= maxSteps || accumulated.a > 0.95 || t > tFar) break;
        
        vec3 pos = rayOrigin + rayDir * t;
        
        float density = getCloudDensity(pos);
        
        if (density > 0.01) {
            vec3 color = getCloudLighting(pos, density, rayDir);
            
            // Beer's law absorption
            float alpha = density * stepSize * 0.02;
            alpha = clamp(alpha, 0.0, 1.0);
            
            // Front-to-back compositing
            color *= alpha;
            accumulated.rgb += (1.0 - accumulated.a) * color;
            accumulated.a += (1.0 - accumulated.a) * alpha;
            
            // Adaptive stepping - smaller steps in dense regions
            t += stepSize * (1.0 - density * 0.5);
        } else {
            // Large steps in empty space
            t += stepSize * 2.0;
        }
    }
    
    return accumulated;
}

// ============== MAIN ==============

void main() {
    // Reconstruct ray from screen coordinates
    vec2 ndc = texCoord * 2.0 - 1.0;
    
    vec4 nearClip = vec4(ndc, -1.0, 1.0);
    vec4 farClip = vec4(ndc, 1.0, 1.0);
    
    vec4 nearWorld4 = uInvViewProjMat * nearClip;
    vec4 farWorld4 = uInvViewProjMat * farClip;
    
    vec3 nearWorld = nearWorld4.xyz / nearWorld4.w;
    vec3 farWorld = farWorld4.xyz / farWorld4.w;
    
    vec3 rayOrigin = uCameraPos;
    vec3 rayDir = normalize(farWorld - nearWorld);
    
    // Only render clouds above horizon
    if (rayDir.y < 0.0) {
        discard;
    }
    
    // Raymarch clouds
    vec4 result = raymarchClouds(rayOrigin, rayDir);
    
    if (result.a < 0.001) {
        discard;
    }
    
    fragColor = result;
}

