#version 150

/*
 * Tornado Volumetric Fragment Shader
 * Uses raymarching with SDF to render realistic tornado funnel
 * 
 * The core concept: Instead of rendering triangles, we render "math"
 * - getDensity(x,y,z) returns how much "tornado" exists at that point
 * - 1.0 = solid tornado, 0.0 = air
 * - We raymarch through the volume, accumulating density
 */

// Uniforms passed from Java
uniform vec3 uTornadoPosition;      // World position of tornado entity
uniform float uTime;                 // Animation time (tickCount + partialTicks)
uniform vec3 uSunDirection;          // For lighting
uniform float uSunIntensity;         // Sun brightness
uniform vec3 uCameraPos;             // Camera world position
uniform mat4 uInvViewProjMat;        // Inverse view-projection matrix
uniform float uTier;                 // Tornado tier (1-5)
uniform vec2 uScreenSize;            // Screen resolution
uniform float uNearPlane;            // Camera near plane
uniform float uFarPlane;             // Camera far plane
uniform vec3 uBoxSize;               // Bounding box size
uniform vec3 uBoxCenter;             // Bounding box center

// Tier-based parameters (matching Java)
const float TORNADO_HEIGHT[5] = float[5](25.0, 40.0, 60.0, 85.0, 120.0);
const float MAX_RADIUS[5] = float[5](8.0, 12.0, 18.0, 26.0, 38.0);
const float BASE_RADIUS[5] = float[5](2.0, 2.5, 3.0, 4.0, 5.0);

in vec2 texCoord;
in vec3 worldPos;
in vec3 viewRayDir;
out vec4 fragColor;

// ============== NOISE FUNCTIONS ==============

vec3 hash33(vec3 p) {
    p = vec3(dot(p, vec3(127.1, 311.7, 74.7)),
             dot(p, vec3(269.5, 183.3, 246.1)),
             dot(p, vec3(113.5, 271.9, 124.6)));
    return fract(sin(p) * 43758.5453123);
}

vec4 permute(vec4 x) { return mod(((x*34.0)+1.0)*x, 289.0); }
vec4 taylorInvSqrt(vec4 r) { return 1.79284291400159 - 0.85373472095314 * r; }

float snoise(vec3 v) {
    const vec2 C = vec2(1.0/6.0, 1.0/3.0);
    vec3 i = floor(v + dot(v, C.yyy));
    vec3 x0 = v - i + dot(i, C.xxx);
    vec3 g = step(x0.yzx, x0.xyz);
    vec3 l = 1.0 - g;
    vec3 i1 = min(g.xyz, l.zxy);
    vec3 i2 = max(g.xyz, l.zxy);
    vec3 x1 = x0 - i1 + C.xxx;
    vec3 x2 = x0 - i2 + C.yyy;
    vec3 x3 = x0 - 0.5;
    i = mod(i, 289.0);
    vec4 p = permute(permute(permute(
              i.z + vec4(0.0, i1.z, i2.z, 1.0))
            + i.y + vec4(0.0, i1.y, i2.y, 1.0))
            + i.x + vec4(0.0, i1.x, i2.x, 1.0));
    float n_ = 0.142857142857;
    vec3 ns = n_ * vec3(2.0, 0.5, 1.0) - vec3(0.0, 1.0, 0.0);
    vec4 j = p - 49.0 * floor(p * ns.z * ns.z);
    vec4 x_ = floor(j * ns.z);
    vec4 y_ = floor(j - 7.0 * x_);
    vec4 x = x_ * ns.x + ns.yyyy;
    vec4 y = y_ * ns.x + ns.yyyy;
    vec4 h = 1.0 - abs(x) - abs(y);
    vec4 b0 = vec4(x.xy, y.xy);
    vec4 b1 = vec4(x.zw, y.zw);
    vec4 s0 = floor(b0) * 2.0 + 1.0;
    vec4 s1 = floor(b1) * 2.0 + 1.0;
    vec4 sh = -step(h, vec4(0.0));
    vec4 a0 = b0.xzyw + s0.xzyw * sh.xxyy;
    vec4 a1 = b1.xzyw + s1.xzyw * sh.zzww;
    vec3 p0 = vec3(a0.xy, h.x);
    vec3 p1 = vec3(a0.zw, h.y);
    vec3 p2 = vec3(a1.xy, h.z);
    vec3 p3 = vec3(a1.zw, h.w);
    vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2,p2), dot(p3,p3)));
    p0 *= norm.x; p1 *= norm.y; p2 *= norm.z; p3 *= norm.w;
    vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);
    m = m * m;
    return 42.0 * dot(m*m, vec4(dot(p0,x0), dot(p1,x1), dot(p2,x2), dot(p3,x3)));
}

// FBM for detailed turbulence
float fbm(vec3 p, int octaves) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;
    for (int i = 0; i < octaves; i++) {
        value += amplitude * snoise(p * frequency);
        amplitude *= 0.5;
        frequency *= 2.0;
    }
    return value;
}

// Worley noise for cloud structures
float worley(vec3 p) {
    vec3 n = floor(p);
    vec3 f = fract(p);
    float minDist = 1.0;
    for (int x = -1; x <= 1; x++) {
        for (int y = -1; y <= 1; y++) {
            for (int z = -1; z <= 1; z++) {
                vec3 neighbor = vec3(float(x), float(y), float(z));
                vec3 point = hash33(n + neighbor);
                vec3 diff = neighbor + point - f;
                minDist = min(minDist, dot(diff, diff));
            }
        }
    }
    return sqrt(minDist);
}

// ============== SDF FUNCTIONS ==============

// Tornado funnel SDF with tier-based sizing
float sdTornadoFunnel(vec3 p, int tier) {
    float height = TORNADO_HEIGHT[tier];
    float baseR = BASE_RADIUS[tier];
    float topR = MAX_RADIUS[tier];
    
    // Progress along height
    float t = clamp(p.y / height, 0.0, 1.0);
    
    // Exponential funnel expansion
    float radius = mix(baseR, topR, pow(t, 2.2));
    
    // Add swirl distortion
    float angle = atan(p.z, p.x);
    float swirlOffset = sin(angle * 3.0 + uTime * 2.0 + p.y * 0.15) * (0.5 + t * 1.5);
    radius += swirlOffset;
    
    // Distance from axis
    float r = length(p.xz);
    
    float dist = r - radius;
    
    // Height bounds
    if (p.y < 0.0) dist = max(dist, -p.y);
    if (p.y > height) dist = max(dist, p.y - height);
    
    return dist;
}

// ============== DENSITY FUNCTION ==============
// This is the heart of the shader - defines "how much tornado" at any point

float getDensity(vec3 worldPos, int tier) {
    // Transform to tornado-local space
    vec3 localPos = worldPos - uTornadoPosition;
    
    float height = TORNADO_HEIGHT[tier];
    float baseR = BASE_RADIUS[tier];
    float topR = MAX_RADIUS[tier];
    
    // Height progress (0 = ground, 1 = top)
    float heightProgress = clamp(localPos.y / height, 0.0, 1.0);
    
    // Quick reject if way outside
    if (localPos.y < -5.0 || localPos.y > height + 10.0) return 0.0;
    
    // Calculate funnel radius at this height
    float funnelRadius = mix(baseR, topR, pow(heightProgress, 2.2));
    
    // Distance from center axis
    float distFromAxis = length(localPos.xz);
    
    // Apply spiral rotation to sample position
    float spiralAngle = uTime * 3.0 - localPos.y * 0.2;
    vec2 rotatedXZ = vec2(
        localPos.x * cos(spiralAngle) - localPos.z * sin(spiralAngle),
        localPos.x * sin(spiralAngle) + localPos.z * cos(spiralAngle)
    );
    vec3 spiralPos = vec3(rotatedXZ.x, localPos.y, rotatedXZ.y);
    
    // Shell thickness varies with height (thicker at top)
    float shellThickness = 1.5 + heightProgress * 4.0 + float(tier) * 0.5;
    
    // Distance to funnel shell
    float shellDist = abs(distFromAxis - funnelRadius);
    
    // Base density from shell distance
    float density = 1.0 - smoothstep(0.0, shellThickness, shellDist);
    
    // Add turbulent detail noise
    vec3 noisePos = spiralPos * 0.15 + vec3(0.0, uTime * 0.8, 0.0);
    float turbulence = fbm(noisePos, 4) * 0.5 + 0.5;
    
    // Add fine detail noise
    float fineDetail = snoise(spiralPos * 0.4 + vec3(uTime * 0.3)) * 0.3;
    
    // Modulate density with noise
    density *= turbulence * 0.7 + 0.3;
    density += fineDetail * density;
    
    // Core is more dense
    float coreDist = distFromAxis / funnelRadius;
    if (coreDist < 0.3) {
        density += (0.3 - coreDist) * 2.0;
    }
    
    // Fade at top and bottom
    density *= smoothstep(0.0, 0.1, heightProgress);
    density *= smoothstep(1.0, 0.85, heightProgress);
    
    // Add debris/dust at bottom
    if (heightProgress < 0.2) {
        float dustNoise = fbm(localPos * 0.1 + vec3(uTime * 0.2), 3);
        density += dustNoise * 0.5 * (0.2 - heightProgress) * 5.0;
    }
    
    return clamp(density, 0.0, 1.0);
}

// ============== CLOUD DENSITY ==============

float getCloudDensity(vec3 worldPos) {
    // Cloud layer at tornado top
    float cloudBase = uTornadoPosition.y + TORNADO_HEIGHT[int(uTier) - 1] * 0.8;
    float cloudTop = cloudBase + 30.0;
    
    if (worldPos.y < cloudBase - 5.0 || worldPos.y > cloudTop + 5.0) return 0.0;
    
    float heightInCloud = (worldPos.y - cloudBase) / (cloudTop - cloudBase);
    
    // Use worley noise for fluffy cloud look
    vec3 cloudPos = worldPos * 0.03 + vec3(uTime * 0.02, 0.0, uTime * 0.01);
    float cloudNoise = 1.0 - worley(cloudPos);
    cloudNoise += fbm(worldPos * 0.05, 3) * 0.5;
    
    // Shape cloud layer
    float density = cloudNoise - 0.3;
    density *= smoothstep(0.0, 0.3, heightInCloud);
    density *= smoothstep(1.0, 0.7, heightInCloud);
    
    return clamp(density, 0.0, 0.8);
}

// ============== LIGHTING ==============

vec3 calculateLighting(vec3 pos, float density, vec3 rayDir) {
    // Base tornado color (dark gray to black)
    vec3 baseColor = vec3(0.15, 0.15, 0.17);
    
    // Lighter color for edges (atmospheric scattering)
    vec3 edgeColor = vec3(0.4, 0.42, 0.45);
    
    // Mix based on density
    vec3 color = mix(edgeColor, baseColor, density);
    
    // Simple directional lighting from sun
    float sunLight = max(dot(normalize(vec3(0.0, 1.0, 0.0)), uSunDirection), 0.0);
    color += sunLight * uSunIntensity * vec3(0.1, 0.1, 0.08);
    
    // Ambient occlusion approximation (denser = darker)
    color *= 1.0 - density * 0.3;
    
    // Height-based color variation
    float height = (pos.y - uTornadoPosition.y) / TORNADO_HEIGHT[int(uTier) - 1];
    color = mix(color * 0.7, color, height); // Darker at base
    
    return color;
}

// ============== EDGE FADE ==============
// Smooth fade near bounding box edges for seamless blending

float getEdgeFade(vec3 pos) {
    // Calculate distance from box center in normalized coordinates
    vec3 boxHalfSize = uBoxSize * 0.5;
    vec3 relPos = pos - uBoxCenter;
    
    // Normalized distance from center (0 at center, 1 at edge)
    vec3 normDist = abs(relPos) / boxHalfSize;
    
    // Fade starts at 70% distance from center
    float fadeStart = 0.7;
    float fadeEnd = 1.0;
    
    // Calculate fade for each axis
    float fadeX = 1.0 - smoothstep(fadeStart, fadeEnd, normDist.x);
    float fadeY = 1.0 - smoothstep(fadeStart, fadeEnd, normDist.y);
    float fadeZ = 1.0 - smoothstep(fadeStart, fadeEnd, normDist.z);
    
    // Combine fades (multiply for smooth corner falloff)
    return fadeX * fadeY * fadeZ;
}

// ============== RAYMARCHING ==============

vec4 raymarch(vec3 rayOrigin, vec3 rayDir) {
    int tier = int(uTier) - 1;
    tier = clamp(tier, 0, 4);
    
    float height = TORNADO_HEIGHT[tier];
    float maxRadius = MAX_RADIUS[tier] + 20.0; // Padding for effects
    
    // Calculate bounding box intersection using the actual box bounds
    vec3 boxHalfSize = uBoxSize * 0.5;
    vec3 boxMin = uBoxCenter - boxHalfSize;
    vec3 boxMax = uBoxCenter + boxHalfSize;
    
    // Ray-box intersection
    vec3 invDir = 1.0 / rayDir;
    vec3 t0 = (boxMin - rayOrigin) * invDir;
    vec3 t1 = (boxMax - rayOrigin) * invDir;
    vec3 tMin = min(t0, t1);
    vec3 tMax = max(t0, t1);
    float tNear = max(max(tMin.x, tMin.y), tMin.z);
    float tFar = min(min(tMax.x, tMax.y), tMax.z);
    
    // No intersection with bounding box
    if (tNear > tFar || tFar < 0.0) {
        return vec4(0.0);
    }
    
    tNear = max(tNear, 0.0);
    
    // Raymarching parameters
    int maxSteps = 64;
    float stepSize = (tFar - tNear) / float(maxSteps);
    stepSize = max(stepSize, 0.5); // Minimum step for performance
    
    vec4 accumulated = vec4(0.0);
    float t = tNear;
    
    for (int i = 0; i < maxSteps; i++) {
        if (accumulated.a > 0.95 || t > tFar) break;
        
        vec3 pos = rayOrigin + rayDir * t;
        
        // Get edge fade factor for seamless blending
        float edgeFade = getEdgeFade(pos);
        
        // Skip if fully faded at edges
        if (edgeFade < 0.01) {
            t += stepSize;
            continue;
        }
        
        // Sample tornado density
        float density = getDensity(pos, tier);
        
        // Sample clouds
        float cloudDensity = getCloudDensity(pos);
        
        // Combine densities and apply edge fade
        float totalDensity = (density + cloudDensity * 0.5) * edgeFade;
        
        if (totalDensity > 0.01) {
            // Calculate color with lighting
            vec3 color = calculateLighting(pos, totalDensity, rayDir);
            
            // Clouds are lighter
            if (cloudDensity > density) {
                color = mix(color, vec3(0.7, 0.72, 0.75), cloudDensity);
            }
            
            // Accumulate with front-to-back blending
            float alpha = totalDensity * stepSize * 0.5;
            alpha = clamp(alpha, 0.0, 1.0);
            
            color *= alpha;
            accumulated.rgb += (1.0 - accumulated.a) * color;
            accumulated.a += (1.0 - accumulated.a) * alpha;
        }
        
        t += stepSize;
    }
    
    return accumulated;
}

// ============== MAIN ==============

void main() {
    // Ray origin is camera position
    vec3 rayOrigin = uCameraPos;
    
    // Ray direction from camera through this fragment's world position
    vec3 rayDir = normalize(worldPos - uCameraPos);
    
    // Perform raymarching
    vec4 result = raymarch(rayOrigin, rayDir);
    
    // Discard fully transparent pixels for better blending
    if (result.a < 0.001) {
        discard;
    }
    
    // Output with premultiplied alpha for proper blending
    fragColor = vec4(result.rgb, result.a);
}

