// Signed Distance Functions for volumetric shapes

// Cone SDF - perfect for tornado funnel
// c is the sin/cos of the angle, h is height
float sdCone(vec3 p, vec2 c, float h) {
    vec2 q = h * vec2(c.x / c.y, -1.0);
    vec2 w = vec2(length(p.xz), p.y);
    vec2 a = w - q * clamp(dot(w, q) / dot(q, q), 0.0, 1.0);
    vec2 b = w - q * vec2(clamp(w.x / q.x, 0.0, 1.0), 1.0);
    float k = sign(q.y);
    float d = min(dot(a, a), dot(b, b));
    float s = max(k * (w.x * q.y - w.y * q.x), k * (w.y - q.y));
    return sqrt(d) * sign(s);
}

// Infinite cone
float sdInfCone(vec3 p, vec2 c) {
    vec2 q = vec2(length(p.xz), -p.y);
    float d = length(q - c * max(dot(q, c), 0.0));
    return d * ((q.x * c.y - q.y * c.x < 0.0) ? -1.0 : 1.0);
}

// Tornado funnel shape - narrow at bottom, wide at top
float sdTornadoFunnel(vec3 p, float baseRadius, float topRadius, float height) {
    // Flip Y so bottom is at origin
    float y = p.y;
    
    // Progress along height (0 at bottom, 1 at top)
    float t = clamp(y / height, 0.0, 1.0);
    
    // Exponential radius expansion - characteristic tornado shape
    float radius = mix(baseRadius, topRadius, pow(t, 2.2));
    
    // Distance from central axis
    float r = length(p.xz);
    
    // Signed distance to funnel surface
    float dist = r - radius;
    
    // Clamp to height bounds
    if (y < 0.0) {
        dist = max(dist, -y);
    }
    if (y > height) {
        dist = max(dist, y - height);
    }
    
    return dist;
}

// Cylinder SDF for debris column
float sdCylinder(vec3 p, float r, float h) {
    vec2 d = abs(vec2(length(p.xz), p.y)) - vec2(r, h);
    return min(max(d.x, d.y), 0.0) + length(max(d, 0.0));
}

// Sphere SDF
float sdSphere(vec3 p, float r) {
    return length(p) - r;
}

// Box SDF for bounding optimization
float sdBox(vec3 p, vec3 b) {
    vec3 q = abs(p) - b;
    return length(max(q, 0.0)) + min(max(q.x, max(q.y, q.z)), 0.0);
}

// Smooth union for combining shapes
float smin(float a, float b, float k) {
    float h = clamp(0.5 + 0.5 * (b - a) / k, 0.0, 1.0);
    return mix(b, a, h) - k * h * (1.0 - h);
}

// Smooth subtraction
float smax(float a, float b, float k) {
    return -smin(-a, -b, k);
}

