#version 150

/*
 * Cloud Upscale and Temporal Blend Fragment Shader
 * 
 * Takes the half-resolution cloud render and:
 * 1. Upscales with bilinear filtering
 * 2. Blends with previous frame for temporal stability
 * 3. Edge-aware filtering to reduce artifacts
 */

uniform sampler2D uCurrentFrame;      // Current half-res cloud render
uniform sampler2D uPreviousFrame;     // Previous frame for temporal blend
uniform sampler2D uDepthTexture;      // Scene depth for edge detection
uniform vec2 uScreenSize;             // Full resolution screen size
uniform vec2 uCloudSize;              // Half-res cloud texture size
uniform float uBlendFactor;           // Temporal blend amount (0-1)
uniform int uHasPreviousFrame;        // Whether previous frame is valid

in vec2 texCoord;
out vec4 fragColor;

// Catmull-Rom bicubic sampling for sharper upscale
vec4 sampleBicubic(sampler2D tex, vec2 uv, vec2 texSize) {
    vec2 texelSize = 1.0 / texSize;
    vec2 samplePos = uv * texSize - 0.5;
    vec2 f = fract(samplePos);
    vec2 i = floor(samplePos);
    
    // Catmull-Rom weights
    vec2 w0 = f * (-0.5 + f * (1.0 - 0.5 * f));
    vec2 w1 = 1.0 + f * f * (-2.5 + 1.5 * f);
    vec2 w2 = f * (0.5 + f * (2.0 - 1.5 * f));
    vec2 w3 = f * f * (-0.5 + 0.5 * f);
    
    vec2 w12 = w1 + w2;
    vec2 tc12 = (i + 1.0 + w2 / w12) * texelSize;
    vec2 tc0 = (i - 0.5) * texelSize;
    vec2 tc3 = (i + 2.5) * texelSize;
    
    vec4 s0 = texture(tex, vec2(tc0.x, tc0.y));
    vec4 s1 = texture(tex, vec2(tc12.x, tc0.y));
    vec4 s2 = texture(tex, vec2(tc3.x, tc0.y));
    vec4 s3 = texture(tex, vec2(tc0.x, tc12.y));
    vec4 s4 = texture(tex, vec2(tc12.x, tc12.y));
    vec4 s5 = texture(tex, vec2(tc3.x, tc12.y));
    vec4 s6 = texture(tex, vec2(tc0.x, tc3.y));
    vec4 s7 = texture(tex, vec2(tc12.x, tc3.y));
    vec4 s8 = texture(tex, vec2(tc3.x, tc3.y));
    
    float sx = w12.x / (w0.x + w12.x + w3.x);
    float sy = w12.y / (w0.y + w12.y + w3.y);
    
    vec4 a = mix(mix(s0, s1, sx), mix(s1, s2, sx), sx);
    vec4 b = mix(mix(s3, s4, sx), mix(s4, s5, sx), sx);
    vec4 c = mix(mix(s6, s7, sx), mix(s7, s8, sx), sx);
    
    return mix(mix(a, b, sy), mix(b, c, sy), sy);
}

// Edge detection using depth buffer
float getEdgeFactor(vec2 uv) {
    vec2 texelSize = 1.0 / uScreenSize;
    
    float center = texture(uDepthTexture, uv).r;
    float left = texture(uDepthTexture, uv - vec2(texelSize.x, 0.0)).r;
    float right = texture(uDepthTexture, uv + vec2(texelSize.x, 0.0)).r;
    float up = texture(uDepthTexture, uv - vec2(0.0, texelSize.y)).r;
    float down = texture(uDepthTexture, uv + vec2(0.0, texelSize.y)).r;
    
    float dx = abs(left - center) + abs(right - center);
    float dy = abs(up - center) + abs(down - center);
    
    return clamp((dx + dy) * 100.0, 0.0, 1.0);
}

void main() {
    // Sample current frame with bicubic upscaling
    vec4 current = sampleBicubic(uCurrentFrame, texCoord, uCloudSize);
    
    // Edge detection - reduce temporal blending on edges
    float edge = getEdgeFactor(texCoord);
    
    // Temporal blending with previous frame
    vec4 result = current;
    
    if (uHasPreviousFrame > 0 && uBlendFactor > 0.0) {
        vec4 previous = texture(uPreviousFrame, texCoord);
        
        // Reduce blend on edges and areas with large changes
        float colorDiff = length(current.rgb - previous.rgb);
        float dynamicBlend = uBlendFactor * (1.0 - edge) * (1.0 - smoothstep(0.1, 0.5, colorDiff));
        
        // Exponential moving average for temporal stability
        result = mix(current, previous, dynamicBlend);
    }
    
    // Prevent alpha accumulation issues
    result.a = clamp(result.a, 0.0, 1.0);
    
    fragColor = result;
}

