#version 150

/*
 * Volumetric Cloud Fragment Shader - PMW-Style Realistic Clouds
 * 
 * Features:
 * - FBM (Fractal Brownian Motion) for multi-scale detail
 * - Worley Noise for fluffy cauliflower-like appearance
 * - Beer's Law for realistic light absorption
 * - Light scattering simulation (bright tops, dark bottoms)
 * - Adaptive ray stepping for performance
 * - Half-resolution rendering support
 * 
 * Based on GPU Pro techniques and Horizon Zero Dawn cloud rendering
 */

// ============== UNIFORMS ==============

uniform vec3 uCameraPos;
uniform mat4 uInvViewProjMat;
uniform mat4 uViewProjMat;
uniform float uTime;
uniform vec3 uSunDirection;
uniform float uSunIntensity;
uniform vec2 uScreenSize;
uniform float uCloudCoverage;       // 0-1, how much of sky is covered
uniform float uCloudAltitude;       // Base altitude of clouds
uniform float uCloudThickness;      // Vertical thickness of cloud layer
uniform vec3 uWindDirection;        // Wind movement
uniform float uWindSpeed;
uniform sampler2D uDepthTexture;    // Scene depth buffer

// Quality settings
uniform int uQualityLevel;          // 0=low, 1=medium, 2=high

// Camera direction vectors for reliable ray reconstruction
uniform vec3 uCameraLook;           // Camera forward direction
uniform vec3 uCameraUp;             // Camera up direction  
uniform vec3 uCameraRight;          // Camera right direction
uniform vec2 uFovScale;             // x = tan(fov/2), y = aspect ratio

// Storm cloud uniforms (when tornado is active)
uniform int uStormActive;
uniform vec3 uStormCenter;          // Tornado position
uniform float uStormRadius;         // Storm cloud radius around tornado

in vec2 texCoord;
out vec4 fragColor;

// ============== CONSTANTS ==============

const float PI = 3.14159265359;
const float CLOUD_DENSITY_MULTIPLIER = 1.5;
const float ABSORPTION_COEFFICIENT = 0.04;  // Beer's Law coefficient
const float SCATTERING_COEFFICIENT = 0.03;  // How much light scatters
const float SILVER_LINING_INTENSITY = 0.8;
const float POWDER_EFFECT = 2.0;            // Powder sugar effect strength

// Quality-based step counts
const int STEPS_LOW = 24;
const int STEPS_MEDIUM = 48;
const int STEPS_HIGH = 64;
const int LIGHT_STEPS = 6;  // Steps for light sampling

// ============== HASH FUNCTIONS ==============

vec3 hash33(vec3 p) {
    p = vec3(dot(p, vec3(127.1, 311.7, 74.7)),
             dot(p, vec3(269.5, 183.3, 246.1)),
             dot(p, vec3(113.5, 271.9, 124.6)));
    return fract(sin(p) * 43758.5453123);
}

float hash31(vec3 p) {
    return fract(sin(dot(p, vec3(127.1, 311.7, 74.7))) * 43758.5453123);
}

vec2 hash22(vec2 p) {
    p = vec2(dot(p, vec2(127.1, 311.7)), dot(p, vec2(269.5, 183.3)));
    return fract(sin(p) * 43758.5453123);
}

// ============== NOISE FUNCTIONS ==============

// Permutation helpers
vec4 permute(vec4 x) { return mod(((x*34.0)+1.0)*x, 289.0); }
vec4 taylorInvSqrt(vec4 r) { return 1.79284291400159 - 0.85373472095314 * r; }

// 3D Simplex Noise - fast and smooth
float snoise(vec3 v) {
    const vec2 C = vec2(1.0/6.0, 1.0/3.0);
    vec3 i = floor(v + dot(v, C.yyy));
    vec3 x0 = v - i + dot(i, C.xxx);
    vec3 g = step(x0.yzx, x0.xyz);
    vec3 l = 1.0 - g;
    vec3 i1 = min(g.xyz, l.zxy);
    vec3 i2 = max(g.xyz, l.zxy);
    vec3 x1 = x0 - i1 + C.xxx;
    vec3 x2 = x0 - i2 + C.yyy;
    vec3 x3 = x0 - 0.5;
    i = mod(i, 289.0);
    vec4 p = permute(permute(permute(
              i.z + vec4(0.0, i1.z, i2.z, 1.0))
            + i.y + vec4(0.0, i1.y, i2.y, 1.0))
            + i.x + vec4(0.0, i1.x, i2.x, 1.0));
    float n_ = 0.142857142857;
    vec3 ns = n_ * vec3(2.0, 0.5, 1.0) - vec3(0.0, 1.0, 0.0);
    vec4 j = p - 49.0 * floor(p * ns.z * ns.z);
    vec4 x_ = floor(j * ns.z);
    vec4 y_ = floor(j - 7.0 * x_);
    vec4 x = x_ * ns.x + ns.yyyy;
    vec4 y = y_ * ns.x + ns.yyyy;
    vec4 h = 1.0 - abs(x) - abs(y);
    vec4 b0 = vec4(x.xy, y.xy);
    vec4 b1 = vec4(x.zw, y.zw);
    vec4 s0 = floor(b0) * 2.0 + 1.0;
    vec4 s1 = floor(b1) * 2.0 + 1.0;
    vec4 sh = -step(h, vec4(0.0));
    vec4 a0 = b0.xzyw + s0.xzyw * sh.xxyy;
    vec4 a1 = b1.xzyw + s1.xzyw * sh.zzww;
    vec3 p0 = vec3(a0.xy, h.x);
    vec3 p1 = vec3(a0.zw, h.y);
    vec3 p2 = vec3(a1.xy, h.z);
    vec3 p3 = vec3(a1.zw, h.w);
    vec4 norm = taylorInvSqrt(vec4(dot(p0,p0), dot(p1,p1), dot(p2,p2), dot(p3,p3)));
    p0 *= norm.x; p1 *= norm.y; p2 *= norm.z; p3 *= norm.w;
    vec4 m = max(0.6 - vec4(dot(x0,x0), dot(x1,x1), dot(x2,x2), dot(x3,x3)), 0.0);
    m = m * m;
    return 42.0 * dot(m*m, vec4(dot(p0,x0), dot(p1,x1), dot(p2,x2), dot(p3,x3)));
}

// Worley Noise (Voronoi) - creates cellular, fluffy patterns
float worley(vec3 p) {
    vec3 n = floor(p);
    vec3 f = fract(p);
    float minDist = 1.0;
    float secondMinDist = 1.0;
    
    for (int x = -1; x <= 1; x++) {
        for (int y = -1; y <= 1; y++) {
            for (int z = -1; z <= 1; z++) {
                vec3 neighbor = vec3(float(x), float(y), float(z));
                vec3 point = hash33(n + neighbor);
                vec3 diff = neighbor + point - f;
                float dist = dot(diff, diff);
                
                if (dist < minDist) {
                    secondMinDist = minDist;
                    minDist = dist;
                } else if (dist < secondMinDist) {
                    secondMinDist = dist;
                }
            }
        }
    }
    return sqrt(minDist);
}

// F2 - F1 Worley for more interesting patterns
float worleyF2F1(vec3 p) {
    vec3 n = floor(p);
    vec3 f = fract(p);
    float minDist = 1.0;
    float secondMinDist = 1.0;
    
    for (int x = -1; x <= 1; x++) {
        for (int y = -1; y <= 1; y++) {
            for (int z = -1; z <= 1; z++) {
                vec3 neighbor = vec3(float(x), float(y), float(z));
                vec3 point = hash33(n + neighbor);
                vec3 diff = neighbor + point - f;
                float dist = dot(diff, diff);
                
                if (dist < minDist) {
                    secondMinDist = minDist;
                    minDist = dist;
                } else if (dist < secondMinDist) {
                    secondMinDist = dist;
                }
            }
        }
    }
    return sqrt(secondMinDist) - sqrt(minDist);
}

// ============== FBM (FRACTAL BROWNIAN MOTION) ==============

// Standard FBM with Perlin/Simplex noise
float fbmPerlin(vec3 p, int octaves) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;
    float maxValue = 0.0;
    
    for (int i = 0; i < 6; i++) {  // Max 6 octaves
        if (i >= octaves) break;
        value += amplitude * (snoise(p * frequency) * 0.5 + 0.5);
        maxValue += amplitude;
        amplitude *= 0.5;
        frequency *= 2.0;
    }
    
    return value / maxValue;
}

// FBM with Worley noise for fluffy detail
float fbmWorley(vec3 p, int octaves) {
    float value = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;
    float maxValue = 0.0;
    
    for (int i = 0; i < 4; i++) {  // Max 4 octaves for performance
        if (i >= octaves) break;
        value += amplitude * (1.0 - worley(p * frequency));
        maxValue += amplitude;
        amplitude *= 0.5;
        frequency *= 2.0;
    }
    
    return value / maxValue;
}

// Combined Perlin-Worley noise for realistic clouds
// Perlin for large shapes, Worley for fluffy detail
float cloudNoise(vec3 p) {
    // Large-scale shape from Perlin
    float perlin = fbmPerlin(p, 4);
    
    // Fluffy detail from Worley
    float worleyDetail = fbmWorley(p * 2.0, 3);
    
    // Combine: Perlin provides shape, Worley adds "cauliflower" detail
    // Use Worley to erode edges of Perlin shapes
    return perlin * (0.6 + 0.4 * worleyDetail);
}

// High-detail noise for cloud edges
float cloudDetailNoise(vec3 p) {
    float detail = 0.0;
    detail += snoise(p * 8.0) * 0.5;
    detail += snoise(p * 16.0) * 0.25;
    detail += (1.0 - worley(p * 4.0)) * 0.25;
    return detail;
}

// ============== DEPTH FUNCTIONS ==============

float depthToDistance(float depth, float near, float far) {
    float z_ndc = depth * 2.0 - 1.0;
    return (2.0 * near * far) / (far + near - z_ndc * (far - near));
}

bool isOccludedByScene(vec3 sampleWorldPos, vec2 screenUV) {
    float sceneDepthRaw = texture(uDepthTexture, screenUV).r;
    float near = 0.05;
    float far = 1024.0;
    float sceneDistance = depthToDistance(sceneDepthRaw, near, far);
    float sampleDistance = length(sampleWorldPos - uCameraPos);
    return sampleDistance > (sceneDistance + 1.0);
}

// ============== CLOUD SHAPE FUNCTIONS ==============

// Height gradient - controls vertical density distribution
float getHeightGradient(float heightFraction) {
    // Cumulus profile: denser in lower-middle, thinner at edges
    // Creates the classic "anvil" top and flat bottom
    
    // Round bottom
    float bottomFade = smoothstep(0.0, 0.15, heightFraction);
    
    // Gradual fade at top (anvil shape)
    float topFade = smoothstep(1.0, 0.6, heightFraction);
    
    // Slightly denser in middle-lower region
    float densityBias = 1.0 - pow(abs(heightFraction - 0.35), 2.0);
    
    return bottomFade * topFade * (0.7 + 0.3 * densityBias);
}

// Storm height gradient - darker, more dramatic
float getStormHeightGradient(float heightFraction) {
    float bottomFade = smoothstep(0.0, 0.1, heightFraction);
    float topFade = smoothstep(1.0, 0.4, heightFraction);
    return bottomFade * topFade;
}

// Sample cloud density at a point
float getCloudDensity(vec3 worldPos, bool highDetail) {
    // Cloud layer bounds
    float cloudBase = uCloudAltitude;
    float cloudTop = uCloudAltitude + uCloudThickness;
    
    // Storm clouds extend lower
    if (uStormActive > 0) {
        float distToStorm = length(worldPos.xz - uStormCenter.xz);
        float stormInfluence = 1.0 - smoothstep(0.0, uStormRadius * 1.5, distToStorm);
        cloudBase -= stormInfluence * 40.0;
    }
    
    // Outside cloud layer
    if (worldPos.y < cloudBase || worldPos.y > cloudTop) {
        return 0.0;
    }
    
    // Height within cloud layer (0 at base, 1 at top)
    float thickness = max(cloudTop - cloudBase, 1.0);
    float heightFraction = (worldPos.y - cloudBase) / thickness;
    
    // Height gradient for vertical shape
    float heightGradient = getHeightGradient(heightFraction);
    
    // Wind animation - clouds drift over time but are anchored to world position
    float timeOffset = uTime * 0.01;
    vec3 windOffset = uWindDirection * uWindSpeed * timeOffset;
    
    // IMPORTANT: Use absolute world position for cloud sampling
    // This ensures clouds don't move with the player
    // Scale factor determines cloud size - smaller = more detail, larger = bigger clouds
    vec3 samplePos = worldPos * 0.003 + windOffset;  // Reduced scale for more visible movement when walking
    
    // === The Secret Sauce: Multi-layer noise ===
    // Using absolute world-space coordinates - clouds are fixed in world, not moving with player
    
    // Layer 1 (Big): General cloud shape from Perlin FBM
    float baseShape = fbmPerlin(samplePos * 1.5, 4);
    
    // Layer 2 (Medium): Worley for fluffy bumps - use higher frequency for more visible structure
    float worleyBumps = 1.0 - worley(samplePos * 4.0);
    
    // Layer 3 (Small): Fine detail that changes noticeably as you move
    float fineDetail = 0.0;
    if (highDetail) {
        fineDetail = snoise(samplePos * 16.0) * 0.15;
        fineDetail += (1.0 - worley(samplePos * 10.0)) * 0.1;
    }
    
    // Combine noise layers
    float cloudShape = baseShape * 0.55 + worleyBumps * 0.35 + fineDetail * 0.35;
    
    // Add billowing effect - clouds "puff out" more at certain heights
    float billowEffect = sin(heightFraction * PI) * 0.15;
    cloudShape += billowEffect * worleyBumps;
    
    // Apply coverage threshold (turns noise into distinct clouds)
    float coverageThreshold = 1.0 - uCloudCoverage;
    cloudShape = smoothstep(coverageThreshold, coverageThreshold + 0.25, cloudShape);
    
    // Apply height gradient
    float density = cloudShape * heightGradient;
    
    // === Storm Cloud Enhancement ===
    if (uStormActive > 0) {
        float distToStormXZ = length(worldPos.xz - uStormCenter.xz);
        float stormInfluence = 1.0 - smoothstep(0.0, uStormRadius, distToStormXZ);
        
        // === TORNADO VISIBILITY CLEARING ===
        // Create a conical clear zone that follows tornado shape
        // Tornado is narrow at bottom, wider at top
        float tornadoHeight = 120.0;  // Approximate tornado height
        float heightAboveGround = worldPos.y - uStormCenter.y;
        float heightFactor = clamp(heightAboveGround / tornadoHeight, 0.0, 1.0);
        
        // Tornado funnel radius - narrow at base, wide at top
        float funnelBaseRadius = 15.0;
        float funnelTopRadius = 50.0;
        float funnelRadius = mix(funnelBaseRadius, funnelTopRadius, heightFactor);
        
        // Clear zone is larger than funnel to ensure visibility
        float innerClearRadius = funnelRadius + 80.0;  // Always clear this far from funnel
        float outerClearRadius = funnelRadius + 150.0; // Fade zone
        
        // Calculate clearing factor - 0 = no clouds, 1 = full clouds
        float clearingFactor = smoothstep(innerClearRadius, outerClearRadius, distToStormXZ);
        
        // If we're within the tornado funnel + margin, no clouds at all
        if (distToStormXZ < innerClearRadius && heightAboveGround > -20.0 && heightAboveGround < tornadoHeight + 50.0) {
            return 0.0;  // Completely clear around tornado
        }
        
        // Apply clearing falloff
        density *= clearingFactor;
        
        // Storm clouds only appear outside the clearing zone
        if (stormInfluence > 0.0 && clearingFactor > 0.4) {
            // Storm clouds are denser and more turbulent
            float stormTurbulence = snoise(samplePos * 4.0 + vec3(uTime * 0.15, 0, 0));
            float stormWorley = 1.0 - worley(samplePos * 2.0 + vec3(uTime * 0.1, 0, 0));
            
            float stormDensity = (0.4 + stormTurbulence * 0.3 + stormWorley * 0.2);
            stormDensity *= getStormHeightGradient(heightFraction);
            stormDensity *= stormInfluence;
            stormDensity *= clearingFactor;
            
            // Storm clouds extend into the lower cloud base area
            if (worldPos.y < uCloudAltitude && worldPos.y > cloudBase) {
                float lowCloudFactor = (worldPos.y - cloudBase) / (uCloudAltitude - cloudBase);
                stormDensity *= smoothstep(0.0, 0.4, lowCloudFactor);
            }
            
            density = max(density, stormDensity * 0.5);
        }
    }
    
    return clamp(density * CLOUD_DENSITY_MULTIPLIER, 0.0, 1.0);
}

// ============== LIGHTING (BEER'S LAW + SCATTERING) ==============

// Beer's Law: light attenuates exponentially as it passes through medium
float beerLambert(float density, float distance) {
    return exp(-density * distance * ABSORPTION_COEFFICIENT);
}

// Powder effect: makes cloud edges brighter (light bouncing inside)
float powder(float density) {
    return 1.0 - exp(-density * POWDER_EFFECT);
}

// Henyey-Greenstein phase function for anisotropic scattering
float henyeyGreenstein(float cosTheta, float g) {
    float g2 = g * g;
    return (1.0 - g2) / (4.0 * PI * pow(1.0 + g2 - 2.0 * g * cosTheta, 1.5));
}

// Sample light reaching a point by raymarching toward sun
float sampleLightEnergy(vec3 pos) {
    // March toward sun
    vec3 lightDir = normalize(uSunDirection);
    float stepSize = uCloudThickness / float(LIGHT_STEPS);
    
    float totalDensity = 0.0;
    vec3 samplePos = pos;
    
    for (int i = 0; i < LIGHT_STEPS; i++) {
        samplePos += lightDir * stepSize;
        totalDensity += getCloudDensity(samplePos, false) * stepSize;
    }
    
    // Beer's Law - how much light survives passage through cloud
    float lightTransmittance = beerLambert(totalDensity, 1.0);
    
    // Powder effect - brightens thin cloud edges
    float powderEffect = powder(totalDensity);
    
    return lightTransmittance * (1.0 + powderEffect * 0.5);
}

// Calculate full cloud lighting
vec3 calculateCloudLighting(vec3 pos, float density, vec3 rayDir, float heightFraction) {
    // Base cloud colors
    vec3 sunColor = vec3(1.0, 0.95, 0.85);   // Warm sun
    vec3 skyColor = vec3(0.5, 0.65, 0.9);    // Blue sky ambient
    vec3 cloudWhite = vec3(1.0, 1.0, 1.0);
    vec3 cloudDark = vec3(0.3, 0.35, 0.4);   // Shadowed cloud
    
    // Storm colors
    vec3 stormDark = vec3(0.15, 0.17, 0.22);
    vec3 stormMid = vec3(0.25, 0.27, 0.32);
    
    // Sun intensity based on time of day
    float sunHeight = max(uSunDirection.y, 0.0);
    float sunFactor = smoothstep(0.0, 0.3, sunHeight) * uSunIntensity;
    
    // Sample light energy reaching this point
    float lightEnergy = sampleLightEnergy(pos);
    
    // Phase function - forward scattering creates silver lining effect
    float cosTheta = dot(rayDir, normalize(uSunDirection));
    float phase = henyeyGreenstein(cosTheta, 0.5) * 0.5 + 0.5;
    
    // Silver lining effect (bright edges when sun is behind cloud)
    float silverLining = 0.0;
    if (cosTheta > 0.7) {
        silverLining = pow(cosTheta, 8.0) * SILVER_LINING_INTENSITY * (1.0 - density);
    }
    
    // Height-based color (brighter at top, darker at bottom)
    float heightBrightness = 0.5 + 0.5 * heightFraction;
    
    // Ambient occlusion approximation (denser = darker)
    float ao = 1.0 - density * 0.4;
    
    // Combine lighting
    vec3 directLight = sunColor * lightEnergy * sunFactor * phase;
    vec3 ambientLight = skyColor * 0.3 * heightBrightness;
    vec3 silverLight = sunColor * silverLining * sunFactor;
    
    // Base cloud color
    vec3 cloudColor = mix(cloudDark, cloudWhite, lightEnergy * heightBrightness);
    
    // Storm cloud darkening
    if (uStormActive > 0) {
        float distToStorm = length(pos.xz - uStormCenter.xz);
        float stormInfluence = 1.0 - smoothstep(0.0, uStormRadius * 1.5, distToStorm);
        cloudColor = mix(cloudColor, mix(stormMid, stormDark, density), stormInfluence * 0.8);
        
        // Reduce direct lighting in storm
        directLight *= 1.0 - stormInfluence * 0.6;
    }
    
    // Final color
    vec3 finalColor = cloudColor * (directLight + ambientLight) * ao + silverLight;
    
    // Sunset/sunrise tinting
    if (sunHeight < 0.2) {
        vec3 sunsetColor = vec3(1.0, 0.5, 0.2);
        float sunsetFactor = 1.0 - sunHeight / 0.2;
        finalColor = mix(finalColor, finalColor * sunsetColor, sunsetFactor * 0.5);
    }
    
    return finalColor;
}

// ============== RAYMARCHING ==============

vec4 raymarchClouds(vec3 rayOrigin, vec3 rayDir) {
    // Cloud layer bounds
    float cloudBase = uCloudAltitude;
    float cloudTop = uCloudAltitude + uCloudThickness;
    
    // Storm extends cloud base lower
    if (uStormActive > 0) {
        cloudBase -= 40.0;
    }
    
    // Determine camera position relative to clouds
    bool insideClouds = (rayOrigin.y >= cloudBase && rayOrigin.y <= cloudTop);
    bool aboveClouds = rayOrigin.y > cloudTop;
    
    float tNear, tFar;
    
    if (insideClouds) {
        // Camera is inside cloud layer - start from camera position
        tNear = 0.1;
        
        // Find exit point (either top or bottom depending on ray direction)
        float tBase = (cloudBase - rayOrigin.y) / rayDir.y;
        float tTop = (cloudTop - rayOrigin.y) / rayDir.y;
        
        // tFar is whichever plane we hit first in the positive direction
        if (rayDir.y > 0.0) {
            tFar = tTop;  // Going up, exit through top
        } else if (rayDir.y < 0.0) {
            tFar = tBase; // Going down, exit through bottom
        } else {
            tFar = 6000.0; // Horizontal ray, use max distance
        }
        
        // Also consider going through and hitting the other side
        if (tBase > 0.0 && tTop > 0.0) {
            tFar = max(tBase, tTop);
        }
        
    } else {
        // Camera is outside cloud layer
        float tBase = (cloudBase - rayOrigin.y) / rayDir.y;
        float tTop = (cloudTop - rayOrigin.y) / rayDir.y;
        
        tNear = min(tBase, tTop);
        tFar = max(tBase, tTop);
        
        // Skip if cloud layer is entirely behind camera
        if (tFar < 0.0) return vec4(0.0);
        tNear = max(tNear, 0.1);
    }
    
    // Limit ray length for performance
    float maxDist = 6000.0;
    tFar = min(tFar, maxDist);
    
    if (tNear >= tFar) return vec4(0.0);
    
    // Determine step count based on quality and distance
    int maxSteps = STEPS_MEDIUM;
    if (uQualityLevel == 0) maxSteps = STEPS_LOW;
    else if (uQualityLevel == 2) maxSteps = STEPS_HIGH;
    
    // Reduce steps for distant clouds
    float distFactor = 1.0 - smoothstep(500.0, 3000.0, tNear);
    maxSteps = max(int(float(maxSteps) * max(distFactor, 0.4)), 12);
    
    float rayLength = tFar - tNear;
    float baseStepSize = rayLength / float(maxSteps);
    
    // Jitter start position to reduce banding
    float jitter = hash31(vec3(texCoord * uScreenSize, uTime)) * baseStepSize;
    
    vec4 accumulated = vec4(0.0);
    float t = tNear + jitter;
    float transmittance = 1.0;
    
    // Use high detail only for close clouds
    bool useHighDetail = tNear < 1000.0;
    
    for (int i = 0; i < 64; i++) {  // Max iterations
        if (i >= maxSteps) break;
        if (transmittance < 0.01) break;  // Early termination
        if (t > tFar) break;
        
        vec3 pos = rayOrigin + rayDir * t;
        
        // Check occlusion by scene geometry
        if (isOccludedByScene(pos, texCoord)) {
            break;
        }
        
        // Sample density
        float density = getCloudDensity(pos, useHighDetail && i < maxSteps/2);
        
        if (density > 0.001) {
            // Height fraction for lighting
            float thickness = max(cloudTop - (cloudBase), 1.0);
            float heightFraction = clamp((pos.y - cloudBase) / thickness, 0.0, 1.0);
            
            // Calculate lighting
            vec3 lightColor = calculateCloudLighting(pos, density, rayDir, heightFraction);
            
            // Beer's Law for this step
            float stepTransmittance = exp(-density * baseStepSize * ABSORPTION_COEFFICIENT * 50.0);
            
            // Accumulate color with energy-conserving integration
            float alpha = (1.0 - stepTransmittance) * transmittance;
            accumulated.rgb += lightColor * alpha;
            accumulated.a += alpha;
            
            transmittance *= stepTransmittance;
            
            // Adaptive stepping - smaller steps in dense regions
            t += baseStepSize * (0.5 + 0.5 * stepTransmittance);
        } else {
            // Large steps in empty space
            t += baseStepSize * 1.5;
        }
    }
    
    // Clamp final alpha
    accumulated.a = clamp(accumulated.a, 0.0, 1.0);
    
    return accumulated;
}

// ============== MAIN ==============

void main() {
    // Convert texCoord (0-1) to normalized device coordinates (-1 to 1)
    vec2 ndc = texCoord * 2.0 - 1.0;
    
    // Use inverse view-projection matrix to unproject screen point to world ray
    // This is the most accurate method for ray reconstruction
    vec4 clipNear = vec4(ndc, -1.0, 1.0);
    vec4 clipFar = vec4(ndc, 1.0, 1.0);
    
    vec4 worldNear = uInvViewProjMat * clipNear;
    vec4 worldFar = uInvViewProjMat * clipFar;
    
    worldNear /= worldNear.w;
    worldFar /= worldFar.w;
    
    vec3 rayDir = normalize(worldFar.xyz - worldNear.xyz);
    vec3 rayOrigin = uCameraPos;
    
    // Determine if camera is inside, below, or above cloud layer
    float cloudBase = uCloudAltitude;
    float cloudTop = uCloudAltitude + uCloudThickness;
    
    // Storm clouds extend lower
    if (uStormActive > 0) {
        cloudBase -= 40.0;
    }
    
    bool cameraInsideClouds = (rayOrigin.y >= cloudBase && rayOrigin.y <= cloudTop);
    bool cameraAboveClouds = rayOrigin.y > cloudTop;
    bool cameraBelowClouds = rayOrigin.y < cloudBase;
    
    // Don't render if looking away from clouds entirely
    // - Below clouds and looking down sharply: discard
    // - Above clouds and looking up sharply: discard
    // - Inside clouds: never discard based on direction
    if (cameraBelowClouds && rayDir.y < -0.3) {
        discard;
    }
    if (cameraAboveClouds && rayDir.y > 0.3) {
        discard;
    }
    
    // Fade at horizon/extremes for smooth blending
    float horizonFade = 1.0;
    if (cameraBelowClouds) {
        horizonFade = smoothstep(-0.3, 0.1, rayDir.y);
    } else if (cameraAboveClouds) {
        horizonFade = smoothstep(0.3, -0.1, rayDir.y);
    }
    // Inside clouds - no direction-based fade
    
    // Raymarch clouds
    vec4 result = raymarchClouds(rayOrigin, rayDir);
    
    // Apply horizon fade
    result.a *= horizonFade;
    
    if (result.a < 0.001) {
        discard;
    }
    
    // Output with premultiplied alpha
    fragColor = vec4(result.rgb, result.a);
}
