/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.pufferLike.engine;

import java.lang.invoke.LambdaMetafactory;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Function;
import java.util.regex.Pattern;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.entity.Entity;
import org.bukkit.entity.EntityType;
import org.bukkit.entity.Player;
import org.bukkit.entity.Tameable;
import org.bukkit.permissions.Permission;
import org.bukkit.permissions.PermissionDefault;
import org.bukkit.plugin.Plugin;
import org.texboobcat.pufferLike.engine.Action;
import org.texboobcat.pufferLike.engine.TickService;
import org.texboobcat.pufferLike.proximity.ProximityService;

public class BudgetEngine {
    private final Plugin plugin;
    private final ProximityService proximity;
    private final TickService ticks;
    private final boolean enabled;
    private final double activationDistMod;
    private final double power;
    private final long basePeriodTicks;
    private final long minPeriod;
    private final long maxPeriod;
    private final Map<Action, Double> actionMultipliers = new EnumMap<Action, Double>(Action.class);
    private final boolean exemptNametagged;
    private final boolean exemptTamed;
    private final Set<EntityType> whitelistEntities;
    private final Set<EntityType> blacklistEntities;
    private final boolean tpsAdaptationEnabled;
    private final double tpsThreshold;
    private final double tpsMultiplier;
    private final boolean sharedBudgetEnabled;
    private final int tickBudgetPerTick;
    private final Map<Action, Integer> actionPriorities = new EnumMap<Action, Integer>(Action.class);
    private int currentTickBudgetUsed = 0;
    private long lastBudgetResetTick = 0L;
    private final boolean perWorldBudgetEnabled;
    private final Map<String, Integer> perWorldBudgets = new ConcurrentHashMap<String, Integer>();
    private final Map<String, Integer> perWorldBudgetUsed = new ConcurrentHashMap<String, Integer>();
    private final int defaultWorldBudget;
    private final boolean playerExemptionsEnabled;
    private final Set<String> exemptPlayerNames = ConcurrentHashMap.newKeySet();
    private final Set<String> exemptPlayerPermissions = ConcurrentHashMap.newKeySet();
    private final Map<String, Pattern> exemptPlayerPatterns = new ConcurrentHashMap<String, Pattern>();
    private final boolean exemptOps;
    private final Map<UUID, EntityBudgetData> entityData = new ConcurrentHashMap<UUID, EntityBudgetData>();
    private final Random jitterRandom = ThreadLocalRandom.current();
    private final boolean enableJitter;
    private final double jitterPercent;
    private final Set<UUID> entitiesToCleanup = ConcurrentHashMap.newKeySet();
    private long lastCleanupTick = 0L;
    private final int cleanupIntervalTicks;
    private final Map<Action, Long> actionCounts = new ConcurrentHashMap<Action, Long>();
    private final Map<Action, Long> throttledCounts = new ConcurrentHashMap<Action, Long>();
    private final Map<String, Map<Action, Long>> perWorldActionCounts = new ConcurrentHashMap<String, Map<Action, Long>>();
    private final Map<String, Map<Action, Long>> perWorldThrottledCounts = new ConcurrentHashMap<String, Map<Action, Long>>();
    private final Map<String, Long> playerExemptionCounts = new ConcurrentHashMap<String, Long>();
    private volatile FileConfiguration currentConfig;

    public BudgetEngine(Plugin plugin, ProximityService proximity, TickService ticks, FileConfiguration cfg) {
        this.plugin = plugin;
        this.proximity = proximity;
        this.ticks = ticks;
        this.currentConfig = cfg;
        ConfigurationSection dab = cfg.getConfigurationSection("dab");
        this.enabled = dab == null || dab.getBoolean("enabled", true);
        this.activationDistMod = dab != null ? Math.max(6.0, dab.getDouble("activation-dist-mod", 8.0)) : 8.0;
        this.power = dab != null ? Math.max(1.0, dab.getDouble("power", 2.0)) : 2.0;
        this.basePeriodTicks = dab != null ? (long)Math.max(1, dab.getInt("base-period-ticks", 1)) : 1L;
        this.minPeriod = dab != null ? (long)Math.max(1, dab.getInt("min-period-ticks", 1)) : 1L;
        this.maxPeriod = dab != null ? Math.max(this.minPeriod, (long)dab.getInt("max-period-ticks", 40)) : 40L;
        this.exemptNametagged = dab != null && dab.getBoolean("exempt-nametagged", true);
        this.exemptTamed = dab != null && dab.getBoolean("exempt-tamed", true);
        this.whitelistEntities = this.parseEntityTypes(dab, "whitelist-entities");
        this.blacklistEntities = this.parseEntityTypes(dab, "blacklist-entities");
        this.tpsAdaptationEnabled = dab != null && dab.getBoolean("tps-adaptation.enabled", true);
        this.tpsThreshold = dab != null ? dab.getDouble("tps-adaptation.threshold", 18.5) : 18.5;
        this.tpsMultiplier = dab != null ? dab.getDouble("tps-adaptation.multiplier", 1.5) : 1.5;
        ConfigurationSection perf = cfg.getConfigurationSection("performance");
        this.sharedBudgetEnabled = perf != null && perf.getBoolean("shared-budget.enabled", false);
        this.tickBudgetPerTick = perf != null ? perf.getInt("tick-budget-per-tick", 5000) : 5000;
        this.perWorldBudgetEnabled = dab != null && dab.getBoolean("per-world-budgets.enabled", false);
        this.defaultWorldBudget = dab != null ? dab.getInt("per-world-budgets.default-budget", 2500) : 2500;
        this.initializePerWorldBudgets(dab);
        this.playerExemptionsEnabled = dab != null && dab.getBoolean("player-exemptions.enabled", false);
        this.exemptOps = dab != null && dab.getBoolean("player-exemptions.exempt-ops", true);
        this.initializePlayerExemptions(dab);
        this.enableJitter = dab != null && dab.getBoolean("jitter.enabled", true);
        this.jitterPercent = dab != null ? Math.max(0.0, Math.min(0.5, dab.getDouble("jitter.percent", 0.1))) : 0.1;
        this.cleanupIntervalTicks = dab != null ? dab.getInt("cleanup-interval-ticks", 1200) : 1200;
        this.initializeActionSettings(dab, perf);
        for (Action action : Action.values()) {
            this.actionCounts.put(action, 0L);
            this.throttledCounts.put(action, 0L);
        }
        this.registerDynamicPermissions();
    }

    private void initializeActionSettings(ConfigurationSection dab, ConfigurationSection perf) {
        ConfigurationSection priorities;
        ConfigurationSection mult;
        for (Action a : Action.values()) {
            this.actionMultipliers.put(a, 1.0);
            this.actionPriorities.put(a, 50);
        }
        if (dab != null && (mult = dab.getConfigurationSection("per-action-multipliers")) != null) {
            for (Action a : Action.values()) {
                this.actionMultipliers.put(a, mult.getDouble(a.name(), this.actionMultipliers.get((Object)a).doubleValue()));
            }
        }
        if (perf != null && (priorities = perf.getConfigurationSection("priorities")) != null) {
            this.actionPriorities.put(Action.PATHFIND, priorities.getInt("AI", 40));
            this.actionPriorities.put(Action.TARGET, priorities.getInt("AI", 40));
            this.actionPriorities.put(Action.BREED, priorities.getInt("AI", 40));
            this.actionPriorities.put(Action.REPLENISH, priorities.getInt("AI", 40));
            this.actionPriorities.put(Action.ANGER, priorities.getInt("AI", 40));
        }
    }

    private void initializePerWorldBudgets(ConfigurationSection dab) {
        if (!this.perWorldBudgetEnabled || dab == null) {
            return;
        }
        ConfigurationSection worldBudgets = dab.getConfigurationSection("per-world-budgets.worlds");
        if (worldBudgets != null) {
            for (String worldName : worldBudgets.getKeys(false)) {
                int budget = worldBudgets.getInt(worldName, this.defaultWorldBudget);
                this.perWorldBudgets.put(worldName, Math.max(100, budget));
                this.perWorldBudgetUsed.put(worldName, 0);
                this.perWorldActionCounts.put(worldName, new ConcurrentHashMap());
                this.perWorldThrottledCounts.put(worldName, new ConcurrentHashMap());
                for (Action action : Action.values()) {
                    this.perWorldActionCounts.get(worldName).put(action, 0L);
                    this.perWorldThrottledCounts.get(worldName).put(action, 0L);
                }
            }
        }
    }

    private void initializePlayerExemptions(ConfigurationSection dab) {
        if (!this.playerExemptionsEnabled || dab == null) {
            return;
        }
        ConfigurationSection playerExemptions = dab.getConfigurationSection("player-exemptions");
        if (playerExemptions == null) {
            return;
        }
        List names = playerExemptions.getStringList("exempt-names");
        this.exemptPlayerNames.addAll(names);
        List permissions = playerExemptions.getStringList("exempt-permissions");
        this.exemptPlayerPermissions.addAll(permissions);
        List patterns = playerExemptions.getStringList("exempt-patterns");
        for (String patternStr : patterns) {
            try {
                Pattern pattern = Pattern.compile(patternStr, 2);
                this.exemptPlayerPatterns.put(patternStr, pattern);
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Invalid player exemption pattern: " + patternStr + " - " + e.getMessage());
            }
        }
    }

    private void registerDynamicPermissions() {
        if (!this.playerExemptionsEnabled) {
            return;
        }
        try {
            Permission exemptPerm = new Permission("pufferlike.budget.exempt", "Exempts player from budget restrictions", PermissionDefault.FALSE);
            this.plugin.getServer().getPluginManager().addPermission(exemptPerm);
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.exemptPlayerPermissions.add("pufferlike.budget.exempt");
    }

    private Set<EntityType> parseEntityTypes(ConfigurationSection section, String key) {
        if (section == null) {
            return EnumSet.noneOf(EntityType.class);
        }
        List names = section.getStringList(key);
        EnumSet<EntityType> types = EnumSet.noneOf(EntityType.class);
        for (String name : names) {
            try {
                types.add(EntityType.valueOf((String)name.toUpperCase()));
            }
            catch (IllegalArgumentException e) {
                this.plugin.getLogger().warning("Invalid entity type in " + key + ": " + name);
            }
        }
        return types;
    }

    public boolean isExempt(Entity entity) {
        if (!this.enabled) {
            return true;
        }
        if (entity instanceof Player) {
            Player player = (Player)entity;
            return this.isPlayerExempt(player);
        }
        EntityType type = entity.getType();
        if (!this.whitelistEntities.isEmpty()) {
            return !this.whitelistEntities.contains(type);
        }
        if (this.blacklistEntities.contains(type)) {
            return true;
        }
        if (this.exemptNametagged && entity.getCustomName() != null) {
            return true;
        }
        return this.exemptTamed && entity instanceof Tameable && ((Tameable)entity).isTamed();
    }

    private boolean isPlayerExempt(Player player) {
        if (!this.playerExemptionsEnabled) {
            return true;
        }
        if (this.exemptOps && player.isOp()) {
            this.playerExemptionCounts.merge("ops", 1L, Long::sum);
            return true;
        }
        if (this.exemptPlayerNames.contains(player.getName())) {
            this.playerExemptionCounts.merge("names", 1L, Long::sum);
            return true;
        }
        for (String perm : this.exemptPlayerPermissions) {
            if (!player.hasPermission(perm)) continue;
            this.playerExemptionCounts.merge("permissions", 1L, Long::sum);
            return true;
        }
        String playerName = player.getName();
        for (Map.Entry<String, Pattern> entry : this.exemptPlayerPatterns.entrySet()) {
            if (!entry.getValue().matcher(playerName).matches()) continue;
            this.playerExemptionCounts.merge("patterns", 1L, Long::sum);
            return true;
        }
        return false;
    }

    public long getNextAllowed(Entity entity, Action action) {
        EntityBudgetData data = this.entityData.get(entity.getUniqueId());
        if (data == null) {
            return 0L;
        }
        return data.nextAllowed[action.ordinal()];
    }

    public void setNextAllowed(Entity entity, Action action, long tick) {
        this.entityData.computeIfAbsent((UUID)entity.getUniqueId(), (Function<UUID, EntityBudgetData>)LambdaMetafactory.metafactory(null, null, null, (Ljava/lang/Object;)Ljava/lang/Object;, lambda$setNextAllowed$0(java.util.UUID ), (Ljava/util/UUID;)Lorg/texboobcat/pufferLike/engine/BudgetEngine$EntityBudgetData;)()).nextAllowed[action.ordinal()] = tick;
    }

    public long computeAllowedPeriod(Entity entity, Action action) {
        double currentTps;
        if (!this.enabled || this.isExempt(entity)) {
            return 1L;
        }
        double dist = this.proximity.nearestPlayerDistance(entity).orElse(Double.MAX_VALUE);
        if (dist == 0.0) {
            return 1L;
        }
        double distanceFactor = Math.max(1.0, dist / this.activationDistMod);
        double allowed = (double)this.basePeriodTicks * Math.pow(distanceFactor, this.power);
        allowed *= this.actionMultipliers.getOrDefault((Object)action, 1.0).doubleValue();
        if (this.tpsAdaptationEnabled && (currentTps = this.getCurrentTPS()) < this.tpsThreshold) {
            double tpsRatio = currentTps / 20.0;
            allowed *= (2.0 - tpsRatio) * this.tpsMultiplier;
        }
        if (this.enableJitter && allowed > 1.0) {
            double jitter = 1.0 + this.jitterRandom.nextGaussian() * this.jitterPercent;
            allowed *= Math.max(0.5, Math.min(1.5, jitter));
        }
        long result = this.clamp(Math.round(allowed), this.minPeriod, this.maxPeriod);
        return result;
    }

    public boolean shouldThrottle(Entity entity, Action action) {
        int actionCost;
        long next;
        if (!this.enabled || this.isExempt(entity)) {
            return false;
        }
        long now = this.ticks.currentTick();
        String worldName = entity.getWorld().getName();
        if (this.sharedBudgetEnabled && now > this.lastBudgetResetTick) {
            this.currentTickBudgetUsed = 0;
            this.lastBudgetResetTick = now;
        }
        if (this.perWorldBudgetEnabled && now > this.lastBudgetResetTick) {
            this.perWorldBudgetUsed.replaceAll((k, v) -> 0);
        }
        if (now < (next = this.getNextAllowed(entity, action))) {
            this.throttledCounts.merge(action, 1L, Long::sum);
            this.updatePerWorldThrottledCount(worldName, action);
            return true;
        }
        if (this.sharedBudgetEnabled && this.currentTickBudgetUsed + (actionCost = this.actionPriorities.getOrDefault((Object)action, 50).intValue()) > this.tickBudgetPerTick) {
            this.throttledCounts.merge(action, 1L, Long::sum);
            this.updatePerWorldThrottledCount(worldName, action);
            return true;
        }
        if (this.perWorldBudgetEnabled) {
            int worldBudget = this.perWorldBudgets.getOrDefault(worldName, this.defaultWorldBudget);
            int actionCost2 = this.actionPriorities.getOrDefault((Object)action, 50);
            int currentUsed = this.perWorldBudgetUsed.getOrDefault(worldName, 0);
            if (currentUsed + actionCost2 > worldBudget) {
                this.throttledCounts.merge(action, 1L, Long::sum);
                this.updatePerWorldThrottledCount(worldName, action);
                return true;
            }
        }
        return false;
    }

    public long markAndGetNext(Entity entity, Action action) {
        int actionCost;
        if (!this.enabled || this.isExempt(entity)) {
            return 1L;
        }
        long period = this.computeAllowedPeriod(entity, action);
        long now = this.ticks.currentTick();
        long next = now + period;
        String worldName = entity.getWorld().getName();
        this.setNextAllowed(entity, action, next);
        if (this.sharedBudgetEnabled) {
            actionCost = this.actionPriorities.getOrDefault((Object)action, 50);
            this.currentTickBudgetUsed += actionCost;
        }
        if (this.perWorldBudgetEnabled) {
            actionCost = this.actionPriorities.getOrDefault((Object)action, 50);
            this.perWorldBudgetUsed.merge(worldName, actionCost, Integer::sum);
        }
        this.actionCounts.merge(action, 1L, Long::sum);
        this.updatePerWorldActionCount(worldName, action);
        return period;
    }

    public void scheduleCleanup(Entity entity) {
        this.entitiesToCleanup.add(entity.getUniqueId());
    }

    public void clear(Entity entity) {
        this.entityData.remove(entity.getUniqueId());
        this.entitiesToCleanup.remove(entity.getUniqueId());
    }

    public void performMaintenance() {
        long now = this.ticks.currentTick();
        if (now - this.lastCleanupTick >= (long)this.cleanupIntervalTicks) {
            for (UUID uuid : this.entitiesToCleanup) {
                this.entityData.remove(uuid);
            }
            this.entitiesToCleanup.clear();
            this.entityData.entrySet().removeIf(entry -> {
                EntityBudgetData data = (EntityBudgetData)entry.getValue();
                return now - data.lastAccessed > (long)(this.cleanupIntervalTicks * 5);
            });
            this.lastCleanupTick = now;
        }
    }

    private void updatePerWorldActionCount(String worldName, Action action) {
        if (!this.perWorldBudgetEnabled) {
            return;
        }
        this.perWorldActionCounts.computeIfAbsent(worldName, k -> {
            ConcurrentHashMap<Action, Long> map = new ConcurrentHashMap<Action, Long>();
            for (Action a : Action.values()) {
                map.put(a, 0L);
            }
            return map;
        }).merge(action, 1L, Long::sum);
    }

    private void updatePerWorldThrottledCount(String worldName, Action action) {
        if (!this.perWorldBudgetEnabled) {
            return;
        }
        this.perWorldThrottledCounts.computeIfAbsent(worldName, k -> {
            ConcurrentHashMap<Action, Long> map = new ConcurrentHashMap<Action, Long>();
            for (Action a : Action.values()) {
                map.put(a, 0L);
            }
            return map;
        }).merge(action, 1L, Long::sum);
    }

    public synchronized void reloadConfiguration(FileConfiguration newConfig) {
        this.plugin.getLogger().info("Reloading BudgetEngine configuration...");
        try {
            ConfigurationSection priorities;
            ConfigurationSection perf;
            this.currentConfig = newConfig;
            this.perWorldBudgets.clear();
            this.perWorldBudgetUsed.clear();
            this.exemptPlayerNames.clear();
            this.exemptPlayerPermissions.clear();
            this.exemptPlayerPatterns.clear();
            ConfigurationSection dab = newConfig.getConfigurationSection("dab");
            if (dab != null) {
                this.initializePerWorldBudgets(dab);
                this.initializePlayerExemptions(dab);
                this.actionMultipliers.clear();
                for (Action a : Action.values()) {
                    this.actionMultipliers.put(a, 1.0);
                }
                ConfigurationSection mult = dab.getConfigurationSection("per-action-multipliers");
                if (mult != null) {
                    for (Action a : Action.values()) {
                        this.actionMultipliers.put(a, mult.getDouble(a.name(), this.actionMultipliers.get((Object)a).doubleValue()));
                    }
                }
            }
            if ((perf = newConfig.getConfigurationSection("performance")) != null && (priorities = perf.getConfigurationSection("priorities")) != null) {
                for (Action a : Action.values()) {
                    this.actionPriorities.put(a, priorities.getInt("AI", 40));
                }
            }
            this.plugin.getLogger().info("BudgetEngine configuration reloaded successfully!");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to reload BudgetEngine configuration: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void setWorldBudget(String worldName, int budget) {
        if (!this.perWorldBudgetEnabled) {
            this.plugin.getLogger().warning("Per-world budgets are not enabled!");
            return;
        }
        this.perWorldBudgets.put(worldName, Math.max(100, budget));
        this.perWorldBudgetUsed.putIfAbsent(worldName, 0);
        this.perWorldActionCounts.putIfAbsent(worldName, new ConcurrentHashMap());
        this.perWorldThrottledCounts.putIfAbsent(worldName, new ConcurrentHashMap());
        for (Action action : Action.values()) {
            this.perWorldActionCounts.get(worldName).putIfAbsent(action, 0L);
            this.perWorldThrottledCounts.get(worldName).putIfAbsent(action, 0L);
        }
        this.plugin.getLogger().info("Updated budget for world '" + worldName + "' to " + budget);
    }

    public void addPlayerExemption(String playerName) {
        if (!this.playerExemptionsEnabled) {
            this.plugin.getLogger().warning("Player exemptions are not enabled!");
            return;
        }
        this.exemptPlayerNames.add(playerName);
        this.plugin.getLogger().info("Added player exemption for: " + playerName);
    }

    public boolean removePlayerExemption(String playerName) {
        boolean removed = this.exemptPlayerNames.remove(playerName);
        if (removed) {
            this.plugin.getLogger().info("Removed player exemption for: " + playerName);
        }
        return removed;
    }

    public void addPlayerExemptionPattern(String pattern) {
        if (!this.playerExemptionsEnabled) {
            this.plugin.getLogger().warning("Player exemptions are not enabled!");
            return;
        }
        try {
            Pattern compiledPattern = Pattern.compile(pattern, 2);
            this.exemptPlayerPatterns.put(pattern, compiledPattern);
            this.plugin.getLogger().info("Added player exemption pattern: " + pattern);
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Invalid player exemption pattern: " + pattern + " - " + e.getMessage());
        }
    }

    public boolean removePlayerExemptionPattern(String pattern) {
        boolean removed;
        boolean bl = removed = this.exemptPlayerPatterns.remove(pattern) != null;
        if (removed) {
            this.plugin.getLogger().info("Removed player exemption pattern: " + pattern);
        }
        return removed;
    }

    public BudgetStatistics getStatistics() {
        HashSet<String> patternStrings = new HashSet<String>();
        for (Pattern pattern : this.exemptPlayerPatterns.values()) {
            patternStrings.add(pattern.pattern());
        }
        return new BudgetStatistics(new HashMap<Action, Long>(this.actionCounts), new HashMap<Action, Long>(this.throttledCounts), this.entityData.size(), this.sharedBudgetEnabled ? this.currentTickBudgetUsed : 0, this.sharedBudgetEnabled ? this.tickBudgetPerTick : 0, this.perWorldBudgetEnabled, new HashMap<String, Integer>(this.perWorldBudgets), new HashMap<String, Integer>(this.perWorldBudgetUsed), this.deepCopyPerWorldStats(this.perWorldActionCounts), this.deepCopyPerWorldStats(this.perWorldThrottledCounts), new HashMap<String, Long>(this.playerExemptionCounts), this.playerExemptionsEnabled, this.exemptPlayerNames.size(), this.exemptPlayerPermissions.size(), this.exemptPlayerPatterns.size(), new HashSet<String>(this.exemptPlayerNames), new HashSet<String>(this.exemptPlayerPermissions), patternStrings);
    }

    private Map<String, Map<Action, Long>> deepCopyPerWorldStats(Map<String, Map<Action, Long>> original) {
        HashMap<String, Map<Action, Long>> copy = new HashMap<String, Map<Action, Long>>();
        for (Map.Entry<String, Map<Action, Long>> entry : original.entrySet()) {
            copy.put(entry.getKey(), new HashMap<Action, Long>(entry.getValue()));
        }
        return copy;
    }

    public EntityBudgetInfo getEntityInfo(Entity entity) {
        if (this.isExempt(entity)) {
            return new EntityBudgetInfo(true, 0.0, new long[Action.values().length], 1L);
        }
        double distance = this.proximity.nearestPlayerDistance(entity).orElse(Double.MAX_VALUE);
        EntityBudgetData data = this.entityData.get(entity.getUniqueId());
        long[] nextAllowed = data != null ? Arrays.copyOf(data.nextAllowed, data.nextAllowed.length) : new long[Action.values().length];
        long currentPeriod = this.computeAllowedPeriod(entity, Action.PATHFIND);
        return new EntityBudgetInfo(false, distance, nextAllowed, currentPeriod);
    }

    public void resetStatistics() {
        this.actionCounts.replaceAll((k, v) -> 0L);
        this.throttledCounts.replaceAll((k, v) -> 0L);
    }

    private double getCurrentTPS() {
        try {
            return this.plugin.getServer().getTPS()[0];
        }
        catch (Exception e) {
            return Math.min(20.0, 20.0 * (1.0 - (double)this.currentTickBudgetUsed / (double)this.tickBudgetPerTick));
        }
    }

    private long clamp(long v, long min, long max) {
        return Math.max(min, Math.min(max, v));
    }

    public void setPerWorldBudget(String worldName, int budget) {
        this.perWorldBudgets.put(worldName, budget);
        this.perWorldBudgetUsed.putIfAbsent(worldName, 0);
    }

    private static /* synthetic */ EntityBudgetData lambda$setNextAllowed$0(UUID k) {
        return new EntityBudgetData();
    }

    private static class EntityBudgetData {
        final long[] nextAllowed = new long[Action.values().length];
        long lastAccessed = System.currentTimeMillis();

        private EntityBudgetData() {
        }

        void updateAccess() {
            this.lastAccessed = System.currentTimeMillis();
        }
    }

    public static class BudgetStatistics {
        private final Map<Action, Long> actionCounts;
        private final Map<Action, Long> throttledCounts;
        private final int trackedEntities;
        private final int currentBudgetUsed;
        private final int maxBudgetPerTick;
        private final boolean perWorldBudgetEnabled;
        private final Map<String, Integer> perWorldBudgets;
        private final Map<String, Integer> perWorldBudgetUsed;
        private final Map<String, Map<Action, Long>> perWorldActionCounts;
        private final Map<String, Map<Action, Long>> perWorldThrottledCounts;
        private final Map<String, Long> playerExemptionCounts;
        private final boolean playerExemptionsEnabled;
        private final int exemptPlayerNamesCount;
        private final int exemptPlayerPermissionsCount;
        private final int exemptPlayerPatternsCount;
        private final Set<String> playerExemptionsByName;
        private final Set<String> playerExemptionsByPermission;
        private final Set<String> playerExemptionsByPattern;

        public BudgetStatistics(Map<Action, Long> actionCounts, Map<Action, Long> throttledCounts, int trackedEntities, int currentBudgetUsed, int maxBudgetPerTick, boolean perWorldBudgetEnabled, Map<String, Integer> perWorldBudgets, Map<String, Integer> perWorldBudgetUsed, Map<String, Map<Action, Long>> perWorldActionCounts, Map<String, Map<Action, Long>> perWorldThrottledCounts, Map<String, Long> playerExemptionCounts, boolean playerExemptionsEnabled, int exemptPlayerNamesCount, int exemptPlayerPermissionsCount, int exemptPlayerPatternsCount, Set<String> playerExemptionsByName, Set<String> playerExemptionsByPermission, Set<String> playerExemptionsByPattern) {
            this.actionCounts = actionCounts;
            this.throttledCounts = throttledCounts;
            this.trackedEntities = trackedEntities;
            this.currentBudgetUsed = currentBudgetUsed;
            this.maxBudgetPerTick = maxBudgetPerTick;
            this.perWorldBudgetEnabled = perWorldBudgetEnabled;
            this.perWorldBudgets = perWorldBudgets;
            this.perWorldBudgetUsed = perWorldBudgetUsed;
            this.perWorldActionCounts = perWorldActionCounts;
            this.perWorldThrottledCounts = perWorldThrottledCounts;
            this.playerExemptionCounts = playerExemptionCounts;
            this.playerExemptionsEnabled = playerExemptionsEnabled;
            this.exemptPlayerNamesCount = exemptPlayerNamesCount;
            this.exemptPlayerPermissionsCount = exemptPlayerPermissionsCount;
            this.exemptPlayerPatternsCount = exemptPlayerPatternsCount;
            this.playerExemptionsByName = playerExemptionsByName;
            this.playerExemptionsByPermission = playerExemptionsByPermission;
            this.playerExemptionsByPattern = playerExemptionsByPattern;
        }

        public Map<Action, Long> getActionCounts() {
            return this.actionCounts;
        }

        public Map<Action, Long> getThrottledCounts() {
            return this.throttledCounts;
        }

        public int getTrackedEntities() {
            return this.trackedEntities;
        }

        public int getCurrentBudgetUsed() {
            return this.currentBudgetUsed;
        }

        public int getMaxBudgetPerTick() {
            return this.maxBudgetPerTick;
        }

        public boolean isPerWorldBudgetEnabled() {
            return this.perWorldBudgetEnabled;
        }

        public Map<String, Integer> getPerWorldBudgets() {
            return this.perWorldBudgets;
        }

        public Map<String, Integer> getPerWorldBudgetUsed() {
            return this.perWorldBudgetUsed;
        }

        public Map<String, Map<Action, Long>> getPerWorldActionCounts() {
            return this.perWorldActionCounts;
        }

        public Map<String, Map<Action, Long>> getPerWorldThrottledCounts() {
            return this.perWorldThrottledCounts;
        }

        public Map<String, Long> getPlayerExemptionCounts() {
            return this.playerExemptionCounts;
        }

        public boolean isPlayerExemptionsEnabled() {
            return this.playerExemptionsEnabled;
        }

        public int getExemptPlayerNamesCount() {
            return this.exemptPlayerNamesCount;
        }

        public int getExemptPlayerPermissionsCount() {
            return this.exemptPlayerPermissionsCount;
        }

        public int getExemptPlayerPatternsCount() {
            return this.exemptPlayerPatternsCount;
        }

        public Set<String> getPlayerExemptionsByName() {
            return this.playerExemptionsByName;
        }

        public Set<String> getPlayerExemptionsByPermission() {
            return this.playerExemptionsByPermission;
        }

        public Set<String> getPlayerExemptionsByPattern() {
            return this.playerExemptionsByPattern;
        }

        public double getThrottleRatio(Action action) {
            long total = this.actionCounts.getOrDefault((Object)action, 0L);
            long throttled = this.throttledCounts.getOrDefault((Object)action, 0L);
            return total > 0L ? (double)throttled / (double)total : 0.0;
        }

        public double getPerWorldThrottleRatio(String worldName, Action action) {
            if (!this.perWorldBudgetEnabled || !this.perWorldActionCounts.containsKey(worldName)) {
                return 0.0;
            }
            long total = this.perWorldActionCounts.get(worldName).getOrDefault((Object)action, 0L);
            long throttled = this.perWorldThrottledCounts.get(worldName).getOrDefault((Object)action, 0L);
            return total > 0L ? (double)throttled / (double)total : 0.0;
        }

        public int getPerWorldBudgetUtilization(String worldName) {
            if (!this.perWorldBudgetEnabled) {
                return 0;
            }
            int budget = this.perWorldBudgets.getOrDefault(worldName, 0);
            int used = this.perWorldBudgetUsed.getOrDefault(worldName, 0);
            return budget > 0 ? (int)(100.0 * (double)used / (double)budget) : 0;
        }
    }

    public static class EntityBudgetInfo {
        private final boolean exempt;
        private final double distanceToNearestPlayer;
        private final long[] nextAllowedTicks;
        private final long currentPeriod;

        public EntityBudgetInfo(boolean exempt, double distance, long[] nextAllowed, long currentPeriod) {
            this.exempt = exempt;
            this.distanceToNearestPlayer = distance;
            this.nextAllowedTicks = nextAllowed;
            this.currentPeriod = currentPeriod;
        }

        public boolean isExempt() {
            return this.exempt;
        }

        public double getDistanceToNearestPlayer() {
            return this.distanceToNearestPlayer;
        }

        public long[] getNextAllowedTicks() {
            return this.nextAllowedTicks;
        }

        public long getCurrentPeriod() {
            return this.currentPeriod;
        }
    }
}

