/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.pufferLike.engine;

import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitRunnable;
import org.bukkit.scheduler.BukkitTask;

public class TickService {
    private final Plugin plugin;
    private final FileConfiguration config;
    private final AtomicLong currentTick = new AtomicLong(0L);
    private volatile boolean started = false;
    private volatile BukkitTask tickTask;
    private final AtomicLong[] tpsHistory = new AtomicLong[20];
    private volatile int tpsHistoryIndex = 0;
    private volatile double currentTPS = 20.0;
    private volatile long lastTPSUpdate = System.currentTimeMillis();
    private final AtomicLong totalTicks = new AtomicLong(0L);
    private final AtomicLong missedTicks = new AtomicLong(0L);
    private volatile long startTime = System.currentTimeMillis();
    private volatile long lastTickTime = System.currentTimeMillis();
    private volatile boolean healthWarningIssued = false;
    private final boolean enableTPSMonitoring;
    private final boolean enableHealthChecks;
    private final double tpsWarningThreshold;
    private final int maxMissedTicks;

    public TickService(Plugin plugin) {
        this(plugin, plugin.getConfig());
    }

    public TickService(Plugin plugin, FileConfiguration config) {
        this.plugin = plugin;
        this.config = config;
        ConfigurationSection tickConfig = config.getConfigurationSection("tick-service");
        this.enableTPSMonitoring = tickConfig == null || tickConfig.getBoolean("tps-monitoring.enabled", true);
        this.enableHealthChecks = tickConfig == null || tickConfig.getBoolean("health-checks.enabled", true);
        this.tpsWarningThreshold = tickConfig != null ? tickConfig.getDouble("tps-monitoring.warning-threshold", 15.0) : 15.0;
        this.maxMissedTicks = tickConfig != null ? tickConfig.getInt("health-checks.max-missed-ticks", 100) : 100;
        for (int i = 0; i < this.tpsHistory.length; ++i) {
            this.tpsHistory[i] = new AtomicLong(20L);
        }
        this.start();
    }

    private void start() {
        if (this.started) {
            return;
        }
        this.started = true;
        this.startTime = System.currentTimeMillis();
        this.tickTask = new BukkitRunnable(){
            private long expectedTick = 1L;

            public void run() {
                try {
                    long currentTime = System.currentTimeMillis();
                    long actualTick = TickService.this.currentTick.incrementAndGet();
                    TickService.this.totalTicks.incrementAndGet();
                    if (actualTick > this.expectedTick) {
                        long missed = actualTick - this.expectedTick;
                        TickService.this.missedTicks.addAndGet(missed);
                        if (TickService.this.enableHealthChecks && missed > (long)TickService.this.maxMissedTicks && !TickService.this.healthWarningIssued) {
                            TickService.this.plugin.getLogger().warning("TickService: Detected " + missed + " missed ticks (threshold: " + TickService.this.maxMissedTicks + ")");
                            TickService.this.healthWarningIssued = true;
                        }
                    }
                    this.expectedTick = actualTick + 1L;
                    TickService.this.lastTickTime = currentTime;
                    if (TickService.this.enableTPSMonitoring) {
                        TickService.this.updateTPS(currentTime);
                    }
                    if (TickService.this.healthWarningIssued && TickService.this.missedTicks.get() == 0L) {
                        TickService.this.healthWarningIssued = false;
                    }
                }
                catch (Exception e) {
                    TickService.this.plugin.getLogger().log(Level.SEVERE, "TickService task encountered an error", e);
                }
            }
        }.runTaskTimer(this.plugin, 1L, 1L);
    }

    private void updateTPS(long currentTime) {
        if (currentTime - this.lastTPSUpdate >= 1000L) {
            long ticksThisSecond = this.tpsHistory[this.tpsHistoryIndex].get();
            this.tpsHistory[this.tpsHistoryIndex].set(0L);
            double totalTPS = 0.0;
            int validSamples = 0;
            for (AtomicLong tps : this.tpsHistory) {
                long value = tps.get();
                if (value <= 0L) continue;
                totalTPS += (double)value;
                ++validSamples;
            }
            if (validSamples > 0) {
                this.currentTPS = Math.min(20.0, totalTPS / (double)validSamples);
                if (this.currentTPS < this.tpsWarningThreshold) {
                    this.plugin.getLogger().warning(String.format("TickService: Low TPS detected: %.2f (threshold: %.2f)", this.currentTPS, this.tpsWarningThreshold));
                }
            }
            this.tpsHistoryIndex = (this.tpsHistoryIndex + 1) % this.tpsHistory.length;
            this.lastTPSUpdate = currentTime;
        }
        this.tpsHistory[this.tpsHistoryIndex].incrementAndGet();
    }

    public long currentTick() {
        return this.currentTick.get();
    }

    public double getCurrentTPS() {
        return this.currentTPS;
    }

    public boolean isTPSHealthy() {
        return this.currentTPS >= this.tpsWarningThreshold;
    }

    public boolean hasTicksPassed(long lastTick, int requiredTicks) {
        return this.currentTick() - lastTick >= (long)requiredTicks;
    }

    public boolean hasTimePassed(long lastTick, long milliseconds) {
        long requiredTicks = milliseconds / 50L;
        return this.hasTicksPassed(lastTick, (int)requiredTicks);
    }

    public long getMillisecondsSinceTick(long tick) {
        return (this.currentTick() - tick) * 50L;
    }

    public long getMillisecondsUntilTick(long targetTick) {
        long ticksRemaining = targetTick - this.currentTick();
        return Math.max(0L, ticksRemaining * 50L);
    }

    public int getAdaptiveCooldown(int baseCooldownTicks) {
        if (!this.enableTPSMonitoring) {
            return baseCooldownTicks;
        }
        double tpsRatio = this.currentTPS / 20.0;
        if (tpsRatio < 0.75) {
            return (int)((double)baseCooldownTicks * (1.5 / tpsRatio));
        }
        return baseCooldownTicks;
    }

    public TickServiceStats getStats() {
        long uptime = System.currentTimeMillis() - this.startTime;
        return new TickServiceStats(this.currentTick.get(), this.totalTicks.get(), this.missedTicks.get(), this.currentTPS, uptime, this.isHealthy(), this.enableTPSMonitoring, this.enableHealthChecks);
    }

    public boolean isHealthy() {
        return this.started && this.currentTPS >= this.tpsWarningThreshold * 0.8 && this.missedTicks.get() < (long)this.maxMissedTicks && System.currentTimeMillis() - this.lastTickTime < 5000L;
    }

    public void resetStats() {
        this.totalTicks.set(this.currentTick.get());
        this.missedTicks.set(0L);
        this.startTime = System.currentTimeMillis();
        this.healthWarningIssued = false;
        for (AtomicLong tps : this.tpsHistory) {
            tps.set(20L);
        }
        this.currentTPS = 20.0;
        this.plugin.getLogger().info("TickService statistics reset");
    }

    public void shutdown() {
        if (this.tickTask != null && !this.tickTask.isCancelled()) {
            this.tickTask.cancel();
        }
        this.started = false;
        this.plugin.getLogger().info("TickService shutdown completed");
    }

    public void restart() {
        this.shutdown();
        this.started = false;
        this.start();
        this.plugin.getLogger().info("TickService restarted");
    }

    public static class TickServiceStats {
        private final long currentTick;
        private final long totalTicks;
        private final long missedTicks;
        private final double currentTPS;
        private final long uptimeMs;
        private final boolean isHealthy;
        private final boolean tpsMonitoringEnabled;
        private final boolean healthChecksEnabled;

        public TickServiceStats(long currentTick, long totalTicks, long missedTicks, double currentTPS, long uptimeMs, boolean isHealthy, boolean tpsMonitoringEnabled, boolean healthChecksEnabled) {
            this.currentTick = currentTick;
            this.totalTicks = totalTicks;
            this.missedTicks = missedTicks;
            this.currentTPS = currentTPS;
            this.uptimeMs = uptimeMs;
            this.isHealthy = isHealthy;
            this.tpsMonitoringEnabled = tpsMonitoringEnabled;
            this.healthChecksEnabled = healthChecksEnabled;
        }

        public long getCurrentTick() {
            return this.currentTick;
        }

        public long getTotalTicks() {
            return this.totalTicks;
        }

        public long getMissedTicks() {
            return this.missedTicks;
        }

        public double getCurrentTPS() {
            return this.currentTPS;
        }

        public long getUptimeMs() {
            return this.uptimeMs;
        }

        public boolean isHealthy() {
            return this.isHealthy;
        }

        public boolean isTpsMonitoringEnabled() {
            return this.tpsMonitoringEnabled;
        }

        public boolean isHealthChecksEnabled() {
            return this.healthChecksEnabled;
        }

        public double getMissedTickRate() {
            return this.totalTicks > 0L ? (double)this.missedTicks / (double)this.totalTicks * 100.0 : 0.0;
        }

        public double getAverageTPSOverUptime() {
            return this.uptimeMs > 0L ? (double)this.totalTicks / ((double)this.uptimeMs / 1000.0) : 0.0;
        }

        public String toString() {
            return String.format("TickServiceStats{tick=%d, totalTicks=%d, missedTicks=%d (%.2f%%), currentTPS=%.2f, uptime=%ds, healthy=%s}", this.currentTick, this.totalTicks, this.missedTicks, this.getMissedTickRate(), this.currentTPS, this.uptimeMs / 1000L, this.isHealthy);
        }
    }
}

