package net.mehvahdjukaar.moonlight.api.entity;

import net.mehvahdjukaar.moonlight.api.misc.RollingBuffer;
import net.minecraft.class_1297;
import net.minecraft.class_1937;
import net.minecraft.class_2394;
import net.minecraft.class_243;
import net.minecraft.class_3532;

/**
 * A utility class for emitting particles along the trail at regular intervals no matter the speed of the entity.
 */

public class ParticleTrailEmitter {
    private final double wantedSpacing;
    private final int maxParticlesPerTick;
    private final int minParticlesPerTick;
    private final double minSpeed;
    private class_243 lastEmittedPos = null; // Track last emitted particle position

    private final RollingBuffer<class_243> previousVelocities = new RollingBuffer<>(3);
    private final RollingBuffer<class_243> previousPositions = new RollingBuffer<>(3);

    private ParticleTrailEmitter(Builder builder) {
        this.wantedSpacing = builder.idealSpacing;
        this.maxParticlesPerTick = builder.maxParticlesPerTick;
        this.minParticlesPerTick = builder.minParticlesPerTick;
        this.minSpeed = builder.minSpeed;
    }

    public void tick(class_1297 obj, class_2394 particleOptions) {
        tick(obj, particleOptions, true);
    }

    public void tick(class_1297 obj, class_2394 particleOptions, boolean followSpeed) {
        tick(obj, (position, velocity) -> {
            var level = obj.method_37908();
            if (followSpeed) {
                level.method_8406(particleOptions, position.field_1352, position.field_1351, position.field_1350, velocity.field_1352, velocity.field_1351, velocity.field_1350);
            } else {
                level.method_8406(particleOptions, position.field_1352, position.field_1351, position.field_1350, 0, 0, 0);
            }
        });
    }

    public void tick(class_1297 obj, Emitter emitter) {
        class_243 movement = obj.method_18798();
        previousVelocities.push(movement);
        previousPositions.push(obj.method_19538());

        if (previousPositions.size() < 2) return;

        if (movement.method_1027() < (minSpeed * minSpeed)) return;

        class_243 startPos = previousPositions.get(0);
        class_243 endPos = previousPositions.get(1);
        class_243 startVel = previousVelocities.get(0);
        class_243 endVel = previousVelocities.get(1);

        if (lastEmittedPos == null) {
            lastEmittedPos = startPos;
            return;
        }

        double segmentLength = startPos.method_1022(endPos);
        Double startT = intersectSphereSegment(lastEmittedPos, wantedSpacing, startPos, endPos);
        if (startT == null) {
            return;
        }

        double remainingLength = segmentLength * (1 - startT);

        float spacing = (float) wantedSpacing;

        int particlesToEmit = 1 + (int) (remainingLength / wantedSpacing); // +1 to include the first particle

        // Ensure min/max limits
        if (particlesToEmit > maxParticlesPerTick) {
            // If we have too many particles, adjust spacing to fit max particles per tick, equally spaced
            particlesToEmit = maxParticlesPerTick;
            spacing = (float) (remainingLength / particlesToEmit);
        } else if (particlesToEmit < minParticlesPerTick) {
            particlesToEmit = minParticlesPerTick;
            spacing = (float) (remainingLength / particlesToEmit);
        }


        float h = obj.method_17682() / 2;
        for (int i = 0; i < particlesToEmit; i++) {
            double t = startT + (i * spacing / (float) segmentLength);
            if (t > 1.0f) {
                break; // Avoid going beyond the end of the segment
            }
            class_243 position = startPos.method_35590(endPos, t);
            class_243 velocity = startVel.method_35590(endVel, t);
            emitter.emitParticle(position.method_1031(0, h, 0), velocity);
            lastEmittedPos = position;
        }
    }


    /**
     * Returns the segment percentage (t in [0, 1]) along the segment p1→p2 where the first intersection
     * with the sphere occurs. Returns null if no intersection on the segment.
     */
    private static Double intersectSphereSegment(class_243 center, double radius, class_243 start, class_243 end) {
        class_243 direction = end.method_1020(start);        // Direction vector of the segment
        class_243 oldDirection = start.method_1020(center);    // Vector from center to p1

        double a = direction.method_1026(direction);
        double b = 2 * oldDirection.method_1026(direction);
        double c = oldDirection.method_1026(oldDirection) - radius * radius;

        double discriminant = b * b - 4 * a * c;

        if (discriminant < 0) {
            return null; // No intersection
        }

        double sqrtDiscriminant = (float) Math.sqrt(discriminant);
        double t1 = (-b - sqrtDiscriminant) / (2 * a);
        double t2 = (-b + sqrtDiscriminant) / (2 * a);

        // Return the first intersection that lies on the segment
        if (t1 >= 0 && t1 <= 1) {
            return class_3532.method_15350(t2, 0, 1); // Ensure t2 is clamped to [0, 1]
        }
        if (t2 >= 0 && t2 <= 1) {
            return class_3532.method_15350(t2, 0, 1); // Ensure t2 is clamped to [0, 1]
        }

        return null; // No intersection on the segment
    }


    public static Builder builder() {
        return new Builder();
    }

    // === Builder Class ===
    public static class Builder {
        private double idealSpacing = 0.5;
        private int maxParticlesPerTick = 5;
        private int minParticlesPerTick = 0;
        private double minSpeed = 0.0;

        public Builder spacing(double spacing) {
            this.idealSpacing = spacing;
            return this;
        }

        public Builder maxParticlesPerTick(int max) {
            this.maxParticlesPerTick = max;
            return this;
        }

        public Builder minParticlesPerTick(int min) {
            this.minParticlesPerTick = min;
            return this;
        }

        public Builder minSpeed(double speed) {
            this.minSpeed = speed;
            return this;
        }

        public ParticleTrailEmitter build() {
            if (minParticlesPerTick > maxParticlesPerTick) {
                throw new IllegalArgumentException("minParticlesPerTick cannot be greater than maxParticlesPerTick");
            }
            return new ParticleTrailEmitter(this);
        }
    }

    public interface Emitter {
        void emitParticle(class_243 position, class_243 velocity);
    }
}
