package net.mehvahdjukaar.moonlight.api.fluids;

import com.mojang.datafixers.util.Either;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.architectury.injectables.annotations.ExpectPlatform;
import net.mehvahdjukaar.moonlight.api.misc.Triplet;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.api.util.math.ColorUtils;
import net.minecraft.class_1792;
import net.minecraft.class_1802;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3417;
import net.minecraft.class_3542;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_5381;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import net.minecraft.class_7924;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import net.minecraft.class_9331;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.Locale;
import java.util.Optional;


/**
 * Never store an instance of this directly. Always use SoftFluidReference otherwise it will not work on reload
 */
@SuppressWarnings({"unused", "OptionalUsedAsFieldOrParameterType"})
public class SoftFluid {

    private final class_2561 name;
    private final class_6885<class_3611> equivalentFluids;
    private final FluidContainerList containerList;
    private final FoodProvider food;
    private final class_6885<class_9331<?>> preservedComponentsFromItem;

    //used to indicate if it has been directly converted from a forge fluid
    public final boolean isGenerated;

    //client only

    private final class_2960 stillTexture;
    private final class_2960 flowingTexture;
    @Nullable
    private final class_2960 useTexturesFrom;
    private final int luminosity;
    private final int emissivity;
    private final int tintColor;
    //determines to which textures to apply tintColor
    private final TintMethod tintMethod;

    //populated with reload listener. Includes tintColor information
    protected int averageTextureTint = -1;

    protected SoftFluid(class_2960 still, class_2960 flowing,
                        class_2561 name, int luminosity, int emissivity,
                        int color, TintMethod tintMethod,
                        FoodProvider food, class_6885<class_9331<?>> components,
                        FluidContainerList containers, class_6885<class_3611> equivalent,
                        Optional<class_2960> textureFrom) {

        this.tintMethod = tintMethod;
        this.equivalentFluids = equivalent;
        this.luminosity = luminosity;
        this.emissivity = Math.max(emissivity, luminosity);
        this.containerList = containers;
        this.food = food;
        this.name = name;
        this.preservedComponentsFromItem = components;

        this.useTexturesFrom = textureFrom.orElse(null);

        int tint = color;

        Triplet<class_2960, class_2960, Integer> renderingData;
        if (this.useTexturesFrom != null && PlatHelper.getPhysicalSide().isClient()) {
            var data = getRenderingData(useTexturesFrom);
            if (data != null) {
                still = data.left();
                flowing = data.middle();
                tint = data.right();
            }
        }
        this.stillTexture = still;
        this.flowingTexture = flowing;
        this.tintColor = tint;

        this.isGenerated = false;
    }

    //from vanilla fluid
    public SoftFluid(class_6880<class_3611> fluid) {
        var still = class_2960.method_60654("block/water_still");
        var flowing = class_2960.method_60654("block/water_flowing");
        this.tintMethod = TintMethod.STILL_AND_FLOWING;
        this.containerList = new FluidContainerList();
        this.food = FoodProvider.EMPTY;
        this.preservedComponentsFromItem = class_6885.method_58563();

        //these textures are later overwritten by copy textures from;
        this.useTexturesFrom = fluid.method_40230().get().method_29177();
        this.equivalentFluids = class_6885.method_40246(fluid);
        var pair = getFluidSpecificAttributes(fluid.comp_349());
        this.name = pair.getSecond() == null ? class_2561.method_43470("generic fluid") : pair.getSecond();
        this.luminosity = pair.getFirst();
        this.emissivity = pair.getFirst();

        int tint = -1;

        Triplet<class_2960, class_2960, Integer> renderingData;
        if (this.useTexturesFrom != null && PlatHelper.getPhysicalSide().isClient()) {
            var data = getRenderingData(useTexturesFrom);
            if (data != null) {
                still = data.left();
                flowing = data.middle();
                tint = data.right();
            }
        }
        this.stillTexture = still;
        this.flowingTexture = flowing;
        this.tintColor = tint;

        this.isGenerated = true;
    }


    public void afterInit() {
        for (var f : equivalentFluids) {
            class_1792 filled = f.comp_349().method_15774();
            if (filled != class_1802.field_8162 && filled != class_1802.field_8550) {
                this.containerList.add(class_1802.field_8550, filled, BUCKET_COUNT,
                        class_3417.field_15126, class_3417.field_14834);
            }
        }
    }

    @Nullable
    public class_2960 getTextureOverride() {
        return useTexturesFrom;
    }

    /**
     * @return associated food
     */
    public FoodProvider getFoodProvider() {
        return food;
    }

    public class_2561 getTranslatedName() {
        return name;
    }

    public boolean isEnabled() {
        return equivalentFluids.method_40247() != 0 || !containerList.getPossibleFilled().isEmpty();
    }

    /**
     * gets equivalent forge fluid if present
     *
     * @return forge fluid
     */
    public class_6880<class_3611> getVanillaFluid() {
        for (var fluid : this.getEquivalentFluids()) {
            return fluid;
        }
        return class_3612.field_15906.method_40178();
    }

    /**
     * @return name of nbt tag that will be transferred from container item to fluid
     */
    public class_6885<class_9331<?>> getPreservedComponents() {
        return preservedComponentsFromItem;
    }

    /**
     * gets equivalent forge fluids if present
     *
     * @return forge fluid
     */
    public class_6885<class_3611> getEquivalentFluids() {
        return this.equivalentFluids;
    }

    /**
     * is equivalent to forge fluid
     *
     * @param fluid forge fluid
     * @return equivalent
     */
    public boolean isEquivalent(class_6880<class_3611> fluid) {
        return this.equivalentFluids.method_40241(fluid);
    }

    @Deprecated(forRemoval = true)
    public boolean isEmptyFluid() {
        return this == SoftFluidRegistry.empty();
    }

    /**
     * gets filled item category if container can be emptied
     *
     * @param emptyContainer empty item container
     * @return empty item.
     */
    public Optional<class_1792> getFilledContainer(class_1792 emptyContainer) {
        return this.containerList.getFilled(emptyContainer);
    }

    /**
     * gets empty item if container can be emptied
     *
     * @param filledContainer empty item container
     * @return empty item.
     */
    public Optional<class_1792> getEmptyContainer(class_1792 filledContainer) {
        return this.containerList.getEmpty(filledContainer);
    }

    /**
     * use this to access containers and possibly add new container items
     *
     * @return container map
     */
    public FluidContainerList getContainerList() {
        return containerList;
    }

    public int getLuminosity() {
        return luminosity;
    }

    public int getEmissivity() {
        return emissivity;
    }

    /**
     * @return tint color. default is -1 (white) so effectively no tint
     */
    public int getTintColor() {
        return tintColor;
    }

    public int getAverageTextureTintColor() {
        return averageTextureTint;
    }

    /**
     * @return used for fluids that only have a colored still texture and a grayscaled flowing one
     */
    public TintMethod getTintMethod() {
        return tintMethod;
    }

    /**
     * @return tint color to be used on flowing fluid texture. -1 for no tint
     */
    public boolean isColored() {
        return this.tintColor != -1;
    }

    public class_2960 getFlowingTexture() {
        return flowingTexture;
    }

    public class_2960 getStillTexture() {
        return stillTexture;
    }

    public boolean isFood() {
        return !this.food.isEmpty();
    }

    public static final int BOTTLE_COUNT = Capacity.BOTTLE.getValue();
    public static final int BOWL_COUNT = Capacity.BOWL.getValue();
    public static final int BUCKET_COUNT = Capacity.BUCKET.getValue();
    public static final int WATER_BUCKET_COUNT = 3;

    /**
     * NO_TINT for both colored textures
     * FLOWING for when only flowing texture is grayscaled
     * STILL_AND_FLOWING for when both are grayscaled
     */
    public enum TintMethod implements class_3542 {
        NO_TINT, //allows special color
        FLOWING, //use particle for flowing. Still texture wont have any color
        STILL_AND_FLOWING; //both texture needs to be gray-scaled and will be colored

        public static final Codec<TintMethod> CODEC = class_3542.method_28140(TintMethod::values);

        @Override
        public String method_15434() {
            return this.name().toLowerCase(Locale.ROOT);
        }

        public boolean appliesToFlowing() {
            return this == FLOWING || this == STILL_AND_FLOWING;
        }

        public boolean appliesToStill() {
            return this == STILL_AND_FLOWING;
        }
    }

    public static final Codec<class_6880<SoftFluid>> HOLDER_CODEC = class_5381.method_29749(SoftFluidRegistry.KEY, SoftFluid.CODEC);

    //dynamic holder impl is dumb... we must encode the key directly
    public static final class_9139<class_9129, class_6880<SoftFluid>> STREAM_CODEC =
            class_9135.method_56383(SoftFluidRegistry.KEY);

    public static final Codec<class_2561> TRANSLATABLE_COMPONENT = Codec.STRING.xmap(
            class_2561::method_43471,
            class_2561::getString
    );

    //Direct codec
    public static final Codec<SoftFluid> CODEC = RecordCodecBuilder.create((instance) -> instance.group(
            class_2960.field_25139.fieldOf("still_texture").forGetter(SoftFluid::getStillTexture),
            class_2960.field_25139.fieldOf("flowing_texture").forGetter(SoftFluid::getFlowingTexture),
            TRANSLATABLE_COMPONENT.optionalFieldOf("translation_key", class_2561.method_43471("fluid.moonlight.generic_fluid"))
                    .forGetter(SoftFluid::getTranslatedName),
            Codec.intRange(0, 15).optionalFieldOf("luminosity", 0).forGetter(SoftFluid::getLuminosity),
            Codec.intRange(0, 15).optionalFieldOf("emissivity", 0).forGetter(SoftFluid::getEmissivity),
            ColorUtils.CODEC.optionalFieldOf("color", -1).forGetter(SoftFluid::getTintColor),
            TintMethod.CODEC.optionalFieldOf("tint_method", TintMethod.STILL_AND_FLOWING).forGetter(SoftFluid::getTintMethod),
            FoodProvider.CODEC.optionalFieldOf("food", FoodProvider.EMPTY).forGetter(SoftFluid::getFoodProvider),
            Utils.lenientHomogeneousList(class_7924.field_49659)
                    .optionalFieldOf("preserved_components_from_item", class_6885.method_58563())
                    .forGetter(SoftFluid::getPreservedComponents),
            FluidContainerList.CODEC.optionalFieldOf("containers", new FluidContainerList()).forGetter(SoftFluid::getContainerList),
            Utils.lenientHomogeneousList(class_7924.field_41270).optionalFieldOf("equivalent_fluids", class_6885.method_58563())
                    .forGetter(s -> s.equivalentFluids),
            class_2960.field_25139.optionalFieldOf("use_texture_from").forGetter(s -> Optional.ofNullable(s.getTextureOverride()))
    ).apply(instance, SoftFluid::new));


    @ApiStatus.Internal
    @ExpectPlatform
    public static Pair<Integer, class_2561> getFluidSpecificAttributes(class_3611 fluid) {
        throw new AssertionError(); //fabric gets nothing here :/
    }

    //this is client only!
    @ApiStatus.Internal
    @Nullable
    @ExpectPlatform
    public static Triplet<class_2960, class_2960, Integer> getRenderingData(class_2960 useTexturesFrom) {
        throw new AssertionError();
    }

    public enum Capacity implements class_3542 {
        BOTTLE(1, 1), BOWL(2, 1), BUCKET(4, 3), BLOCK(4, 4);
        public final int value;

        Capacity(int forge, int fabric) {
            value = PlatHelper.getPlatform().isForge() ? forge : fabric;
        }

        public static final Codec<Capacity> CODEC = class_3542.method_28140(Capacity::values);
        public static final Codec<Integer> INT_CODEC = Codec.either(Codec.INT, Capacity.CODEC).xmap(
                either -> either.map(i -> i, Capacity::getValue), Either::left);


        @Override
        public String method_15434() {
            return this.name().toUpperCase(Locale.ROOT);
        }

        public int getValue() {
            return value;
        }
    }
}
