package net.mehvahdjukaar.moonlight.api.client.texture_renderer;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import net.mehvahdjukaar.moonlight.api.client.util.RenderUtil;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.class_1011;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_276;
import net.minecraft.class_286;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_308;
import net.minecraft.class_310;
import net.minecraft.class_4587;
import net.minecraft.class_757;
import net.minecraft.class_8251;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;

import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

public class RenderedTexturesManager {

    private static final LoadingCache<class_2960, FrameBufferBackedDynamicTexture> TEXTURE_CACHE =
            CacheBuilder.newBuilder()
                    .removalListener(i -> {
                        //REQUEST_FOR_CLOSING.add((FrameBufferBackedDynamicTexture) i.getValue())
                        RenderSystem.recordRenderCall(((FrameBufferBackedDynamicTexture) i.getValue())::close);
                    })
                    .expireAfterAccess(2, TimeUnit.MINUTES)
                    .build(new CacheLoader<>() {
                        @Override
                        public FrameBufferBackedDynamicTexture load(class_2960 key) {
                            return null;
                        }
                    });

    //clears the texture cache and forge all to be re-rendered
    public static void clearCache() {
        TEXTURE_CACHE.invalidateAll();
    }

    /**
     * Gets a texture object on which you'll be able to directly draw onto as its in essence a frame buffer
     * Remember to call isInitialized() as the returned texture might be empty
     * For practical purposes you are only interested to call something like buffer.getBuffer(RenderType.entityCutout(texture.getTextureLocation()));
     *
     * @param id                     id of this texture. must be unique
     * @param textureSize            dimension
     * @param textureDrawingFunction this is the function responsible to draw things onto this texture
     * @param updateEachFrame        if this texture should be redrawn each frame. Useful if you are drawing an entity or animated item
     * @return texture instance
     */
    public static FrameBufferBackedDynamicTexture requestTexture(
            class_2960 id, int textureSize,
            Consumer<FrameBufferBackedDynamicTexture> textureDrawingFunction,
            boolean updateEachFrame) {

        var texture = TEXTURE_CACHE.getIfPresent(id);
        if (texture == null) {
            texture = updateEachFrame ?
                    new TickableFrameBufferBackedDynamicTexture(id, textureSize, textureDrawingFunction) :
                    new FrameBufferBackedDynamicTexture(id, textureSize, textureDrawingFunction);
            TEXTURE_CACHE.put(id, texture);
            //add to queue which will render them next rendering cycle. Returned texture will be blank
            //REQUESTED_FOR_RENDERING.add(texture);

            RenderSystem.recordRenderCall(texture::initialize);
        }
        return texture;
    }

    public static FrameBufferBackedDynamicTexture requestFlatItemStackTexture(class_2960 res, class_1799 stack, int size) {
        return requestTexture(res, size, t -> drawItem(t, stack), true);
    }

    public static FrameBufferBackedDynamicTexture requestFlatItemTexture(class_1792 item, int size) {
        return requestFlatItemTexture(item, size, null);
    }

    public static FrameBufferBackedDynamicTexture requestFlatItemTexture(class_1792 item, int size, @Nullable Consumer<class_1011> postProcessing) {
        class_2960 id = Moonlight.res(Utils.getID(item).toString().replace(":", "/") + "/" + size);
        return requestFlatItemTexture(id, item, size, postProcessing, false);
    }

    public static FrameBufferBackedDynamicTexture requestFlatItemTexture(
            class_2960 id, class_1792 item, int size, @Nullable Consumer<class_1011> postProcessing) {
        return requestFlatItemTexture(id, item, size, postProcessing, false);
    }

    /**
     * Draws a flax GUI-like item onto this texture with the given size
     *
     * @param item           item you want to draw
     * @param size           texture size
     * @param id             texture id. Needs to be unique
     * @param postProcessing some extra drawing functions to be applied on the native image. Can be slow as its cpu sided
     */
    public static FrameBufferBackedDynamicTexture requestFlatItemTexture(
            class_2960 id, class_1792 item, int size,
            @Nullable Consumer<class_1011> postProcessing, boolean updateEachFrame) {
        return requestTexture(id, size, t -> {
            drawItem(t, item.method_7854());
            if (postProcessing != null) {
                t.download();
                class_1011 img = t.getPixels();
                postProcessing.accept(img);
                t.upload();
            }
        }, updateEachFrame);
    }


    //Utility methods

    public static void drawItem(FrameBufferBackedDynamicTexture tex, class_1799 stack) {
        drawAsInGUI(tex, s -> {
            //render stuff
            RenderUtil.getGuiDummy(s).method_51445(stack, 0, 0);
        });
    }

    public static void drawTexture(FrameBufferBackedDynamicTexture tex, class_2960 texture) {
        RenderedTexturesManager.drawAsInGUI(tex, s -> {
            RenderSystem.setShaderTexture(0, texture);
            var matrix = s.method_23760().method_23761();
            RenderSystem.disableDepthTest();
            RenderSystem.depthMask(false);
            RenderSystem.disableBlend();
            RenderSystem.setShader(class_757::method_34549);
            RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1);
            class_287 bufferBuilder = class_289.method_1348().method_1349();
            bufferBuilder.method_1328(class_293.class_5596.field_27382, class_290.field_1585);
            bufferBuilder.method_22918(matrix, 0.0f, 16, 0).method_22913(0, 0).method_1344();
            bufferBuilder.method_22918(matrix, 16, 16, 0).method_22913(1, 0).method_1344();
            bufferBuilder.method_22918(matrix, 16, 0.0f, 0).method_22913(1, 1).method_1344();
            bufferBuilder.method_22918(matrix, 0.0f, 0.0f, 0).method_22913(0, 1).method_1344();
            class_286.method_43433(bufferBuilder.method_1326());
        });
    }

    /**
     * Coordinates here are from 0 to 1
     */
    public static void drawNormalized(FrameBufferBackedDynamicTexture tex, Consumer<class_4587> drawFunction) {
        drawAsInGUI(tex, s -> {
            float scale = 1f / 16f;
            s.method_46416(8, 8, 0);
            s.method_22905(scale, scale, 1);
            drawFunction.accept(s);
        });
    }

    /**
     * Utility method that sets up an environment akin to gui rendering with a box from 0 t0 16.
     * If you render an item at 0,0 it will be centered
     */
    public static void drawAsInGUI(FrameBufferBackedDynamicTexture tex, Consumer<class_4587> drawFunction) {
        float fogStart = RenderSystem.getShaderFogStart();
        float fogEnd = RenderSystem.getShaderFogEnd();
        RenderSystem.setShaderFogStart(Integer.MAX_VALUE);
        RenderSystem.setShaderFogEnd(Integer.MAX_VALUE);

        RenderSystem.clear(256, class_310.field_1703);

        class_310 mc = class_310.method_1551();
        class_276 frameBuffer = tex.getFrameBuffer();
        frameBuffer.method_1230(class_310.field_1703);

        //render to this one
        frameBuffer.method_1235(true);

        int size = 16;
        //save old projection and sets new orthographic
        RenderSystem.backupProjectionMatrix();
        //like this so object center is exactly at 0 0 0
        Matrix4f matrix4f = new Matrix4f().setOrtho(0.0F, size, size, 0, -1000.0F, 1000);
        RenderSystem.setProjectionMatrix(matrix4f, class_8251.field_43361);

        //model view stuff
        class_4587 posestack = RenderSystem.getModelViewStack();
        posestack.method_22903();
        posestack.method_34426();

        //apply new model view transformation
        RenderSystem.applyModelViewMatrix();
        class_308.method_24211();
        //end gui setup code

        //item renderer needs a new pose stack as it applies its last to render system itself. for the rest tbh idk
        drawFunction.accept(new class_4587());

        //reset stuff
        posestack.method_22909();
        //reset model view
        RenderSystem.applyModelViewMatrix();

        //reset projection
        RenderSystem.restoreProjectionMatrix();
        //RenderSystem.clear(256, Minecraft.ON_OSX);
        //returns render calls to main render target
        mc.method_1522().method_1235(true);

        RenderSystem.setShaderFogStart(fogStart);
        RenderSystem.setShaderFogEnd(fogEnd);

    }


}