package net.mehvahdjukaar.moonlight.api.fluids;

import com.mojang.datafixers.util.Pair;
import dev.architectury.injectables.annotations.ExpectPlatform;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.minecraft.class_1268;
import net.minecraft.class_1271;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1920;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_3414;
import net.minecraft.class_3419;
import net.minecraft.class_3468;
import net.minecraft.class_3532;
import org.jetbrains.annotations.Nullable;

/**
 * instance this fluid tank in your tile entity
 */
@SuppressWarnings("unused")
public class SoftFluidTank {

    public static final int BOTTLE_COUNT = 1;
    public static final int BOWL_COUNT = 2;
    public static final int BUCKET_COUNT = 4;

    protected final int capacity;
    protected SoftFluidStack fluidStack = SoftFluidStack.empty();

    //Minor optimization. Caches the tint color for the fluid
    protected int stillTintCache = 0;
    protected int flowingTintCache = 0;
    protected int particleTintCache = 0;
    protected boolean needsColorRefresh = true;

    protected SoftFluidTank(int capacity) {
        this.capacity = capacity;
    }

    @ExpectPlatform
    public static SoftFluidTank create(int capacity) {
        throw new AssertionError();
    }

    /**
     * call this method from your block when the player interacts. tries to fill or empty current held item in tank
     *
     * @param player player
     * @param hand   hand
     * @return interaction successful
     */
    public boolean interactWithPlayer(class_1657 player, class_1268 hand, @Nullable class_1937 world, @Nullable class_2338 pos) {
        class_1799 handStack = player.method_5998(hand);

        class_1799 returnStack = this.interactWithItem(handStack, world, pos, false);
        //for items that have no bottle
        if (returnStack != null) {
            Utils.swapItem(player, hand, returnStack);

            if (!handStack.method_7960()) player.method_7259(class_3468.field_15372.method_14956(handStack.method_7909()));
            return true;
        }
        return false;
    }

    /**
     * makes current item interact with fluid tank. returns empty stack if
     *
     * @param stack ItemStack to be interacted with
     * @param world world. null if no sound is to be played
     * @param pos   position. null if no sound is to be played
     * @return resulting ItemStack: empty for empty hand return, null if it failed
     */
    @Nullable
    public class_1799 interactWithItem(class_1799 stack, class_1937 world, @Nullable class_2338 pos, boolean simulate) {
        class_1799 returnStack;
        //try filling
        var fillResult = this.fillItem(stack, world, pos, simulate);
        if (fillResult.method_5467().method_23665()) return fillResult.method_5466();
        //try emptying
        var drainResult = this.drainItem(stack, world, pos, simulate);
        if (drainResult.method_5467().method_23665()) return drainResult.method_5466();

        return null;
    }

    public class_1271<class_1799> drainItem(class_1799 filledContainerStack, @Nullable class_1937 world, @Nullable class_2338 pos, boolean simulate) {
        return drainItem(filledContainerStack, world, pos, simulate, true);
    }

    /**
     * Tries pouring the content of provided item in the tank
     * also plays sound.
     * If simulate is true, it will return the same item as normal but wont alter the container state
     *
     * @return empty container item, PASS if it failed
     */
    public class_1271<class_1799> drainItem(class_1799 filledContainer, class_1937 level, @Nullable class_2338 pos, boolean simulate, boolean playSound) {
        var extracted = SoftFluidStack.fromItem(filledContainer);
        if (extracted == null) return class_1271.method_22430(class_1799.field_8037);
        SoftFluidStack fluidStack = extracted.getFirst();

        //if it can add all of it
        if (addFluid(fluidStack, true) == fluidStack.getCount()) {
            FluidContainerList.Category category = extracted.getSecond();

            class_1799 emptyContainer = category.getEmptyContainer().method_7854();
            if (!simulate) {

                addFluid(fluidStack, false);

                class_3414 sound = category.getEmptySound();
                if (sound != null && pos != null) {
                    level.method_8396(null, pos, sound, class_3419.field_15245, 1, 1);
                }
            }
            return class_1271.method_29237(emptyContainer, level.field_9236);
        }
        return class_1271.method_22430(class_1799.field_8037);

    }


    public class_1271<class_1799> fillItem(class_1799 emptyContainer, @Nullable class_1937 world, @Nullable class_2338 pos, boolean simulate) {
        return fillItem(emptyContainer, world, pos, simulate, true);
    }

    /**
     * tries removing said amount of fluid and returns filled item
     * also plays sound
     *
     * @return filled bottle item. null if it failed or if simulated is true and failed
     */
    public class_1271<class_1799> fillItem(class_1799 emptyContainer, class_1937 level, @Nullable class_2338 pos, boolean simulate, boolean playSound) {
        var pair = this.fluidStack.splitToItem(emptyContainer);
        if (pair != null) {
            FluidContainerList.Category category = pair.getSecond();
            class_3414 sound = category.getEmptySound();
            if (sound != null && pos != null) {
                level.method_8396(null, pos, sound, class_3419.field_15245, 1, 1);
            }
            return class_1271.method_29237(pair.getFirst(), level.field_9236);
        }
        return class_1271.method_22430(class_1799.field_8037);
    }

    /**
     * Called when talk is not empty and a new fluid is added. For most uses just increments the existing one but could alter the fluid content
     * You can assume that canAddSoftFluid has been called before
     */
    protected void addFluidOntoExisting(SoftFluidStack stack) {
        this.fluidStack.grow(stack.getCount());
    }

    /**
     * tries removing bottle amount and returns filled bottle
     *
     * @return filled bottle item. null if it failed
     */
    @Nullable
    public class_1271<class_1799> fillBottle(class_1937 world, class_2338 pos) {
        return fillItem(class_1802.field_8469.method_7854(), world, pos, false);
    }

    /**
     * tries removing bucket amount and returns filled bucket
     *
     * @return filled bucket item. null if it failed
     */
    @Nullable
    public class_1271<class_1799> fillBucket(class_1937 world, class_2338 pos) {
        return fillItem(class_1802.field_8550.method_7854(), world, pos, false);
    }

    /**
     * tries removing bowl amount and returns filled bowl
     *
     * @return filled bowl item. null if it failed
     */
    @Nullable
    public class_1271<class_1799> fillBowl(class_1937 world, class_2338 pos) {
        return fillItem(class_1802.field_8428.method_7854(), world, pos, false);
    }

    /**
     * Main method called when checking if fluid can be added to this or not
     * Does check for size. Will not accept stacks too big even if some could be added
     */
    @Deprecated(forRemoval = true)
    public boolean canAddSoftFluid(SoftFluidStack fluidStack) {
        if (this.getSpace() < fluidStack.getCount()) return false;
        return isFluidCompatible(fluidStack);
    }

    /**
     * Check if fluid TYPE is compatible with content. Does not care about count
     */
    public boolean isFluidCompatible(SoftFluidStack fluidStack) {
        return this.fluidStack.isFluidEqual(fluidStack) || this.isEmpty();
    }

    @Deprecated(forRemoval = true)
    public boolean addFluid(SoftFluidStack stack) {
        return addFluid(stack, false) == stack.getCount();
    }

    /**
     * Like addFluid but can add only a part of the stack
     *
     * @return amount of fluid added. Given stack WILL be modified by subtracting that amount
     */
    public int addFluid(SoftFluidStack stack, boolean simulate) {
        if (!isFluidCompatible(stack)) return 0;
        int space = this.getSpace();
        if (space == 0) return 0;

        int amount = Math.min(space, stack.getCount());

        if (simulate) return amount;

        var toAdd = stack.split(amount);
        if (this.isEmpty()) {
            this.setFluid(toAdd);
        } else {
            this.addFluidOntoExisting(toAdd);
        }
        return amount;
    }

    /**
     * removes fluid from the tank
     *
     * @param amount   amount to remove
     * @param simulate if true, it will not actually remove the fluid
     * @return removed fluid
     */
    public SoftFluidStack removeFluid(int amount, boolean simulate) {
        if (this.isEmpty()) return SoftFluidStack.empty();
        int toRemove = Math.min(amount, this.fluidStack.getCount());
        SoftFluidStack stack = this.fluidStack.copyWithCount(toRemove);
        if (!simulate) {
            this.fluidStack.shrink(toRemove);
        }
        return stack;
    }

    /**
     * Transfers between 2 soft fluid tanks
     */
    @Deprecated(forRemoval = true)
    public boolean transferFluid(SoftFluidTank destination) {
        return this.transferFluid(destination, BOTTLE_COUNT);
    }

    //transfers between two fluid holders
    @Deprecated(forRemoval = true)
    public boolean transferFluid(SoftFluidTank destination, int amount) {
        if (this.isEmpty()) return false;
        var removed = this.removeFluid(amount, false);
        if (destination.addFluid(removed, true) == removed.getCount()) {
            destination.addFluid(removed, false);
            return true;
        }
        return false;
    }

    public int getSpace() {
        return Math.max(0, capacity - fluidStack.getCount());
    }

    public int getFluidCount() {
        return fluidStack.getCount();
    }

    public boolean isFull() {
        return fluidStack.getCount() == this.capacity;
    }

    public boolean isEmpty() {
        //count 0 should always = to fluid.empty
        return this.fluidStack.isEmpty();
    }

    /**
     * gets liquid height for renderer
     *
     * @param maxHeight maximum height in blocks
     * @return fluid height
     */
    public float getHeight(float maxHeight) {
        return maxHeight * fluidStack.getCount() / this.capacity;
    }

    /**
     * @return comparator block redstone power
     */
    public int getComparatorOutput() {
        float f = fluidStack.getCount() / (float) this.capacity;
        return class_3532.method_15375(f * 14.0F) + 1;
    }

    public SoftFluidStack getFluid() {
        return fluidStack;
    }

    public SoftFluid getFluidValue() {
        return fluidStack.getHolder().comp_349();
    }

    public void setFluid(SoftFluidStack fluid) {
        this.fluidStack = fluid.isEmpty() ? SoftFluidStack.empty() : fluid;
        refreshTintCache();
    }

    public void refreshTintCache() {
        stillTintCache = 0;
        needsColorRefresh = true;
    }

    private void fillCount() {
        this.fluidStack.setCount(this.capacity);
    }

    /**
     * resets & clears the tank
     */
    public void clear() {
        this.setFluid(SoftFluidStack.empty());
    }

    /**
     * copies the content of a fluid tank into this
     *
     * @param other other tank
     */
    public void copyContent(SoftFluidTank other) {
        SoftFluidStack stack = other.getFluid();
        this.setFluid(stack.copyWithCount(Math.min(this.capacity, stack.getCount())));
    }

    public int getCapacity() {
        return capacity;
    }

    public void capCapacity() {
        this.fluidStack.setCount(class_3532.method_15340(this.fluidStack.getCount(), 0, capacity));
    }

    private void cacheColors(@Nullable class_1920 world, @Nullable class_2338 pos) {
        stillTintCache = this.fluidStack.getStillColor(world, pos);
        flowingTintCache = this.fluidStack.getFlowingColor(world, pos);
        particleTintCache = this.fluidStack.getParticleColor(world, pos);
        needsColorRefresh = false;
    }

    @Deprecated(forRemoval = true)
    public int getTintColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        return getCachedStillColor(world, pos);
    }

    @Deprecated(forRemoval = true)
    public int getFlowingTint(@Nullable class_1920 world, @Nullable class_2338 pos) {
        return getCachedFlowingColor(world, pos);
    }

    @Deprecated(forRemoval = true)
    public int getParticleColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        return getCachedParticleColor(world, pos);
    }

    public int getCachedStillColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        if (needsColorRefresh) cacheColors(world, pos);
        return stillTintCache;
    }

    public int getCachedFlowingColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        if (needsColorRefresh) cacheColors(world, pos);
        return flowingTintCache;
    }

    public int getCachedParticleColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        if (needsColorRefresh) cacheColors(world, pos);
        return particleTintCache;
    }

    /**
     * @return true if contained fluid has associated food
     */
    public boolean containsFood() {
        return !this.fluidStack.getFoodProvider().isEmpty();
    }

    /**
     * call from tile entity. loads tank from nbt
     *
     * @param compound nbt
     */
    public void load(class_2487 compound) {
        if (compound.method_10545("FluidHolder")) {
            class_2487 cmp = compound.method_10562("FluidHolder");
            this.setFluid(SoftFluidStack.load(cmp));
        }
    }

    /**
     * call from tile entity. saves to nbt
     *
     * @param compound nbt
     * @return nbt
     */
    public class_2487 save(class_2487 compound) {
        class_2487 cmp = new class_2487();
        this.setFluid(this.fluidStack);
        this.fluidStack.save(cmp);
        compound.method_10566("FluidHolder", cmp);
        return compound;
    }

    /**
     * makes player drink 1 bottle and removes it from the tank
     *
     * @param player player
     * @param world  world
     * @return success
     */
    public boolean tryDrinkUpFluid(class_1657 player, class_1937 world) {
        if (!this.isEmpty() && this.containsFood()) {
            if (this.fluidStack.getFoodProvider().consume(player, world, this.fluidStack::applyNBTtoItemStack)) { //crap code right there
                fluidStack.shrink(1);
                return true;
            }
        }
        return false;
    }


    //util functions
    public static int getLiquidCountFromItem(class_1792 i) {
        if (i == class_1802.field_8469) {
            return BOTTLE_COUNT;
        } else if (i == class_1802.field_8428) {
            return BOWL_COUNT;
        } else if (i == class_1802.field_8550) {
            return BUCKET_COUNT;
        }
        return 0;
    }

}
