package net.mehvahdjukaar.moonlight.api.resources.pack;

import com.google.common.base.Preconditions;
import com.google.gson.JsonElement;
import net.mehvahdjukaar.moonlight.api.resources.RPUtils;
import net.mehvahdjukaar.moonlight.api.resources.ResType;
import net.mehvahdjukaar.moonlight.api.resources.SimpleTagBuilder;
import net.mehvahdjukaar.moonlight.api.resources.StaticResource;
import net.mehvahdjukaar.moonlight.api.resources.assets.LangBuilder;
import net.mehvahdjukaar.moonlight.api.resources.textures.TextureImage;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.class_173;
import net.minecraft.class_1935;
import net.minecraft.class_2248;
import net.minecraft.class_2444;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_44;
import net.minecraft.class_52;
import net.minecraft.class_5321;
import net.minecraft.class_55;
import net.minecraft.class_6862;
import net.minecraft.class_77;
import net.minecraft.class_8490;
import java.io.IOException;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

public class ResourceSink {

    private final String modId;
    private final String packId;
    final Map<class_2960, byte[]> resources = new HashMap<>();
    final Set<class_2960> notClearable = new HashSet<>();
    final Map<class_6862<?>, SimpleTagBuilder> tags = new HashMap<>();

    public ResourceSink(String modId, String packId) {
        this.modId = modId;
        this.packId = packId;

    }

    protected void addBytes(class_2960 id, byte[] bytes) {
        this.resources.put(id, Preconditions.checkNotNull(bytes));
    }


    public void addResource(StaticResource resource) {
        this.addBytes(resource.location, resource.data);
    }

    private void addJson(class_2960 path, JsonElement json) {
        try {
            this.addBytes(path, RPUtils.serializeJson(json).getBytes());
        } catch (IOException e) {
            Moonlight.LOGGER.error("Failed to write JSON {} to resource pack.", path, e);
        }
    }

    public void addJson(class_2960 location, JsonElement json, ResType resType) {
        this.addJson(resType.getPath(location), json);
    }

    public void addBytes(class_2960 location, byte[] bytes, ResType resType) {
        this.addBytes(resType.getPath(location), bytes);
    }


    @Deprecated(forRemoval = true)
    public void addAndCloseTexture(class_2960 path, TextureImage image) {
        addAndCloseTexture(path, image, true);
    }

    @Deprecated(forRemoval = true)
    public void addAndCloseTexture(class_2960 path, TextureImage image, boolean isOnAtlas) {
        try (image) {
            addTexture(path, image, isOnAtlas);
        } catch (Exception e) {
            Moonlight.LOGGER.warn("Failed to add image {} to resource pack {}.", path, this, e);
        }
    }

    //equivalent safer version of the old method with same name
    public void addAndCloseTexture(class_2960 path, Supplier<TextureImage> image) {
        try (TextureImage img = image.get()) {
            addTexture(path, img);
        }
    }


    /**
     * Adds a new textures and closes the passed native image
     * Last boolean is for textures that aren't stitched so won't be cleared automatically after stitching
     * Use it for textures such as entity textures of GUI.
     * You must close the texture yourself now
     */
    public void addTexture(class_2960 path, TextureImage image) {
        addTexture(path, image, true);
    }

    public void addTexture(class_2960 path, TextureImage image, boolean isOnAtlas) {
        try {
            this.addBytes(path, image.getImage().method_24036(), ResType.TEXTURES);
            if (!isOnAtlas) this.markNotClearable(ResType.TEXTURES.getPath(path));
            if (image.getMcMeta() != null) {
                this.addJson(path, image.getMcMeta().toJson(), ResType.MCMETA);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void markNotClearable(class_2960 path) {
        this.notClearable.add(path);
    }

    public void addBlockModel(class_2960 modelLocation, JsonElement model) {
        this.addJson(modelLocation, model, ResType.BLOCK_MODELS);
    }

    public void addItemModel(class_2960 modelLocation, JsonElement model) {
        this.addJson(modelLocation, model, ResType.ITEM_MODELS);
    }

    public void addBlockState(class_2960 modelLocation, JsonElement model) {
        this.addJson(modelLocation, model, ResType.BLOCKSTATES);
    }

    public void addLang(class_2960 langName, JsonElement language) {
        this.addJson(langName, language, ResType.LANG);
    }

    public void addLang(class_2960 langName, LangBuilder builder) {
        this.addJson(langName, builder.build(), ResType.LANG);
    }


    public void addTag(SimpleTagBuilder builder, class_5321<?> type) {
        this.tags.merge(class_6862.method_40092((class_5321) type, builder.getId()), builder,
                SimpleTagBuilder::merge);
    }

    /**
     * Adds a simple loot table that only drops the block itself
     *
     * @param block block to be dropped
     */
    public void addSimpleBlockLootTable(class_2248 block) {
        this.addLootTable(block, createSingleItemTable(block)
                .method_334(class_173.field_1172));
    }

    public void addLootTable(class_2248 block, class_52.class_53 table) {
        this.addLootTable(block.method_26162(), table.method_338());
    }

    public void addLootTable(class_2960 id, class_52 table) {
        this.addJson(id, class_8490.field_44498.method_51203().toJsonTree(table), ResType.LOOT_TABLES);
    }

    protected static class_52.class_53 createSingleItemTable(class_1935 itemLike) {
        return class_52.method_324()
                .method_336(
                        class_55.method_347()
                                .method_352(class_44.method_32448(1.0F))
                                .method_351(class_77.method_411(itemLike)).method_354());
    }

    public void addRecipe(class_2444 recipe) {
        this.addJson(recipe.method_10417(), recipe.method_17799(), ResType.RECIPES);
        class_2960 advancementId = recipe.method_10418();
        if (advancementId != null) {
            this.addJson(recipe.method_10418(), recipe.method_10415(), ResType.ADVANCEMENTS);
        }
    }

    public void addRecipeNoAdvancement(class_2444 recipe) {
        this.addJson(recipe.method_10417(), recipe.method_17799(), ResType.RECIPES);
    }


    public void addResourceIfNotPresent(class_3300 manager, StaticResource resource) {
        if (!alreadyHasAssetAtLocation(manager, resource.location)) {
            this.addResource(resource);
        }
    }

    public boolean alreadyHasTextureAtLocation(class_3300 manager, class_2960 res) {
        return alreadyHasAssetAtLocation(manager, res, ResType.TEXTURES);
    }

    public void addTextureIfNotPresent(class_3300 manager, String relativePath, Supplier<TextureImage> textureSupplier) {
        addTextureIfNotPresent(manager, relativePath, textureSupplier, true);
    }

    public void addTextureIfNotPresent(class_3300 manager, String relativePath, Supplier<TextureImage> textureSupplier, boolean isOnAtlas) {

        class_2960 res = relativePath.contains(":") ? new class_2960(relativePath) :
                new class_2960(this.modId, relativePath);
        if (!alreadyHasTextureAtLocation(manager, res)) {
            try (TextureImage textureImage = textureSupplier.get()){
                this.addTexture(res, textureImage, isOnAtlas);
            } catch (Exception e) {
                Moonlight.LOGGER.error("Failed to generate texture {}: {}", res, e);
            }
        }
    }

    public boolean alreadyHasAssetAtLocation(class_3300 manager, class_2960 res, ResType type) {
        return alreadyHasAssetAtLocation(manager, type.getPath(res));
    }

    public boolean alreadyHasAssetAtLocation(class_3300 manager, class_2960 res) {
        var resource = manager.method_14486(res);
        return resource.filter(value -> !value.method_14480().equals(this.packId)).isPresent();
    }

    /**
     * This is a handy method for dynamic resource pack since it allows to specify the name of an existing resource
     * that will then be copied and modified replacing a certain keyword in it with another.
     * This is useful when adding new woodtypes as one can simply manually add a default wood json and provide the method with the
     * default woodtype name and the target name
     * The target location will the one of this pack while its path will be the original one modified following the same principle as the json itself
     *
     * @param resource    target resource that will be copied, modified and saved back
     * @param keyword     keyword to replace
     * @param replaceWith word to replace the keyword with
     */
    public void addSimilarJsonResource(class_3300 manager, StaticResource resource, String keyword, String replaceWith) throws NoSuchElementException {
        addSimilarJsonResource(manager, resource, s -> s.replace(keyword, replaceWith));
    }

    public void addSimilarJsonResource(class_3300 manager, StaticResource resource, Function<String, String> textTransform) throws NoSuchElementException {
        addSimilarJsonResource(manager, resource, textTransform, textTransform);
    }

    public void addSimilarJsonResource(class_3300 manager, StaticResource resource, Function<String, String> textTransform, Function<String, String> pathTransform) throws NoSuchElementException {
        class_2960 fullPath = resource.location;

        //calculates new path
        StringBuilder builder = new StringBuilder();
        String[] partial = fullPath.method_12832().split("/");
        for (int i = 0; i < partial.length; i++) {
            if (i != 0) builder.append("/");
            if (i == partial.length - 1) {
                builder.append(pathTransform.apply(partial[i]));
            } else builder.append(partial[i]);
        }
        //adds modified under my namespace
        class_2960 newRes = new class_2960(this.modId, builder.toString());
        if (!alreadyHasAssetAtLocation(manager, newRes)) {
            String fullText = resource.asString();
            fullText = textTransform.apply(fullText);

            this.addBytes(newRes, fullText.getBytes());
        }
    }

    public void copyResource(class_3300 manager, class_2960 from, class_2960 to, boolean lenient) {
        var resource = manager.method_14486(from);
        if (resource.isPresent()) {
            var s = StaticResource.of(resource.get(), from);
            this.addBytes(to, s.data);
        } else {
            if (lenient) {
                Moonlight.LOGGER.info("Resource {} not found for copying to {}", from, to);
            } else {
                throw new NoSuchElementException("Resource " + from + " not found for copying to " + to);
            }
        }
    }


    /**
     * Utility method to add models overrides in a non-destructive way. Provided overrides will be added on top of whatever model is currently provided by vanilla or mod resources. IE: crossbows
     */
    public void appendModelOverride(class_3300 manager, class_2960 modelRes,
                                    Consumer<RPUtils.OverrideAppender> modelConsumer) {
        JsonElement json = RPUtils.makeModelOverride(manager, modelRes, modelConsumer);
        this.addItemModel(modelRes, json);
    }
}
