package net.mehvahdjukaar.moonlight.api.resources.recipe;

import com.google.gson.JsonObject;
import dev.architectury.injectables.annotations.ExpectPlatform;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.core.recipe.ShapedRecipeTemplate;
import net.mehvahdjukaar.moonlight.core.recipe.ShapelessRecipeTemplate;
import net.mehvahdjukaar.moonlight.core.recipe.SmeltingRecipeTemplate;
import net.mehvahdjukaar.moonlight.core.recipe.StoneCutterRecipeTemplate;
import net.minecraft.class_1865;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_3957;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

public class TemplateRecipeManager {

    private static final Map<class_2960, Function<JsonObject, ? extends IRecipeTemplate<?>>> DESERIALIZERS = new HashMap<>();

    /**
     * Registers a recipe template deserializer. Will be used to parse existing recipes and be able to merge new ones
     *
     * @param deserializer usually IRecipeTemplate::new
     * @param serializer   recipe serializer type
     */
    public static <T extends IRecipeTemplate<?>> void registerTemplate(
            class_1865<?> serializer, Function<JsonObject, T> deserializer) {
        registerTemplate(Utils.getID(serializer), deserializer);
    }

    public static <T extends IRecipeTemplate<?>> void registerTemplate(
            class_2960 serializerId, Function<JsonObject, T> deserializer) {
        DESERIALIZERS.put(serializerId, deserializer);
    }

    public static IRecipeTemplate<?> read(JsonObject recipe) throws UnsupportedOperationException {
        String type = class_3518.method_15265(recipe, "type");

        var templateFactory = DESERIALIZERS.get(new class_2960(type));

        if (templateFactory != null) {
            var template = templateFactory.apply(recipe);
            //special case for shaped with a single item...
            if (template instanceof ShapedRecipeTemplate st && st.shouldBeShapeless()) {
                template = st.toShapeless();
            }
            addRecipeConditions(recipe, template);
            return template;
        } else {
            throw new UnsupportedOperationException(String.format("Invalid recipe serializer: %s. Supported deserializers: %s",
                    type, DESERIALIZERS.keySet()));
        }
    }

    @ExpectPlatform
    private static void addRecipeConditions(JsonObject recipe, IRecipeTemplate<?> template) {
        throw new AssertionError();
    }

    static {
        registerTemplate(class_1865.field_9035, ShapedRecipeTemplate::new);
        registerTemplate(class_1865.field_9031, ShapelessRecipeTemplate::new);
        registerTemplate(class_1865.field_17640, StoneCutterRecipeTemplate::new);
        registerTemplate(class_1865.field_9042, j -> new SmeltingRecipeTemplate(j, (class_3957<?>) class_1865.field_9042));
        registerTemplate(class_1865.field_17084, j -> new SmeltingRecipeTemplate(j, (class_3957<?>) class_1865.field_17084));
        registerTemplate(class_1865.field_17085, j -> new SmeltingRecipeTemplate(j, (class_3957<?>) class_1865.field_17085));
        registerTemplate(class_1865.field_17347, j -> new SmeltingRecipeTemplate(j, (class_3957<?>) class_1865.field_17347));
        registerTemplate(new class_2960("forge:conditional"), TemplateRecipeManager::forgeConditional);
    }

    private static IRecipeTemplate<?> forgeConditional(JsonObject recipe) {
        JsonObject object = class_3518.method_15261(recipe, "recipes").get(0).getAsJsonObject();
        var template = read(object.getAsJsonObject("recipe"));
        addRecipeConditions(object, template);
        return template;
    }

}
