package net.mehvahdjukaar.moonlight.core.set;

import com.google.common.base.Stopwatch;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.mehvahdjukaar.moonlight.api.MoonlightRegistry;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.class_1767;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_6885;
import net.minecraft.class_6885.class_6888;
import net.minecraft.class_7923;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.Map.Entry;

public class BlocksColorInternal {
    public static final List<class_1767> VANILLA_COLORS = List.of(class_1767.field_7952,
            class_1767.field_7946, class_1767.field_7958, class_1767.field_7951, class_1767.field_7947, class_1767.field_7961, class_1767.field_7954, class_1767.field_7944,
            class_1767.field_7967, class_1767.field_7955, class_1767.field_7945, class_1767.field_7966, class_1767.field_7957, class_1767.field_7942, class_1767.field_7964, class_1767.field_7963);
    public static final List<class_1767> MODDED_COLORS = List.of(Arrays.stream(class_1767.values()).filter(v -> !VANILLA_COLORS.contains(v)).toArray(class_1767[]::new));

    private static final Map<String, ColoredSet<class_2248>> BLOCK_COLOR_SETS = new HashMap<>();
    private static final Map<String, ColoredSet<class_1792>> ITEM_COLOR_SETS = new HashMap<>();

    private static final Object2ObjectOpenHashMap<Object, class_1767> OBJ_TO_COLORS = new Object2ObjectOpenHashMap<>();
    private static final Object2ObjectOpenHashMap<Object, String> OBJ_TO_TYPE = new Object2ObjectOpenHashMap<>();


    public static void setup() {
        Stopwatch sw = Stopwatch.createStarted();

        Map<String, class_1767> colors = new HashMap<>();
        VANILLA_COLORS.forEach(d -> colors.put(d.method_7792(), d));
        List<String> colorPriority = new ArrayList<>(colors.keySet().stream().toList());

        addColoredFromRegistry(colors, colorPriority, class_7923.field_41175, BLOCK_COLOR_SETS);
        addColoredFromRegistry(colors, colorPriority, class_7923.field_41178, ITEM_COLOR_SETS);

        Moonlight.LOGGER.info("Initialized color sets in {}ms", sw.elapsed().toMillis());
    }

    public static void registerBlockColorSet(class_2960 key, EnumMap<class_1767, class_2248> blocks, @Nullable class_2248 defaultBlock) {
        BLOCK_COLOR_SETS.put(key.toString(), new ColoredSet<>(key, blocks, class_7923.field_41175, defaultBlock));
    }

    public static void registerItemColorSet(class_2960 key, EnumMap<class_1767, class_1792> items, @Nullable class_1792 defaultItem) {
        ITEM_COLOR_SETS.put(key.toString(), new ColoredSet<>(key, items, class_7923.field_41178, defaultItem));
    }

    private static <T> void addColoredFromRegistry(Map<String, class_1767> colors, List<String> colorPriority,
                                                   class_2378<T> registry, Map<String, ColoredSet<T>> colorSetMap) {
        Map<class_2960, EnumMap<class_1767, T>> groupedByType = new HashMap<>();
        colorPriority.sort(Comparator.comparingInt(String::length));
        Collections.reverse(colorPriority);
        //group by color
        loop1:
        for (var e : registry.method_29722()) {
            class_2960 id = e.getKey().method_29177();
            String name = id.method_12832();
            if (!name.contains("_")) continue;

            for (var c : colorPriority) {
                class_2960 newId = null;
                if (name.startsWith(c + "_")) {
                    newId = new class_2960(id.method_12836(), name.substring((c + "_").length()));
                }
                if (name.endsWith("_" + c)) {
                    newId = new class_2960(id.method_12836(), name.substring(0, name.length() - ("_" + c).length()));
                }
                if (newId != null) {
                    class_1767 dyeColor = colors.get(c);
                    groupedByType.computeIfAbsent(newId, a -> new EnumMap<>(class_1767.class)).put(dyeColor, e.getValue());
                    continue loop1;
                }
            }
        }

        //to qualify all vanilla colors must be found
        for (var j : groupedByType.entrySet()) {
            var map = j.getValue();
            class_2960 id = j.getKey();
            if (isBlacklisted(id)) continue;
            if (map.keySet().containsAll(VANILLA_COLORS)) {
                var set = new ColoredSet<>(id, map, registry);
                colorSetMap.put(id.toString(), set);

                for (var v : set.colorsToObj.entrySet()) {
                    OBJ_TO_COLORS.put(v.getValue(), v.getKey());
                    OBJ_TO_TYPE.put(v.getValue(), id.toString());
                }
                OBJ_TO_TYPE.put(set.defaultObj, id.toString());
            }
        }
    }

    private static boolean isBlacklisted(class_2960 id) {
        String modId = id.method_12836();
        return modId.equals("energeticsheep") || modId.equals("xycraft_world") || modId.equals("botania") || modId.equals("spectrum");
    }

    @Nullable
    public static class_1767 getColor(class_2248 block) {
        return OBJ_TO_COLORS.get(block);
    }

    @Nullable
    public static class_1767 getColor(class_1792 item) {
        return OBJ_TO_COLORS.get(item);
    }

    @Nullable
    public static class_1792 getColoredItem(String key, @Nullable class_1767 color) {
        var set = getItemSet(key);
        if (set != null) {
            return set.with(color);
        }
        return null;
    }

    @Nullable
    public static class_2248 getColoredBlock(String key, @Nullable class_1767 color) {
        var set = getBlockSet(key);
        if (set != null) {
            return set.with(color);
        }
        return null;
    }

    public static Set<String> getBlockKeys() {
        return BLOCK_COLOR_SETS.keySet();
    }

    public static Set<String> getItemKeys() {
        return ITEM_COLOR_SETS.keySet();
    }

    /**
     * Changes this block color
     * If the given color is null it will yield the default colored block, usually uncolored or white
     * Will return null if no block can be found using that color
     */
    @Nullable
    public static class_2248 changeColor(class_2248 old, @Nullable class_1767 newColor) {
        String key = getKey(old);
        if (key != null) {
            var set = getBlockSet(key);
            if (set != null) {
                var b = set.with(newColor);
                if (b != old) return b;
            }
        }
        return null;
    }

    /**
     * Changes this item color
     * If the given color is null it will yield the default colored item, usually uncolored or white
     * Will return null if no item can be found using that color
     */
    @Nullable
    public static class_1792 changeColor(class_1792 old, @Nullable class_1767 newColor) {
        String key = getKey(old);
        if (key != null) {
            var set = getItemSet(key);
            if (set != null) {
                var i = set.with(newColor);
                if (i != old) return i;
            }
        }
        return null;
    }

    @Nullable
    public static String getKey(class_2248 block) {
        return OBJ_TO_TYPE.get(block);
    }

    @Nullable
    public static String getKey(class_1792 item) {
        return OBJ_TO_TYPE.get(item);
    }

    @Nullable
    private static ColoredSet<class_2248> getBlockSet(String key) {
        key = new class_2960(key).toString();
        return BLOCK_COLOR_SETS.get(key);
    }

    @Nullable
    private static ColoredSet<class_1792> getItemSet(String key) {
        key = new class_2960(key).toString();
        return ITEM_COLOR_SETS.get(key);
    }

    @Nullable
    public static class_6885<class_2248> getBlockHolderSet(String key) {
        var set = getBlockSet(key);
        if (set != null) {
            return set.makeHolderSet(class_7923.field_41175);
        }
        return null;
    }

    @Nullable
    public static class_6885<class_1792> getItemHolderSet(String key) {
        var set = getItemSet(key);
        if (set != null) {
            return set.makeHolderSet(class_7923.field_41178);
        }
        return null;
    }

    /**
     * A collection of blocks or items that come in all colors
     */
    private static class ColoredSet<T> {

        private final class_2960 id;
        private final Map<class_1767, T> colorsToObj;
        private final T defaultObj;

        private ColoredSet(class_2960 id, EnumMap<class_1767, T> map, class_2378<T> registry) {
            this(id, map, registry, null);
        }

        private ColoredSet(class_2960 id, EnumMap<class_1767, T> map, class_2378<T> registry, @Nullable T defBlock) {
            this.colorsToObj = map;
            this.id = id;

            //fill optional
            List<String> newColorMods = List.of("tinted", "dye_depot", "dyenamics");
            colors:
            for (var c : MODDED_COLORS) {
                String namespace = id.method_12836();
                String path = id.method_12832();

                for (var mod : newColorMods) {
                    for (var s : new String[]{namespace + ":" + path + "_%s", namespace + ":%s_" + path, mod + ":" + path + "_%s", mod + ":%s_" + path}) {
                        var o = registry.method_17966(new class_2960(String.format(s, c.method_7792())));
                        if (o.isPresent()) {
                            colorsToObj.put(c, o.get());
                            continue colors;
                        }
                    }
                }
            }

            //fill default
            this.defaultObj = defBlock == null ? computeDefault(id, registry) : defBlock;
        }

        private T computeDefault(class_2960 id, class_2378<T> registry) {
            if (id.method_12836().equals("minecraft") && id.method_12832().contains("stained_glass")) {
                id = new class_2960(id.method_12832().replace("stained_", ""));
            } else if (id.method_12836().equals("quark")) {
                if (id.method_12832().equals("rune")) {
                    id = new class_2960("quark", "blank_rune");
                } else if (id.method_12832().equals("shard")) {
                    id = new class_2960("quark", "clear_shard");
                }
            } else if (id.equals(new class_2960("suppsquared:sack"))) {
                id = new class_2960("supplementaries:sack");
            }
            class_2960 finalId = id;
            var o = registry.method_17966(id);
            if (o.isEmpty()) {
                return registry.method_17966(new class_2960(finalId.method_12832()))
                        .orElseGet(() -> colorsToObj.get(class_1767.field_7952));
            } else {
                return o.get();
            }
        }

        /**
         * Kind of expensive. don't call too often
         */
        private class_6885<T> makeHolderSet(class_2378<T> registry) {
            //standard tag location
            var v = registry.method_40266(class_6862.method_40092(registry.method_30517(),
                    new class_2960(id.method_12836(), id.method_12832() + "s")));
            if (v.isEmpty()) {
                v = registry.method_40266(class_6862.method_40092(registry.method_30517(),
                        new class_2960(PlatHelper.getPlatform().isForge() ? "forge" : "c", id.method_12832() + "s")));
            }
            if (v.isPresent()) {
                var tag = v.get();
                boolean success = true;
                for (var t : colorsToObj.values()) {
                    if (!tag.method_40241(registry.method_40290(registry.method_29113(t).get()))) {
                        success = false;
                        break;
                    }
                }
                if (success) return tag;
            }
            return class_6885.method_40244(t -> registry.method_40290(registry.method_29113(t).get()),
                    new ArrayList<>(colorsToObj.values()));
        }

        /**
         * Null if no color is available.
         * If null dye is provided will give the default color
         */
        @Nullable
        private T with(@Nullable class_1767 newColor) {
            if (newColor != null && !colorsToObj.containsKey(newColor)) return null;
            return colorsToObj.getOrDefault(newColor, defaultObj);
        }

    }
}