package net.mehvahdjukaar.moonlight.api.fluids;

import ;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.architectury.injectables.annotations.ExpectPlatform;
import net.mehvahdjukaar.moonlight.api.MoonlightRegistry;
import net.mehvahdjukaar.moonlight.api.misc.HolderReference;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.util.PotionBottleType;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.mehvahdjukaar.moonlight.core.fluid.SoftFluidInternal;
import net.minecraft.core.*;
import net.minecraft.core.HolderLookup.RegistryLookup;
import net.minecraft.core.component.*;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtOps;
import net.minecraft.nbt.Tag;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.chat.Component;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceKey;
import net.minecraft.tags.FluidTags;
import net.minecraft.tags.TagKey;
import net.minecraft.util.ExtraCodecs;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.alchemy.PotionContents;
import net.minecraft.world.item.alchemy.Potions;
import net.minecraft.world.level.BlockAndTintGetter;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.material.Fluid;
import net.minecraft.world.level.material.FluidState;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.ArrayList;
import java.util.ArrayList;
import java.util.List;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

// do NOT have these in a static field as they contain registry holders
public class SoftFluidStack implements DataComponentHolder {

    public static final Codec<SoftFluidStack> CODEC = RecordCodecBuilder.create(i -> i.group(
            SoftFluid.HOLDER_CODEC.fieldOf("id").forGetter(SoftFluidStack::getHolder),
            ExtraCodecs.NON_NEGATIVE_INT.optionalFieldOf("count", 1).forGetter(SoftFluidStack::getCount),
            DataComponentPatch.CODEC.optionalFieldOf("components", DataComponentPatch.EMPTY)
                    .forGetter(stack -> stack.components.asPatch())
    ).apply(i, SoftFluidStack::of));

    public static final StreamCodec<RegistryFriendlyByteBuf, SoftFluidStack> STREAM_CODEC = StreamCodec.composite(
            SoftFluid.STREAM_CODEC, SoftFluidStack::getHolder,
            ByteBufCodecs.VAR_INT, SoftFluidStack::getCount,
            DataComponentPatch.STREAM_CODEC, s -> s.components.asPatch(),
            SoftFluidStack::of
    );

    // dont access directly
    private final Holder<SoftFluid> fluidHolder;
    private final SoftFluid fluid; //reference to avoid calling value all the times. these 2 should always match
    private int count;
    @NotNull
    private final PatchedDataComponentMap components;
    private boolean isEmptyCache;
    private final Holder<SoftFluid> myEmptyFluid;

    protected SoftFluidStack(Holder<SoftFluid> fluid, int count, DataComponentPatch components) {
        this.fluidHolder = fluid;
        if (components == null) {
            //TODO: remove me
            Moonlight.LOGGER.error("Some mod passed a null components, fix me");
            components = DataComponentPatch.EMPTY;
        }
        //validate
        //cant have this because stuff likes to create these from netty thread
        this.fluid = this.fluidHolder.value();
        this.components = PatchedDataComponentMap.fromPatch(DataComponentMap.EMPTY, Objects.requireNonNull(components, "component map cant be null"));
        this.count = count;
        this.updateEmpty();

        this.myEmptyFluid = haxFindEmpty(fluidHolder);
    }

    private Holder<SoftFluid> haxFindEmpty(Holder<SoftFluid> fluidHolder) {
        var ra = Utils.hackyFindRegistryOf(fluidHolder, SoftFluidRegistry.KEY);
        return MLBuiltinSoftFluids.EMPTY.lookup(ra);
    }

    @ExpectPlatform
    public static SoftFluidStack of(Holder<SoftFluid> fluid, int count, @NotNull DataComponentPatch tag) {
        throw new AssertionError();
    }

    public static SoftFluidStack of(Holder<SoftFluid> fluid, int count) {
        return of(fluid, count, DataComponentPatch.EMPTY);
    }

    public static SoftFluidStack of(Holder<SoftFluid> fluid) {
        return of(fluid, 1);
    }

    public static SoftFluidStack bucket(Holder<SoftFluid> fluid) {
        return of(fluid, SoftFluid.BUCKET_COUNT);
    }

    public static SoftFluidStack bowl(Holder<SoftFluid> fluid) {
        return of(fluid, SoftFluid.BOWL_COUNT);
    }

    public static SoftFluidStack bottle(Holder<SoftFluid> fluid) {
        return of(fluid, SoftFluid.BOTTLE_COUNT);
    }

    public static SoftFluidStack fromFluid(Fluid fluid, int amount) {
        return fromFluid(fluid, amount, DataComponentPatch.EMPTY);
    }

    @Deprecated(forRemoval = true)
    @NotNull
    public static SoftFluidStack fromFluid(Fluid fluid, int amount, @NotNull DataComponentPatch component) {
        RegistryAccess reg = Utils.hackyGetRegistryAccess();
        return fromFluid(fluid, amount, component, reg);
    }

    @NotNull
    public static SoftFluidStack fromFluid(Fluid fluid, int amount, @NotNull DataComponentPatch component, HolderLookup.Provider reg) {
        Holder<SoftFluid> f = SoftFluidInternal.fromVanillaFluid(fluid, reg);
        if (f == null) return empty(reg);
        return of(f, amount, component);
    }

    @Deprecated(forRemoval = true)
    @NotNull
    public static SoftFluidStack fromFluid(FluidState fluid) {
        return fromFluid(fluid, Utils.hackyGetRegistryAccess());
    }

    @NotNull
    public static SoftFluidStack fromFluid(FluidState fluid, HolderLookup.Provider reg) {
        if (fluid.is(FluidTags.WATER)) {
            return fromFluid(fluid.getType(), SoftFluid.WATER_BUCKET_COUNT, DataComponentPatch.EMPTY, reg);
        }
        return fromFluid(fluid.getType(), SoftFluid.BUCKET_COUNT, DataComponentPatch.EMPTY, reg);
    }

    @Deprecated(forRemoval = true)
    public static SoftFluidStack empty() {
        return of(SoftFluidRegistry.hackyGetEmpty(), 0);
    }

    public static SoftFluidStack empty(HolderLookup.Provider lookupProvider) {
        return of(SoftFluidRegistry.getEmpty(lookupProvider), 0);
    }

    public static SoftFluidStack empty(HolderGetter<SoftFluid> reg) {
        return of(SoftFluidRegistry.getEmpty(reg), 0);
    }

    public Component getDisplayName() {
        if (MLBuiltinSoftFluids.POTION.is(this.fluidHolder)) {
            PotionBottleType bottle = PotionBottleType.getOrDefault(this);
            return bottle.getTranslatedName();
        }
        return this.fluid().getTranslatedName();
    }

    public Tag save(HolderLookup.Provider lookupProvider) {
        var a = CODEC.encodeStart(lookupProvider.createSerializationContext(NbtOps.INSTANCE), this);
        if (a.isSuccess()) return a.getOrThrow();
        else {
            Moonlight.LOGGER.error("Failed to encode fluid stack. HOW??, {}", a.error().get());
            if (PlatHelper.isDev()) a.getOrThrow();
            return new CompoundTag();
        }
    }

    public static SoftFluidStack load(HolderLookup.Provider lookupProvider, Tag tag) {
        //TODO: add components backwards compat
        return CODEC.parse(lookupProvider.createSerializationContext(NbtOps.INSTANCE), tag).getOrThrow();
    }

    public boolean is(HolderReference<SoftFluid> fluid) {
        return fluid.is(this.fluidHolder);
    }

    public boolean is(TagKey<SoftFluid> tag) {
        return getHolder().is(tag);
    }

    public boolean is(ResourceKey<SoftFluid> location) {
        return getHolder().is(location);
    }

    @Deprecated(forRemoval = true)
    public boolean is(SoftFluid fluid) {
        return this.fluid() == fluid;
    }

    public boolean is(Holder<SoftFluid> fluid) {
        return fluid == this.fluidHolder || fluid.is(this.fluidKey());
    }

    @Deprecated(forRemoval = true)
    private Holder<SoftFluid> getFluid() {
        return isEmptyCache ? myEmptyFluid : fluidHolder;
    }

    public final Holder<SoftFluid> getHolder() {
        return isEmptyCache ? myEmptyFluid : fluidHolder;
    }

    public final SoftFluid fluid() {
        return isEmptyCache ? myEmptyFluid.value() : fluid;
    }

    public final ResourceKey<SoftFluid> fluidKey() {
        return getHolder().unwrapKey().get();
    }

    public boolean isEmpty() {
        return isEmptyCache;
    }

    protected void updateEmpty() {
        isEmptyCache = count <= 0 || MLBuiltinSoftFluids.EMPTY.is(fluidHolder);
    }

    public int getCount() {
        return isEmptyCache ? 0 : count;
    }

    public void setCount(int count) {
        if (MLBuiltinSoftFluids.EMPTY.is(fluidHolder)) {
            if (PlatHelper.isDev()) throw new AssertionError();
            return;
        }
        this.count = count;
        updateEmpty();
    }

    public void grow(int amount) {
        setCount(this.count + amount);
    }

    public void shrink(int amount) {
        setCount(this.count - amount);
    }

    public void consume(int amount, @Nullable LivingEntity entity) {
        if (entity == null || !entity.hasInfiniteMaterials()) {
            this.shrink(amount);
        }
    }

    public SoftFluidStack copy() {
        return of(getHolder(), count, components.copy().asPatch());
    }

    public SoftFluidStack copyWithCount(int count) {
        SoftFluidStack stack = this.copy();
        if (!stack.isEmpty()) {
            stack.setCount(count);
        }
        return stack;
    }

    public SoftFluidStack split(int amount) {
        int i = Math.min(amount, this.getCount());
        SoftFluidStack stack = this.copyWithCount(i);
        if (!this.isEmpty()) this.shrink(i);
        return stack;
    }

    /**
     * Checks if the fluids and NBT Tags are equal. This does not check amounts.
     */
    public boolean isSameFluidSameComponents(SoftFluidStack other) {
        if (!this.is(other.getHolder())) {
            return false;
        } else {
            return this.isEmpty() && other.isEmpty() || Objects.equals(this.components, other.components);
        }
    }

    /**
     * Hashes the fluid and components of this stack, ignoring the amount.
     */
    public static int hashFluidAndComponents(@Nullable SoftFluidStack stack) {
        if (stack != null) {
            int i = 31 + stack.getHolder().hashCode();
            return 31 * i + stack.getComponents().hashCode();
        } else {
            return 0;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof SoftFluidStack that)) return false;
        return count == that.count && Objects.equals(fluidHolder.unwrapKey(), that.fluidHolder.unwrapKey()) && Objects.equals(components, that.components);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fluidHolder.unwrapKey(), count, components);
    }

    @Override
    public String toString() {
        return this.getCount() + " " + this.getHolder();
    }

    // item conversion

    @Deprecated(forRemoval = true)
    public static Pair<SoftFluidStack, FluidContainerList.Category> fromItem(ItemStack itemStack) {
        return fromItem(itemStack, Utils.hackyGetRegistryAccess());
    }

    @Nullable
    public static Pair<SoftFluidStack, FluidContainerList.Category> fromItem(ItemStack itemStack, HolderLookup.Provider reg) {
        Item filledContainer = itemStack.getItem();
        Holder<SoftFluid> fluid = SoftFluidInternal.fromVanillaItem(filledContainer, reg);

        if (fluid != null && !MLBuiltinSoftFluids.EMPTY.is(fluid)) {
            var category = fluid.value().getContainerList()
                    .getCategoryFromFilled(filledContainer);

            if (category.isPresent()) {

                int count = category.get().getCapacity();

                DataComponentPatch.Builder fluidComponents = DataComponentPatch.builder();

                //convert potions to water bottles
                PotionContents potion = itemStack.getOrDefault(DataComponents.POTION_CONTENTS, PotionContents.EMPTY);
                if (potion.is(Potions.WATER)) {
                    fluid = MLBuiltinSoftFluids.WATER.getHolder(reg);
                }
                //add tags to splash and lingering potions
                else if (potion.hasEffects()) {
                    PotionBottleType bottleType = PotionBottleType.getOrDefault(filledContainer);
                    fluidComponents.set(MoonlightRegistry.BOTTLE_TYPE.get(), bottleType);
                }
                SoftFluidStack sfStack = SoftFluidStack.of(fluid, count, fluidComponents.build());

                copyComponentsTo(itemStack, sfStack, fluid.value().getPreservedComponents());

                return Pair.of(sfStack, category.get());
            }
        }
        return null;
    }


    /**
     * Converts to item and decrement the stack by extracted amount
     */
    public Pair<ItemStack, FluidContainerList.Category> splitToItem(ItemStack emptyContainer) {
        var r = toItem(emptyContainer);
        if (r != null) this.shrink(r.getSecond().getCapacity());
        return r;
    }

    @Deprecated(forRemoval = true)
    public Pair<ItemStack, FluidContainerList.Category> toItem(ItemStack emptyContainer, boolean dontModifyStack) {
        var r = toItem(emptyContainer);
        if (r != null && !dontModifyStack) this.shrink(r.getSecond().getCapacity());
        return r;
    }

    /**
     * Fills the item if possible. Returns empty stack if it fails
     *
     * @param emptyContainer empty bottle item
     * @return null if it fails, filled stack otherwise
     */
    @Nullable
    public Pair<ItemStack, FluidContainerList.Category> toItem(ItemStack emptyContainer) {
        var opt = fluid().getContainerList().getCategoryFromEmpty(emptyContainer.getItem());
        if (opt.isPresent()) {
            FluidContainerList.Category category = opt.get();
            var filledStacks = createFilledStacks(category, true);
            if (filledStacks.length != 0) {
                // still return only the *first* one, to stay consistent
                return Pair.of(filledStacks[0], category);
            }
        }
        return null;
    }

    public Multimap<FluidContainerList.Category, ItemStack> toAllPossibleFilledItems() {
        Multimap<FluidContainerList.Category, ItemStack> result = ArrayListMultimap.create();

        for (FluidContainerList.Category category : fluid().getContainerList()) {
            for (ItemStack filled : createFilledStacks(category,false)) {
                result.put(category, filled);
            }
        }

        return result;
    }

    private ItemStack[] createFilledStacks(FluidContainerList.Category category, boolean onlyFirst) {
        int shrinkAmount = category.getCapacity();
        if (shrinkAmount > this.getCount()) {
            return new ItemStack[0];
        }

        List<ItemStack> results = new ArrayList<>();

        for (ItemLike item : category.getFilledItems()) {
            ItemStack filledStack = new ItemStack(item);

            // hardcoded potion handling
            if (category.getEmptyContainer() == Items.GLASS_BOTTLE && this.is(MLBuiltinSoftFluids.POTION)) {
                PotionBottleType type = PotionBottleType.getOrDefault(this);
                filledStack = type.getDefaultItem();
            }

            // converts water bottles into potions
            if (category.getEmptyContainer() == Items.GLASS_BOTTLE && this.is(MLBuiltinSoftFluids.WATER)) {
                filledStack = PotionContents.createItemStack(Items.POTION, Potions.WATER);
            }

            this.copyComponentsTo(filledStack);
            results.add(filledStack);

            if (onlyFirst) {
                break;
            }
        }

        return results.toArray(new ItemStack[0]);
    }




    public void copyComponentsTo(DataComponentHolder to) {
        copyComponentsTo(this, to, this.fluid.getPreservedComponents());
    }

    //handles special nbt items such as potions or soups
    protected static void copyComponentsTo(DataComponentHolder from,
                                           DataComponentHolder to,
                                           HolderSet<DataComponentType<?>> types) {
        for (Holder<DataComponentType<?>> h : types) {
            //ignores bottle tag, handled separately since it's a diff item
            DataComponentType<?> type = h.value();
            copyComponentTo(from, to, type);
        }
    }

    private static <A> void copyComponentTo(DataComponentHolder from, DataComponentHolder to, DataComponentType<A> comp) {
        var componentValue = from.get(comp);
        if (componentValue != null) {
            if (to instanceof ItemStack is)
                is.set(comp, componentValue);
            else if (to instanceof SoftFluidStack sf)
                sf.set(comp, componentValue);
            else PlatHelper.setComponent(to, comp, componentValue);
        }
    }


    // fluid delegates

    public FluidContainerList getContainerList() {
        return fluid().getContainerList();
    }

    public FoodProvider getFoodProvider() {
        return fluid().getFoodProvider();
    }

    public boolean isEquivalent(Holder<Fluid> fluid) {
        return this.isEquivalent(fluid, DataComponentPatch.EMPTY);
    }

    public boolean isEquivalent(Holder<Fluid> fluid, DataComponentPatch componentPatch) {
        return this.fluid().isEquivalent(fluid) && Objects.equals(this.components.asPatch(), componentPatch);
    }

    public Holder<Fluid> getVanillaFluid() {
        return this.fluid().getVanillaFluid();
    }

    /**
     * Client only
     *
     * @return tint color to be applied on the fluid texture
     */
    public int getStillColor(@Nullable BlockAndTintGetter world, @Nullable BlockPos pos) {
        SoftFluid fluid = this.fluid();
        SoftFluid.TintMethod method = fluid.getTintMethod();
        if (method == SoftFluid.TintMethod.NO_TINT) return -1;
        int specialColor = SoftFluidColors.getSpecialColor(this, world, pos);

        if (specialColor != 0) return specialColor;
        return fluid.getTintColor();
    }

    /**
     * Client only
     *
     * @return tint color to be applied on the fluid texture
     */
    public int getFlowingColor(@Nullable BlockAndTintGetter world, @Nullable BlockPos pos) {
        SoftFluid.TintMethod method = this.fluid().getTintMethod();
        if (method == SoftFluid.TintMethod.FLOWING) return this.getParticleColor(world, pos);
        else return this.getStillColor(world, pos);
    }

    /**
     * Client only
     *
     * @return tint color to be used on particle. Differs from getTintColor since it returns an mixWith color extrapolated from their fluid textures
     */
    public int getParticleColor(@Nullable BlockAndTintGetter world, @Nullable BlockPos pos) {
        int tintColor = getStillColor(world, pos);
        //if tint color is white gets averaged color
        if (tintColor == -1) return this.fluid().getAverageTextureTintColor();
        return tintColor;
    }

    @Override
    public @NotNull PatchedDataComponentMap getComponents() {
        return this.components;
    }

    @Nullable
    public <T> T set(DataComponentType<? super T> type, @Nullable T component) {
        return this.components.set(type, component);
    }


}
