package net.mehvahdjukaar.moonlight.api.fluids;

import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.DynamicOps;
import dev.architectury.injectables.annotations.ExpectPlatform;
import java.util.Optional;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.minecraft.class_1268;
import net.minecraft.class_1271;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1920;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2487;
import net.minecraft.class_2520;
import net.minecraft.class_3414;
import net.minecraft.class_3419;
import net.minecraft.class_3468;
import net.minecraft.class_3532;
import net.minecraft.class_6903;
import net.minecraft.class_7225;
import net.minecraft.class_7871;
import net.minecraft.class_9129;
import net.minecraft.class_9135;
import net.minecraft.class_9139;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * instance this fluid tank in your tile entity
 */
@SuppressWarnings("unused")
public class SoftFluidTank {

    public static final class_9139<class_9129, SoftFluidTank> STREAM_CODEC = new class_9139<>() {
        @Override
        public SoftFluidTank decode(class_9129 object) {
            int capacity = class_9135.field_49675.decode(object);
            SoftFluidStack stack = SoftFluidStack.STREAM_CODEC.decode(object);
            return SoftFluidTank.create(stack, capacity, object.method_56349()
                    .method_46759(SoftFluidRegistry.KEY).get());
        }

        @Override
        public void encode(class_9129 object, SoftFluidTank object2) {
            class_9135.field_49675.encode(object, object2.getCapacity());
            SoftFluidStack.STREAM_CODEC.encode(object, object2.getFluid());
        }
    };

    public static final Codec<SoftFluidTank> CODEC = new Codec<>() {

        @Override
        public <T> DataResult<Pair<SoftFluidTank, T>> decode(DynamicOps<T> ops, T input) {
            if (ops instanceof class_6903<?> registryOps) {
                var reg = registryOps.method_46634(SoftFluidRegistry.KEY);
                if (reg.isEmpty()) {
                    return DataResult.error(() -> "Failed to find registry from registry lookup!");
                }
                DataResult<Pair<SoftFluidStack, T>> result1 = SoftFluidStack.CODEC.decode(ops, input);
                DataResult<Pair<Integer, T>> result2 = Codec.INT.decode(ops, input);
                return result1.flatMap(r1 -> result2.map(r2 -> {
                    SoftFluidTank tank = SoftFluidTank.create(r1.getFirst(), r2.getFirst(), reg.get());
                    return Pair.of(tank, r2.getSecond());
                }));
            } else {
                return DataResult.error(() -> "Registry ops required!");
            }
        }

        @Override
        public <T> DataResult<T> encode(SoftFluidTank input, DynamicOps<T> ops, T prefix) {
            DataResult<T> result1 = SoftFluidStack.CODEC.encode(input.getFluid(), ops, prefix);
            DataResult<T> result2 = Codec.INT.encode(input.getCapacity(), ops, prefix);
            if (result1.error().isPresent()) return result1;
            if (result2.error().isPresent()) return result2;
            return DataResult.success(prefix);
        }
    };

    public static final int BOTTLE_COUNT = 1;
    public static final int BOWL_COUNT = 2;
    public static final int BUCKET_COUNT = 4;


    //so we don't need to ask for this on every darn operation
    private final class_7871<SoftFluid> fluidReg;

    protected final int capacity;
    protected @NotNull SoftFluidStack fluidStack;

    //Minor optimization. Caches the tint color for the fluid
    protected int stillTintCache = 0;
    protected int flowingTintCache = 0;
    protected int particleTintCache = 0;
    protected boolean needsColorRefresh = true;

    protected SoftFluidTank(int capacity, class_7225.class_7874 registries) {
        this(capacity, registries.method_46762(SoftFluidRegistry.KEY));
    }

    protected SoftFluidTank(int capacity, class_7871<SoftFluid> fluidReg) {
        this.capacity = capacity;
        this.fluidReg = fluidReg;
        this.fluidStack = SoftFluidStack.empty(fluidReg);
    }

    @Deprecated(forRemoval = true)
    protected SoftFluidTank(int capacity) {
        this(capacity, Utils.hackyGetRegistryAccess());
    }

    public static SoftFluidTank create(int capacity, class_7225.class_7874 registries) {
        return create(capacity, registries.method_46762(SoftFluidRegistry.KEY));
    }

    @ExpectPlatform
    public static SoftFluidTank create(int capacity, class_7871<SoftFluid> fluidReg) {
        throw new AssertionError();
    }

    @Deprecated(forRemoval = true)
    public static SoftFluidTank create(int capacity) {
        return create(capacity, SoftFluidRegistry.get(
                Utils.hackyGetRegistryAccess()).method_46771());
    }

    @Deprecated(forRemoval = true)
    public static SoftFluidTank create(SoftFluidStack stack, int capacity) {
        return create(stack, capacity,
                SoftFluidRegistry.get(Utils.hackyGetRegistryAccess()).method_46771());
    }

    public static SoftFluidTank create(SoftFluidStack stack, int capacity, class_7871<SoftFluid> fluidReg) {
        SoftFluidTank tank = create(capacity, fluidReg);
        tank.setFluid(stack);
        return tank;
    }

    public SoftFluidTank makeCopy() {
        SoftFluidTank tank = create(this.capacity, this.fluidReg);
        tank.copyContent(this);
        return tank;
    }

    /**
     * call this method from your block when the player interacts. tries to fill or empty current held item in tank
     *
     * @param player player
     * @param hand   hand
     * @return interaction successful
     */
    public boolean interactWithPlayer(class_1657 player, class_1268 hand, @Nullable class_1937 world, @Nullable class_2338 pos) {
        class_1799 handStack = player.method_5998(hand);

        class_1799 returnStack = this.interactWithItem(handStack, world, pos, false);
        //for items that have no bottle
        if (returnStack != null) {
            Utils.swapItem(player, hand, returnStack);

            if (!handStack.method_7960()) player.method_7259(class_3468.field_15372.method_14956(handStack.method_7909()));
            return true;
        }
        return false;
    }

    /**
     * makes current item interact with fluid tank. returns empty stack if
     *
     * @param stack ItemStack to be interacted with
     * @param world world. null if no sound is to be played
     * @param pos   position. null if no sound is to be played
     * @return resulting ItemStack: empty for empty hand return, null if it failed
     */
    @Nullable
    public class_1799 interactWithItem(class_1799 stack, class_1937 world, @Nullable class_2338 pos, boolean simulate) {
        class_1799 returnStack;
        //try filling
        var fillResult = this.fillItem(stack, world, pos, simulate);
        if (fillResult.method_5467().method_23665()) return fillResult.method_5466();
        //try emptying
        var drainResult = this.drainItem(stack, world, pos, simulate);
        if (drainResult.method_5467().method_23665()) return drainResult.method_5466();

        return null;
    }

    public class_1271<class_1799> drainItem(class_1799 filledContainerStack, @Nullable class_1937 world, @Nullable class_2338 pos, boolean simulate) {
        return drainItem(filledContainerStack, world, pos, simulate, true);
    }

    /**
     * Tries pouring the content of provided item in the tank
     * also plays sound.
     * If simulate is true, it will return the same item as normal but wont alter the container state
     *
     * @return empty container item, PASS if it failed
     */
    public class_1271<class_1799> drainItem(class_1799 filledContainer, class_1937 level, @Nullable class_2338 pos, boolean simulate, boolean playSound) {
        var extracted = SoftFluidStack.fromItem(filledContainer);
        if (extracted == null) return class_1271.method_22430(class_1799.field_8037);
        SoftFluidStack fluidStack = extracted.getFirst();

        //if it can add all of it
        if (addFluid(fluidStack, true) == fluidStack.getCount()) {
            FluidContainerList.Category category = extracted.getSecond();

            class_1799 emptyContainer = category.getEmptyContainer().method_7854();
            if (!simulate) {

                addFluid(fluidStack, false);

                class_3414 sound = category.getEmptySound();
                if (sound != null && pos != null) {
                    level.method_8396(null, pos, sound, class_3419.field_15245, 1, 1);
                }
            }
            return class_1271.method_29237(emptyContainer, level.field_9236);
        }
        return class_1271.method_22430(class_1799.field_8037);

    }


    public class_1271<class_1799> fillItem(class_1799 emptyContainer, @Nullable class_1937 world, @Nullable class_2338 pos, boolean simulate) {
        return fillItem(emptyContainer, world, pos, simulate, true);
    }

    /**
     * tries removing said amount of fluid and returns filled item
     * also plays sound
     *
     * @return filled bottle item. null if it failed or if simulated is true and failed
     */
    public class_1271<class_1799> fillItem(class_1799 emptyContainer, class_1937 level, @Nullable class_2338 pos, boolean simulate, boolean playSound) {
        var pair = this.fluidStack.splitToItem(emptyContainer);

        if (pair != null) {
            FluidContainerList.Category category = pair.getSecond();
            class_3414 sound = category.getEmptySound();
            if (sound != null && pos != null) {
                level.method_8396(null, pos, sound, class_3419.field_15245, 1, 1);
            }
            return class_1271.method_29237(pair.getFirst(), level.field_9236);
        }
        return class_1271.method_22430(class_1799.field_8037);
    }

    /**
     * Called when talk is not empty and a new fluid is added. For most uses just increments the existing one but could alter the fluid content
     * You can assume that canAddSoftFluid has been called before
     */
    protected void addFluidOntoExisting(SoftFluidStack stack) {
        this.fluidStack.grow(stack.getCount());
    }

    /**
     * tries removing bottle amount and returns filled bottle
     *
     * @return filled bottle item. null if it failed
     */
    @Nullable
    public class_1271<class_1799> fillBottle(class_1937 world, class_2338 pos) {
        return fillItem(class_1802.field_8469.method_7854(), world, pos, false);
    }

    /**
     * tries removing bucket amount and returns filled bucket
     *
     * @return filled bucket item. null if it failed
     */
    @Nullable
    public class_1271<class_1799> fillBucket(class_1937 world, class_2338 pos) {
        return fillItem(class_1802.field_8550.method_7854(), world, pos, false);
    }

    /**
     * tries removing bowl amount and returns filled bowl
     *
     * @return filled bowl item. null if it failed
     */
    @Nullable
    public class_1271<class_1799> fillBowl(class_1937 world, class_2338 pos) {
        return fillItem(class_1802.field_8428.method_7854(), world, pos, false);
    }

    /**
     * Check if fluid TYPE is compatible with content. Does not care about count
     */
    public boolean isFluidCompatible(SoftFluidStack fluidStack) {
        return this.fluidStack.isSameFluidSameComponents(fluidStack) || this.isEmpty();
    }

    /**
     * Like addFluid but can add only a part of the stack
     *
     * @return amount of fluid added. Given stack WILL be modified by subtracting that amount
     */
    public int addFluid(SoftFluidStack stack, boolean simulate) {
        if (!isFluidCompatible(stack)) return 0;
        int space = this.getSpace();
        if (space == 0) return 0;

        int amount = Math.min(space, stack.getCount());

        if (simulate) return amount;

        var toAdd = stack.split(amount);
        if (this.isEmpty()) {
            this.setFluid(toAdd);
        } else {
            this.addFluidOntoExisting(toAdd);
        }
        return amount;
    }

    /**
     * removes fluid from the tank
     *
     * @param amount   amount to remove
     * @param simulate if true, it will not actually remove the fluid
     * @return removed fluid
     */
    public SoftFluidStack removeFluid(int amount, boolean simulate) {
        if (this.isEmpty()) return SoftFluidStack.empty(this.fluidReg);
        int toRemove = Math.min(amount, this.fluidStack.getCount());
        SoftFluidStack stack = this.fluidStack.copyWithCount(toRemove);
        if (!simulate) {
            this.fluidStack.shrink(toRemove);
        }
        return stack;
    }

    /**
     * Transfers between 2 soft fluid tanks
     */
    @Deprecated(forRemoval = true)
    public boolean transferFluid(SoftFluidTank destination) {
        return this.transferFluid(destination, BOTTLE_COUNT);
    }

    //transfers between two fluid holders
    //I forgot why this was deprecated
    @Deprecated(forRemoval = true)
    public boolean transferFluid(SoftFluidTank destination, int amount) {
        if (this.isEmpty()) return false;
        var removed = this.removeFluid(amount, false);
        if (destination.addFluid(removed, true) == removed.getCount()) {
            destination.addFluid(removed, false);
            return true;
        }
        return false;
    }

    public int getSpace() {
        return Math.max(0, capacity - fluidStack.getCount());
    }

    public int getFluidCount() {
        return fluidStack.getCount();
    }

    public boolean isFull() {
        return fluidStack.getCount() == this.capacity;
    }

    public boolean isEmpty() {
        //count 0 should always = to fluid.empty
        return this.fluidStack.isEmpty();
    }

    /**
     * gets liquid height for renderer
     *
     * @param maxHeight maximum height in blocks
     * @return fluid height
     */
    public float getHeight(float maxHeight) {
        return maxHeight * fluidStack.getCount() / this.capacity;
    }

    /**
     * @return comparator block redstone power
     */
    public int getComparatorOutput() {
        float f = fluidStack.getCount() / (float) this.capacity;
        return class_3532.method_15375(f * 14.0F) + 1;
    }

    public SoftFluidStack getFluid() {
        return fluidStack;
    }

    public SoftFluid getFluidValue() {
        return fluidStack.getHolder().comp_349();
    }

    public void setFluid(SoftFluidStack fluid) {
        this.fluidStack = fluid;
        refreshTintCache();
    }

    public void refreshTintCache() {
        stillTintCache = 0;
        needsColorRefresh = true;
    }

    private void fillCount() {
        this.fluidStack.setCount(this.capacity);
    }

    /**
     * resets & clears the tank
     */
    public void clear() {
        this.setFluid(SoftFluidStack.empty(this.fluidReg));
    }

    /**
     * copies the content of a fluid tank into this
     *
     * @param other other tank
     */
    public void copyContent(SoftFluidTank other) {
        SoftFluidStack stack = other.getFluid();
        this.setFluid(stack.copyWithCount(Math.min(this.capacity, stack.getCount())));
    }

    public int getCapacity() {
        return capacity;
    }

    public void capCapacity() {
        this.fluidStack.setCount(class_3532.method_15340(this.fluidStack.getCount(), 0, capacity));
    }

    private void cacheColors(@Nullable class_1920 world, @Nullable class_2338 pos) {
        stillTintCache = this.fluidStack.getStillColor(world, pos);
        flowingTintCache = this.fluidStack.getFlowingColor(world, pos);
        particleTintCache = this.fluidStack.getParticleColor(world, pos);
        needsColorRefresh = false;
    }

    public int getCachedStillColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        if (needsColorRefresh) cacheColors(world, pos);
        return stillTintCache;
    }

    public int getCachedFlowingColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        if (needsColorRefresh) cacheColors(world, pos);
        return flowingTintCache;
    }

    public int getCachedParticleColor(@Nullable class_1920 world, @Nullable class_2338 pos) {
        if (needsColorRefresh) cacheColors(world, pos);
        return particleTintCache;
    }

    /**
     * @return true if contained fluid has associated food
     */
    public boolean containsFood() {
        return !this.fluidStack.getFoodProvider().isEmpty();
    }

    /**
     * call from tile entity. loads tank from nbt
     *
     * @param compound nbt
     */
    public void load(class_2487 compound, class_7225.class_7874 registries) {
        class_2487 fluidTag = compound.method_10562("fluid");
        //TODO: remove this in 1.22
        class_2487 backCompat = compound.method_10562("FluidHolder");
        if (!backCompat.method_33133()) {
            fluidTag = backCompat;
        }
        if (!fluidTag.method_33133()) {
            this.setFluid(SoftFluidStack.load(registries, fluidTag));
        }
    }

    @Deprecated(forRemoval = true)
    public void load(class_2487 compound) {
        this.load(compound, Utils.hackyGetRegistryAccess());
    }

    @Deprecated(forRemoval = true)
    public class_2487 save(class_2487 compound) {
        this.save(compound, Utils.hackyGetRegistryAccess());
        return compound;
    }

    /**
     * call from tile entity. saves to nbt
     *
     * @param compound nbt
     */
    public void save(class_2487 compound, class_7225.class_7874 registries) {
        this.setFluid(this.fluidStack);
        class_2520 tag = this.fluidStack.save(registries);
        compound.method_10566("fluid", tag);
    }

    /**
     * makes player drink 1 bottle and removes it from the tank
     *
     * @param player player
     * @param world  world
     * @return success
     */
    public boolean tryDrinkUpFluid(class_1657 player, class_1937 world) {
        if (!this.isEmpty() && this.containsFood()) {
            if (this.fluidStack.getFoodProvider().consume(player, world, fluidStack::copyComponentsTo)) { //crap code right there
                fluidStack.shrink(1);
                return true;
            }
        }
        return false;
    }


    //util functions
    public static int getLiquidCountFromItem(class_1792 i) {
        if (i == class_1802.field_8469) {
            return BOTTLE_COUNT;
        } else if (i == class_1802.field_8428) {
            return BOWL_COUNT;
        } else if (i == class_1802.field_8550) {
            return BUCKET_COUNT;
        }
        return 0;
    }

}
