package net.mehvahdjukaar.moonlight.api.resources.pack;

import com.google.common.base.Stopwatch;
import net.mehvahdjukaar.moonlight.api.events.EarlyPackReloadEvent;
import net.mehvahdjukaar.moonlight.api.events.MoonlightEventsHelper;
import net.mehvahdjukaar.moonlight.api.misc.IProgressTracker;
import net.mehvahdjukaar.moonlight.api.misc.IProgressTracker.Task;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.resources.ResType;
import net.mehvahdjukaar.moonlight.api.resources.SimpleTagBuilder;
import net.mehvahdjukaar.moonlight.api.resources.StaticResource;
import net.mehvahdjukaar.moonlight.core.CommonConfigs;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3283;
import net.minecraft.class_3300;
import net.minecraft.class_3302;
import net.minecraft.class_3695;
import net.minecraft.class_6862;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import java.util.function.Function;

public abstract class DynResourceGenerator<T extends DynamicResourcePack> implements class_3302 {

    private static final ExecutorService EXECUTOR_SERVICE = Executors.newCachedThreadPool();


    public final T dynamicPack;
    protected final String modId;
    private boolean hasBeenInitialized;

    //creates this object and registers it
    protected DynResourceGenerator(T pack, String modId) {
        this.dynamicPack = pack;
        this.modId = modId;
        this.dynamicPack.registerPack();

        GENERATORS.add(this);
    }

    /**
     * Called on Mod Init
     * Yes this just loads the class
     */
    public final void register() {
    }

    public abstract Logger getLogger();

    public T getPack() {
        return dynamicPack;
    }

    /**
     * If this pack should be cleared on reload. Overrie if you need to have your pack never cleared, for example when making a pack hat just loads once
     */
    public boolean shouldClearOnReload() {
        return runsOnEveryReload();
    }

    public boolean runsOnEveryReload() {
        return true;
    }

    /**
     * just deprecated as it shouldn't be overwritten anymore and will become final private
     */
    private void regenerateDynamicAssets(class_3300 manager, IProgressTracker progressTracker) {
        var genTasks = new ArrayList<ResourceGenTask>();

        Stopwatch watch = Stopwatch.createStarted();
        try {
            regenerateDynamicAssets(manager);
        } catch (Exception e) {
            getLogger().error("Legacy dynamic gen task failed: ", e);
        }
        try {
            regenerateDynamicAssets(genTasks::add);
        } catch (Exception e) {
            getLogger().error("Failed to add tasks to dynamic resource gen: ", e);
        }

        int totalTasks = genTasks.size();
        var reporter = progressTracker.subtask(totalTasks);

        List<CompletableFuture<ResourceSink>> futures = genTasks.stream()
                .map(task -> CompletableFuture.supplyAsync(() -> {
                    try {
                        ResourceSink localSink = createLocalSink();
                        task.accept(manager, localSink);
                        return localSink;
                    } catch (Exception e) {
                        getLogger().error("Resource Gen Task failed", e);
                        return null; // Or a special "empty" sink if you want to keep track
                    } finally {
                        reporter.step();
                    }
                }, getExecutors()))
                .toList();

        // Wait for all tasks to finish, even if some failed
        CompletableFuture<Void> allDone =
                CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));

        allDone.join();

        List<ResourceSink> successfulSinks = futures.stream()
                .map(CompletableFuture::join)
                .filter(Objects::nonNull) // Remove failed ones
                .toList();

        addAllResourceSinks(successfulSinks);

        getLogger().info("Generated runtime {} for pack {} ({}) in: {} ms{} (multithreaded)",
                this.dynamicPack.getPackType(), this.dynamicPack.method_14409(), this.modId,
                watch.elapsed().toMillis(),
                this.dynamicPack.generateDebugResources ? " (debug resource dump on)" : "");
    }

    protected void addAllResourceSinks(List<ResourceSink> sinks) {
        Map<class_6862<?>, SimpleTagBuilder> allTags = new HashMap<>();
        for (ResourceSink sink : sinks) {
            sink.resources.forEach(this.dynamicPack::addBytes);
            sink.notClearable.forEach(this.dynamicPack::markNotClearable);
            for (var e : sink.tags.entrySet()) {
                allTags.merge(e.getKey(), e.getValue(), SimpleTagBuilder::merge);
            }
        }

        //adds tags
        for (var e : allTags.entrySet()) {
            this.dynamicPack.addTag(e.getValue(), e.getKey().comp_326());
        }
    }

    //override if you really need to
    protected @NotNull ResourceSink createLocalSink() {
        return new ResourceSink(this.modId, this.dynamicPack.method_14409());
    }

    protected @NotNull ExecutorService getExecutors() {
        return EXECUTOR_SERVICE;
    }

    @Deprecated(forRemoval = true)
    public void regenerateDynamicAssets(class_3300 manager) {

    }

    //TODO: make abstract
    public void regenerateDynamicAssets(Consumer<ResourceGenTask> executor) {
        //implement this for multi thead
    }

    @Override
    public final @NotNull CompletableFuture<Void> method_25931(class_4045 stage, class_3300 manager,
                                                         class_3695 workerProfiler, class_3695 mainProfiler,
                                                         Executor workerExecutor, Executor mainExecutor) {
        //not used anymore. Loading early instead
        if (Moonlight.HAS_BEEN_INIT && PlatHelper.isModLoadingValid()) { //fail safe since some mods for some god damn reason run a reload event before blocks are registered...
            onNormalReload(manager);
        } else {
            Moonlight.LOGGER.error("Cowardly refusing generate assets for a broken mod state");
        }

        return CompletableFuture.supplyAsync(() -> null, workerExecutor)
                .thenCompose(stage::method_18352)
                .thenAcceptAsync((noResult) -> {
                }, mainExecutor);
    }

    protected void onNormalReload(class_3300 manager) {
    }

    protected final void onEarlyReload(EarlyPackReloadEvent event, IProgressTracker localReporter) {
        if (event.type() == dynamicPack.packType) {
            try {
                this.reloadResources(event.manager(), localReporter);
            } catch (Exception e) {
                Moonlight.LOGGER.error("An error occurred while trying to generate dynamic assets for {}:", this.dynamicPack, e);
            }
        }
    }

    protected final void reloadResources(class_3300 manager, IProgressTracker reporter) {
        //first clear all pack content if it should be cleared

        boolean wasFirstReload = false;
        if (!this.hasBeenInitialized) {
            wasFirstReload = true;
            this.hasBeenInitialized = true;
            if (this.dynamicPack instanceof DynamicTexturePack tp) tp.addPackLogo();
        }
        //generate textures
        if (runsOnEveryReload() || wasFirstReload) {
            Moonlight.LOGGER.info("Generating runtime assets for pack {} ({})", this.dynamicPack.method_14409(), this.modId);
            this.regenerateDynamicAssets(manager, reporter);
        }
    }


    @Nullable
    protected abstract class_3283 getRepository();

    @Deprecated(forRemoval = true)
    public boolean alreadyHasAssetAtLocation(class_3300 manager, class_2960 res, ResType type) {
        return alreadyHasAssetAtLocation(manager, type.getPath(res));
    }

    @Deprecated(forRemoval = true)
    public boolean alreadyHasAssetAtLocation(class_3300 manager, class_2960 res) {
        var resource = manager.method_14486(res);
        return resource.filter(value -> !value.method_14480().equals(this.dynamicPack.method_14409())).isPresent();
    }

    /**
     * This is a handy method for dynamic resource pack since it allows to specify the name of an existing resource
     * that will then be copied and modified replacing a certain keyword in it with another.
     * This is useful when adding new woodtypes as one can simply manually add a default wood json and provide the method with the
     * default woodtype name and the target name
     * The target location will the one of this pack while its path will be the original one modified following the same principle as the json itself
     *
     * @param resource    target resource that will be copied, modified and saved back
     * @param keyword     keyword to replace
     * @param replaceWith word to replace the keyword with
     */
    @Deprecated(forRemoval = true)
    public void addSimilarJsonResource(class_3300 manager, StaticResource resource, String keyword, String replaceWith) throws NoSuchElementException {
        addSimilarJsonResource(manager, resource, s -> s.replace(keyword, replaceWith));
    }

    @Deprecated(forRemoval = true)
    public void addSimilarJsonResource(class_3300 manager, StaticResource resource, Function<String, String> textTransform) throws NoSuchElementException {
        addSimilarJsonResource(manager, resource, textTransform, textTransform);
    }

    @Deprecated(forRemoval = true)
    public void addSimilarJsonResource(class_3300 manager, StaticResource resource, Function<String, String> textTransform, Function<String, String> pathTransform) throws NoSuchElementException {
        class_2960 fullPath = resource.location;

        //calculates new path
        StringBuilder builder = new StringBuilder();
        String[] partial = fullPath.method_12832().split("/");
        for (int i = 0; i < partial.length; i++) {
            if (i != 0) builder.append("/");
            if (i == partial.length - 1) {
                builder.append(pathTransform.apply(partial[i]));
            } else builder.append(partial[i]);
        }
        //adds modified under my namespace
        class_2960 newRes = new class_2960(this.modId, builder.toString());
        if (!alreadyHasAssetAtLocation(manager, newRes)) {

            String fullText = resource.asString();


            fullText = textTransform.apply(fullText);

            this.dynamicPack.addBytes(newRes, fullText.getBytes());
        }
    }

    @Deprecated(forRemoval = true)
    public void addResourceIfNotPresent(class_3300 manager, StaticResource resource) {
        if (!alreadyHasAssetAtLocation(manager, resource.location)) {
            this.dynamicPack.addResource(resource);
        }
    }


    private static final Set<DynResourceGenerator<?>> GENERATORS = new HashSet<>();

    static {
        MoonlightEventsHelper.addListener(earlyPackReloadEvent -> {
            class_3264 type = earlyPackReloadEvent.type();
            List<DynResourceGenerator<?>> validGen = DynResourceGenerator.GENERATORS.stream()
                    .filter(gen -> gen.dynamicPack.packType == type)
                    .toList();
            List<String> modIds = GENERATORS.stream()
                    .map(g -> g.modId).toList();
            Moonlight.LOGGER.info("Starting runtime resource generation for pack type {} with generators from mods {}: {}",
                    type, modIds, validGen);

            if (CommonConfigs.EXTRA_DEBUG.get())
                Moonlight.LOGGER.info("Current stack trace:", new Throwable("EXTRA_DEBUG is enabled. Stack trace dump to see who fired me"));


            Stopwatch stopwatch = Stopwatch.createStarted();

            IProgressTracker reporter = earlyPackReloadEvent.progress();
            //These are not parallel. pass flat
            for (var gen : validGen) {
                gen.onEarlyReload(earlyPackReloadEvent, reporter); // run synchronously
            }

            Moonlight.LOGGER.info("Finished runtime resources generation for {} packs in a total of {} ms",
                    GENERATORS.size(), stopwatch.elapsed().toMillis());
        }, EarlyPackReloadEvent.class);
    }

    @ApiStatus.Internal
    public static void clearAfterReload(class_3264 targetType) {
        //this will be called multiple times. shunt be an issue I hope
        Set<DynamicResourcePack> packs = new HashSet<>();
        for (var g : GENERATORS) {
            if (g.dynamicPack.packType == targetType && g.shouldClearOnReload()) {
                packs.add(g.dynamicPack);
            }
        }
        for (var p : packs) {
            p.clearNonStatic();
        }
    }

    @ApiStatus.Internal
    public static void clearBeforeReload(class_3264 targetType) {
        Set<DynamicResourcePack> packs = new HashSet<>();
        for (var g : GENERATORS) {
            if (g.dynamicPack.packType == targetType && g.shouldClearOnReload()) {
                packs.add(g.dynamicPack);
            }
        }
        for (var p : packs) {
            p.clearAllContent();
        }
    }

}
