package net.mehvahdjukaar.moonlight.api.map.decoration;

import com.mojang.datafixers.Products;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;
import java.util.Optional;
import net.minecraft.class_22;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.minecraft.class_6880;
import net.minecraft.class_8824;

/**
 * represents a block tracker instance which keeps track of a placed block and creates its associated map decoration
 * You can extend this class to add extra data to your markers
 *
 * @param <D> decoration
 */
public abstract class MLMapMarker<D extends MLMapDecoration> {
    //Static ref is fine as data registry cant change with reload command. Just don't hold ref outside of a world

    private final class_6880<MLMapDecorationType<?, ?>> type;
    @NotNull
    protected final class_2338 pos;
    protected final float rot;
    protected final Optional<class_2561> name;

    protected final boolean preventsExtending;
    protected final boolean shouldRefresh;
    protected final boolean shouldSave;

    public static final Codec<MLMapMarker<?>> REFERENCE_CODEC =
            MLMapDecorationType.CODEC.dispatch("type", MLMapMarker::getType,
                    mapWorldMarker -> mapWorldMarker.comp_349().getMarkerCodec());

    public static <T extends MLMapMarker<?>> Products.P7<RecordCodecBuilder.Mu<T>, class_6880<MLMapDecorationType<?, ?>>, class_2338, Float, Optional<class_2561>, Optional<Boolean>, Optional<Boolean>, Boolean> baseCodecGroup(
            RecordCodecBuilder.Instance<T> instance) {
        return instance.group(
                MLMapDecorationType.CODEC.fieldOf("type").forGetter(m -> m.getType()),
                class_2338.field_25064.fieldOf("pos").forGetter(m -> m.getPos()),
                Codec.FLOAT.optionalFieldOf("rot", 0f).forGetter(m -> m.getRotation()),
                class_8824.field_46598.optionalFieldOf("name").forGetter(m -> m.getDisplayName()),
                Codec.BOOL.optionalFieldOf("should_refresh").forGetter(m -> Optional.of(m.shouldRefreshFromWorld())),
                Codec.BOOL.optionalFieldOf("should_save").forGetter(m -> Optional.of(m.shouldSave())),
                Codec.BOOL.optionalFieldOf("prevents_extending", false).forGetter(m -> m.preventsExtending())
        );
    }

    public MLMapMarker(class_6880<MLMapDecorationType<?, ?>> type, class_2338 pos,
                       float rotation, Optional<class_2561> component,
                       Optional<Boolean> shouldRefresh, Optional<Boolean> shouldSave, boolean preventsExtending) {
        this.type = type;
        this.pos = pos;
        this.rot = rotation;
        this.name = component;

        this.shouldRefresh = shouldRefresh.orElse(type.comp_349().isFromWorld());
        this.shouldSave = shouldSave.orElse(type.comp_349().isFromWorld());
        this.preventsExtending = preventsExtending;
    }

    public class_6880<MLMapDecorationType<?, ?>> getType() {
        return type;
    }

    public boolean shouldRefreshFromWorld() {
        return shouldRefresh;
    }

    public boolean shouldSave() {
        return shouldSave;
    }

    public boolean preventsExtending() {
        return false;
    }

    /**
     * implement if you are adding extra data
     * see default example for an implementation
     *
     * @param o another marker object
     * @return true if they are equal
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        MLMapMarker<?> that = (MLMapMarker<?>) o;
        return Objects.equals(type, that.type) && Objects.equals(pos, that.pos) && Objects.equals(name, that.name);
    }

    /**
     * implement if you are adding extra data
     *
     * @return hash
     */
    @Override
    public int hashCode() {
        return Objects.hash(type, pos, name);
    }

    /**
     * ids have to be unique so add here all the data you have in your marker to tell them apart
     *
     * @return suffix
     */
    public String getMarkerUniqueId() {
        return this.type.method_55840() + "-" + pos.method_10263() + "," + pos.method_10264() + "," + pos.method_10260();
    }

    public class_2338 getPos() {
        return this.pos;
    }

    public float getRotation() {
        return rot;
    }

    public Optional<class_2561> getDisplayName() {
        return name;
    }

    /**
     * creates a decoration given its map position and rotation
     *
     * @param mapX map x position
     * @param mapY map y position
     * @param rot  decoration rotation
     * @return decoration instance
     */
    @NotNull
    protected abstract D doCreateDecoration(byte mapX, byte mapY, byte rot);

    /**
     * Creates a decoration from this marker.
     * This its default vanilla implementation.<br>
     * You can do here extra check for a dimension type and so on.<br>
     * For everything else, just implement doCreateDecoration
     *
     * @return new decoration instance
     */
    @Nullable
    public D createDecorationFromMarker(class_22 data) {
        class_2338 pos = this.getPos();
        if (pos == null) return null;
        double worldX = pos.method_10263();
        double worldZ = pos.method_10260();
        double rotation = this.getRotation();
        int i = 1 << data.field_119;
        float f = (float) (worldX - data.field_116) / i;
        float f1 = (float) (worldZ - data.field_115) / i;
        byte mapX = (byte) ((int) ((f * 2.0F) + 0.5D));
        byte mapY = (byte) ((int) ((f1 * 2.0F) + 0.5D));
        byte rot;
        if (f >= -64.0F && f1 >= -64.0F && f <= 64.0F && f1 <= 64.0F) {
            rotation = rotation + (rotation < 0.0D ? -8.0D : 8.0D);
            rot = (byte) ((int) (rotation * 16.0D / 360.0D));
            return doCreateDecoration(mapX, mapY, rot);
        }
        return null;
    }

    // override to give special behaviors
    public int getFlags() {
        return 0;
    }

    public boolean hasFlag(int flag) {
        return (getFlags() & flag) != 0;
    }

    public static final int HAS_SMALL_TEXTURE_FLAG = 1;

}
