package net.mehvahdjukaar.moonlight.api.resources;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.internal.Streams;
import com.google.gson.stream.JsonWriter;
import com.mojang.serialization.JsonOps;
import net.mehvahdjukaar.moonlight.api.client.TextureCache;
import net.mehvahdjukaar.moonlight.api.misc.SearchTrie;
import net.mehvahdjukaar.moonlight.api.resources.pack.DynamicTexturePack;
import net.mehvahdjukaar.moonlight.api.resources.pack.ResourceSink;
import net.mehvahdjukaar.moonlight.api.set.BlockType;
import net.mehvahdjukaar.moonlight.api.set.wood.WoodType;
import net.mehvahdjukaar.moonlight.api.set.wood.WoodTypeRegistry;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.minecraft.class_1792;
import net.minecraft.class_1860;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_799;
import net.minecraft.class_8786;
import org.jetbrains.annotations.NotNull;

import javax.management.openmbean.InvalidOpenTypeException;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.Map.Entry;
import java.util.function.Consumer;
import java.util.function.Predicate;

public class RPUtils {

    public static String serializeJson(JsonElement json) throws IOException {
        try (StringWriter stringWriter = new StringWriter();
             JsonWriter jsonWriter = new JsonWriter(stringWriter)) {

            jsonWriter.setLenient(true);
            jsonWriter.setIndent("  ");

            Streams.write(json, jsonWriter);

            return stringWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    //remember to close this stream
    public static JsonObject deserializeJson(InputStream stream) {
        return class_3518.method_15255(new InputStreamReader(stream, StandardCharsets.UTF_8));
    }

    public static class_2960 findFirstBlockTextureLocation(class_3300 manager, class_2248 block) throws FileNotFoundException {
        return findFirstBlockTextureLocation(manager, block, t -> true);
    }

    /**
     * Grabs the first texture from a given block
     *
     * @param manager          resource manager
     * @param block            target block
     * @param texturePredicate predicate that will be applied to the texture name
     * @return found texture location
     */
    public static class_2960 findFirstBlockTextureLocation(class_3300 manager, class_2248 block, Predicate<String> texturePredicate) throws FileNotFoundException {
        var cached = TextureCache.getCached(block, texturePredicate);
        if (cached != null) {
            return class_2960.method_60654(cached);
        }
        class_2960 blockId = Utils.getID(block);
        var blockState = manager.method_14486(ResType.BLOCKSTATES.getPath(blockId));
        try (var bsStream = blockState.orElseThrow().method_14482()) {

            JsonElement bsElement = RPUtils.deserializeJson(bsStream);
            //grabs the first resource location of a model
            Set<String> models = findAllResourcesInJsonRecursive(bsElement.getAsJsonObject(), s -> s.equals("model"));

            for (var modelPath : models) {
                List<String> textures = findAllTexturesInModelRecursive(manager, modelPath);

                for (var t : textures) {
                    TextureCache.add(block, t);
                    if (texturePredicate.test(t)) return class_2960.method_60654(t);
                }
            }
        } catch (Exception ignored) {
        }
        //if texture is not there try to guess location. Hack for better end
        var hack = guessBlockTextureLocation(blockId, block);
        for (var t : hack) {
            TextureCache.add(block, t);
            if (texturePredicate.test(t)) return class_2960.method_60654(t);
        }

        throw new FileNotFoundException("Could not find any texture associated to the given block " + blockId);
    }

    //if texture is not there try to guess location. Hack for better end
    private static Set<String> guessItemTextureLocation(class_2960 id, class_1792 item) {
        return Set.of(id.method_12836() + ":item/" + item);
    }

    private static List<String> guessBlockTextureLocation(class_2960 id, class_2248 block) {
        String name = id.method_12832();
        List<String> textures = new ArrayList<>();
        //just works for wood types
        WoodType w = WoodTypeRegistry.INSTANCE.getBlockTypeOf(block);
        if (w != null) {
            String key = w.getChildKey(block);
            if (Objects.equals(key, "log") || Objects.equals(key, "stripped_log")) {
                textures.add(id.method_12836() + ":block/" + name + "_top");
                textures.add(id.method_12836() + ":block/" + name + "_side");
            }
        }
        textures.add(id.method_12836() + ":block/" + name);
        return textures;
    }

    @NotNull
    private static List<String> findAllTexturesInModelRecursive(class_3300 manager, String modelPath) throws Exception {
        JsonObject modelElement;
        try (var modelStream = manager.method_14486(ResType.MODELS.getPath(modelPath)).get().method_14482()) {
            modelElement = RPUtils.deserializeJson(modelStream).getAsJsonObject();
        } catch (Exception e) {
            throw new Exception("Failed to parse model at " + modelPath);
        }
        var textures = new ArrayList<>(findAllResourcesInJsonRecursive(modelElement.getAsJsonObject("textures")));
        if (textures.isEmpty()) {
            if (modelElement.has("parent")) {
                var parentPath = modelElement.get("parent").getAsString();
                textures.addAll(findAllTexturesInModelRecursive(manager, parentPath));
            }
        }
        return textures;
    }

    //TODO: account for parents

    public static class_2960 findFirstItemTextureLocation(class_3300 manager, class_1792 block) throws FileNotFoundException {
        return findFirstItemTextureLocation(manager, block, t -> true);
    }

    /**
     * Grabs the first texture from a given item
     *
     * @param manager          resource manager
     * @param item             target item
     * @param texturePredicate predicate that will be applied to the texture name
     * @return found texture location
     */
    public static class_2960 findFirstItemTextureLocation(class_3300 manager, class_1792 item, Predicate<String> texturePredicate) throws FileNotFoundException {
        var cached = TextureCache.getCached(item, texturePredicate);
        if (cached != null) return class_2960.method_60654(cached);
        class_2960 itemId = Utils.getID(item);

        Set<String> textures;
        var itemModel = manager.method_14486(ResType.ITEM_MODELS.getPath(itemId));
        try (var stream = itemModel.orElseThrow().method_14482()) {
            JsonElement bsElement = RPUtils.deserializeJson(stream);

            textures = findAllResourcesInJsonRecursive(bsElement.getAsJsonObject().getAsJsonObject("textures"));
        } catch (Exception ignored) {
            //if texture is not there try to guess location. Hack for better end
            textures = guessItemTextureLocation(itemId, item);
        }
        for (var t : textures) {
            TextureCache.add(item, t);
            if (texturePredicate.test(t)) return class_2960.method_60654(t);
        }

        throw new FileNotFoundException("Could not find any texture associated to the given item " + itemId);
    }

    public static String findFirstResourceInJsonRecursive(JsonElement element) throws NoSuchElementException {
        if (element instanceof JsonArray array) {
            return findFirstResourceInJsonRecursive(array.get(0));
        } else if (element instanceof JsonObject) {
            var entries = element.getAsJsonObject().entrySet();
            JsonElement child = entries.stream().findAny().get().getValue();
            return findFirstResourceInJsonRecursive(child);
        } else return element.getAsString();
    }

    public static Set<String> findAllResourcesInJsonRecursive(JsonElement element) {
        return findAllResourcesInJsonRecursive(element, s -> true);
    }

    public static Set<String> findAllResourcesInJsonRecursive(JsonElement element, Predicate<String> filter) {
        if (element instanceof JsonArray array) {
            Set<String> list = new HashSet<>();

            array.forEach(e -> list.addAll(findAllResourcesInJsonRecursive(e, filter)));
            return list;
        } else if (element instanceof JsonObject json) {
            var entries = json.entrySet();

            Set<String> list = new HashSet<>();
            for (var c : entries) {
                if (c.getValue().isJsonPrimitive() && !filter.test(c.getKey())) continue;
                var l = findAllResourcesInJsonRecursive(c.getValue(), filter);

                list.addAll(l);
            }
            return list;
        } else {
            return Set.of(element.getAsString());
        }
    }
    //recipe stuff

    public static class_1860<?> readRecipe(class_3300 manager, String location) {
        return readRecipeAbsolute(manager, ResType.RECIPES.getPath(location));
    }

    // path is relative to recipe folder
    public static class_1860<?> readRecipe(class_3300 manager, class_2960 location) {
        return readRecipeAbsolute(manager, ResType.RECIPES.getPath(location));
    }

    private static class_1860<?> readRecipeAbsolute(class_3300 manager, class_2960 location) {
        var resource = manager.method_14486(location);
        try (var stream = resource.orElseThrow().method_14482()) {
            JsonObject element = RPUtils.deserializeJson(stream);
            return readRecipe(element);
        } catch (Exception e) {
            throw new InvalidOpenTypeException(String.format("Failed to get recipe at %s: %s", location, e));
        }
    }

    public static class_1860<?> readRecipe(JsonElement element) {
        return class_1860.field_47319.parse(JsonOps.INSTANCE, element).getOrThrow();
    }

    public static <T extends class_1860<?>> JsonElement writeRecipe(T recipe) {
        return class_1860.field_47319.encodeStart(JsonOps.INSTANCE, recipe).getOrThrow();
    }

    @Deprecated(forRemoval = true)
    public static <T extends BlockType> class_8786<?> makeSimilarRecipe(class_1860<?> original, T originalMat, T destinationMat, String baseID) {
        return makeSimilarRecipe(original, originalMat, destinationMat, class_2960.method_60654(baseID));
    }

    /**
     * Use @link ResourceSink#addBlockTypeSwapRecipe
     */
    @Deprecated(forRemoval = true)
    public static <T extends BlockType> class_8786<?> makeSimilarRecipe(class_1860<?> original, T originalMat, T destinationMat, class_2960 baseID) {
        return RecipeTemplate.makeSimilarRecipe(original, originalMat, destinationMat, baseID);
    }

    public static Path getResourcePath(Path path, class_2960 k, class_3264 packType) {
        return path.resolve(packType.method_14413())
                .resolve(k.method_12836())
                .resolve(k.method_12832());
    }

    public static void writeResource(class_2960 id, byte[] bytes, Path path, class_3264 packType) {
        Path p = getResourcePath(path, id, packType);
        try {
            Files.createDirectories(p.getParent());
            Files.write(p, bytes);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @FunctionalInterface
    public interface OverrideAppender {
        void add(class_799 override);

    }

    @Deprecated(forRemoval = true)
    public static void appendModelOverride(class_3300 manager, DynamicTexturePack pack,
                                           class_2960 modelRes, Consumer<OverrideAppender> modelConsumer) {
        var o = manager.method_14486(ResType.ITEM_MODELS.getPath(modelRes));
        if (o.isPresent()) {
            try (var model = o.get().method_14482()) {
                var json = RPUtils.deserializeJson(model);
                JsonArray overrides;
                if (json.has("overrides")) {
                    overrides = json.getAsJsonArray("overrides");
                    ;
                } else overrides = new JsonArray();

                modelConsumer.accept(ov -> overrides.add(serializeModelOverride(ov)));

                json.add("overrides", overrides);
                pack.addItemModel(modelRes, json);
            } catch (Exception ignored) {
            }
        }
    }

    /**
     * Utility method to add models overrides in a non-destructive way. Provided overrides will be added on top of whatever model is currently provided by vanilla or mod resources. IE: crossbows
     */
    @Deprecated(forRemoval = true)
    public static void appendModelOverride(class_3300 manager, ResourceSink pack,
                                           class_2960 modelRes, Consumer<OverrideAppender> modelConsumer) {
        var json = makeModelOverride(manager, modelRes, modelConsumer);
        pack.addItemModel(modelRes, json);
    }

    public static JsonElement makeModelOverride(class_3300 manager,
                                                class_2960 modelRes, Consumer<OverrideAppender> modelConsumer) {
        try (var model = manager.getResourceOrThrow(ResType.ITEM_MODELS.getPath(modelRes)).method_14482()) {
            var json = RPUtils.deserializeJson(model);
            JsonArray overrides;
            if (json.has("overrides")) {
                overrides = json.getAsJsonArray("overrides");
            } else overrides = new JsonArray();

            modelConsumer.accept(ov -> overrides.add(serializeModelOverride(ov)));

            json.add("overrides", overrides);
            return json;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    private static JsonObject serializeModelOverride(class_799 override) {
        JsonObject json = new JsonObject();
        json.addProperty("model", override.method_3472().toString());
        JsonObject predicates = new JsonObject();
        override.method_33690().forEach(p -> {
            predicates.addProperty(p.method_33692().toString(), p.method_33693());
        });
        json.add("predicate", predicates);
        return json;
    }


}
