package net.mehvahdjukaar.moonlight.api.resources.pack;

import com.google.common.base.Stopwatch;
import net.mehvahdjukaar.moonlight.api.misc.IProgressTracker;
import net.mehvahdjukaar.moonlight.api.misc.IProgressTracker.Task;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.resources.assets.LangBuilder;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3262;
import net.minecraft.class_3264;
import net.minecraft.class_3288;
import net.minecraft.class_3300;
import net.minecraft.class_5352;
import net.minecraft.class_9224;
import net.minecraft.class_9225;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;

public abstract class DynamicResourcesProvider implements SimplePackProvider {

    private static final ExecutorService EXECUTOR_SERVICE = Executors.newCachedThreadPool();

    private final class_2960 name;
    private final class_9224 locationInfo;
    private final class_3264 packType;

    protected final IEditablePackResources packResources;
    protected final PackGenerationStrategy generationStrategy;

    private volatile boolean needsRegeneration = true;

    public DynamicResourcesProvider(class_2960 name, class_3264 packType, PackGenerationStrategy generationPolicy) {
        this.name = name;
        this.packType = packType;
        this.generationStrategy = generationPolicy;
        //TODO:maybe make factory with these?
        //TODO:make these configurable
        this.locationInfo = new class_9224(
                name.toString(),    // id
                class_2561.method_43471(LangBuilder.getReadableName(name.toString())), // title
                class_5352.field_25348,
                Optional.empty() //no clue what this is
        );


        this.packResources = generationPolicy.createPackResources(locationInfo, packType);
        this.packResources.addNamespaces(gatherSupportedNamespaces().toArray(new String[0]));
        this.packResources.addNamespaces(name.method_12836());
    }

    public IEditablePackResources getPackResources() {
        return packResources;
    }

    public class_2960 getName() {
        return name;
    }

    public class_9224 getLocationInfo() {
        return locationInfo;
    }

    public class_3264 getPackType() {
        return packType;
    }

    //override if needed
    public class_9225 createSelectionConfig() {
        return new class_9225(
                true,    // required -- this MAY need to be true for the pack to be enabled by default
                class_3288.class_3289.field_14280,
                false // fixed position
        );
    }

    @Override
    public String toString() {
        return "Dynamic " + getPackType() + " Resources Provider [" + name + "]";
    }


    public final void prepare() {
        this.needsRegeneration = needsToRegenerate();
    }

    public boolean needsToRegenerate() {
        if (this.generationStrategy.needsRegeneration(packType)) {
            return this.packResources.clearAllResources();
        } else {
            boolean shouldRegenDueToInvalid = !this.packResources.initializeIfValid();
            if (shouldRegenDueToInvalid) {
                Moonlight.LOGGER.info("Cache for {} at {} is invalid or absent, will regenerate", this, this.packResources);
            }
            return shouldRegenDueToInvalid;
        }
    }

    //called on every reload
    public void reload(class_3300 manager, IProgressTracker reporter) {
        if (this.needsRegeneration) {
            this.needsRegeneration = false;
            try {
                Moonlight.LOGGER.info("Regenerating {}, requested by strategy {}", this, generationStrategy);

                Stopwatch watch = Stopwatch.createStarted();

                runGenerationPipeline(manager, reporter);

                Moonlight.LOGGER.info("Generated runtime {} for pack {} in {}",
                        this.getPackType(), this.packResources.method_14409(), watch);

            } catch (Exception e) {
                Moonlight.LOGGER.error("An error occurred while trying to generate dynamic assets for {}", this, e);
            } finally {
                this.packResources.commitChanges();

                //ugly here but whatever
                if (this.generateDebugResources() && this.packResources instanceof IDebugDumpable d) {
                    getExecutorService().execute(() -> {
                        d.dumpToDisk(Paths.get("debug", "generated_resource_pack"));
                    });
                }
            }
        } else {
            Moonlight.LOGGER.info("Skipping regeneration for {} (cache up-to-date)", this);
        }
    }


    /**
     * just deprecated as it shouldn't be overwritten anymore and will become final private
     */
    private void runGenerationPipeline(class_3300 manager, IProgressTracker progressTracker) {
        List<ResourceGenTask> genTasks = new ArrayList<>();

        try {
            regenerateDynamicAssets(genTasks::add);
        } catch (Exception e) {
            Moonlight.LOGGER.error("Failed to add tasks to dynamic resource gen: ", e);
        }

        int totalTasks = genTasks.size();
        var reporter = progressTracker.subtask(totalTasks);

        List<CompletableFuture<ResourceSink>> futures = genTasks.stream()
                .map(task -> CompletableFuture.supplyAsync(() -> {
                            ResourceSink sink = new ResourceSink(this.name.method_12836(), this.packResources.method_14409());
                            task.accept(manager, sink); // may throw
                            return sink;
                        }, getExecutorService()
                ).handle((sink, ex) -> {
                    reporter.step();
                    if (ex != null) {
                        Moonlight.LOGGER.error("Resource Gen Task failed", ex);
                        return null;
                    }
                    return sink;
                }))
                .toList();

        List<ResourceSink> successful = futures.stream()
                .map(CompletableFuture::join)   // safe: handle() guarantees normal completion
                .filter(Objects::nonNull)//ignore failed tasks
                .toList();

        if (successful.isEmpty()) {
            Moonlight.LOGGER.warn("No resource sinks produced; all tasks failed or none were scheduled.");
            return;
        }

        try {
            ResourceSink.acceptSinks(this.packResources, successful);
        } catch (Exception e) {
            Moonlight.LOGGER.error("Failed to accept generated resource sinks", e);
        }
    }


    protected Executor getExecutorService() {
        return EXECUTOR_SERVICE;
    }


    protected boolean generateDebugResources() {
        return PlatHelper.isDev();
    }

    protected abstract Collection<String> gatherSupportedNamespaces();

    public void addSupportedNamespaces(String... namespace) {
        this.packResources.addNamespaces(namespace);
    }

    protected abstract void regenerateDynamicAssets(Consumer<ResourceGenTask> executor);


    @Override
    public class_3288 createPack() {
        IEditablePackResources resources = this.packResources;
        return class_3288.method_45275(
                this.getLocationInfo(),
                new class_3288.class_7680() {
                    @Override
                    public class_3262 method_52424(class_9224 location) {
                        return resources;
                    }

                    @Override
                    public class_3262 method_52425(class_9224 location, class_3288.class_7679 metadata) {
                        return resources;
                    }
                },
                this.getPackType(),
                this.createSelectionConfig()
        );
    }
}
