package net.mehvahdjukaar.moonlight.api.resources.pack;

import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.util.math.MthUtils;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.mehvahdjukaar.moonlight.core.pack.DynamicResourcesInternals;
import net.minecraft.class_3262;
import net.minecraft.class_3264;
import net.minecraft.class_3272;
import net.minecraft.class_9224;
import org.jetbrains.annotations.NotNull;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.*;

public class GlobalCachedStrategy implements PackGenerationStrategy {

    private static final Map<class_3264, Boolean> NEEDS_REGEN = new HashMap<>();

    private static final Map<class_3264, String> LAST_KNOWN_HASH = new HashMap<>();

    public static void refreshState(class_3264 packType, Collection<class_3262> loadedPacks) {
        String oldHash = readFingerprint(packType);
        String newHash = computeCurrentFingerprint(loadedPacks);
        boolean shouldRegen = !oldHash.equals(newHash);
        Moonlight.LOGGER.info("Resource cache state for {}: {}", packType, shouldRegen ? "needs regeneration" : "up to date");
        NEEDS_REGEN.put(packType, shouldRegen);
        LAST_KNOWN_HASH.put(packType, newHash);

    }

    public static void writeNewState(class_3264 packType) {
        String newHash = LAST_KNOWN_HASH.get(packType);
        if (newHash != null) {
            writeFingerprint(packType, newHash);
            NEEDS_REGEN.put(packType, false);
        }
    }

    private static void writeFingerprint(class_3264 packType, String fp) {
        Path dir = getCachePath(packType);
        Path file = getCacheHashPath(packType);
        try {
            Files.createDirectories(dir);
            Files.writeString(file, fp, StandardCharsets.UTF_8,
                    StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        } catch (Exception e) {
            Moonlight.LOGGER.debug("Failed writing cache fingerprint for {}: {}", packType, e.toString());
        }
    }

    private static String readFingerprint(class_3264 packType) {
        Path file = getCacheHashPath(packType);
        if (!Files.exists(file)) return "";
        try {
            return Files.readString(file, StandardCharsets.UTF_8).trim();
        } catch (Exception e) {
            Moonlight.LOGGER.debug("Failed reading cache fingerprint for {}: {}", packType, e.toString());
            return "";
        }
    }

    private static Path getCacheHashPath(class_3264 packType) {
        return getCachePath(packType).resolve("hash.txt");
    }

    private static Path getCachePath(class_3264 type) {
        return PlatHelper.getGamePath().resolve("dynamic-" +
                (type == class_3264.field_14188 ? "resource" : "data")
                + "-pack-cache");
    }

    private static String computeCurrentFingerprint(Collection<class_3262> packs) {
        return MthUtils.sha256Digest(computeTokens(packs));
    }

    private static @NotNull List<String> computeTokens(Collection<class_3262> packs) {
        List<String> tokens = new ArrayList<>();
        boolean fabric = PlatHelper.getPlatform().isFabric();

        // 1) Packs: keep the given order (order-sensitive)
        int i = 0;
        for (class_3262 p : packs) {
            String id = p.method_14409();
            if (DynamicResourcesInternals.isKnownDynamicPack(p.method_56926().comp_2329())) continue;
            if (id.startsWith("mod/")) continue;
            if (fabric && id.startsWith("fabric")) continue;
            if (id.startsWith("generated")) continue;
            String description = "";
            try {
                class_3272 metadataSection = p.method_14407(class_3272.field_14202);
                if (metadataSection != null) {
                    description = metadataSection.comp_1580().getString();
                }
            } catch (Exception ignored) {
            }
            tokens.add("pack[" + (i++) + "]=" + id + "@" + description);
        }
        // 2) Mods: order-independent (sort deterministically)
        List<String> modTokens = new ArrayList<>();
        for (String mod : PlatHelper.getInstalledMods()) {
            if (fabric && mod.startsWith("fabric")) continue;
            modTokens.add(mod + "@" + PlatHelper.getModVersion(mod));
        }
        Collections.sort(modTokens); // normalize any iteration order
        tokens.addAll(modTokens);

        return tokens;
    }

    @Override
    public boolean needsRegeneration(class_3264 packType) {
        return NEEDS_REGEN.get(packType);
    }

    protected Path getPath(class_3264 type) {
        return GlobalCachedStrategy.getCachePath(type);
    }

    @Override
    public IEditablePackResources createPackResources(class_9224 info, class_3264 type) {
        //this editable pack resources will save sutf to file whenver its added to it
        return new CachePathPackResources(info, type, getPath(type)
                .resolve(info.comp_2329().replace(":", "-")));
    }

    @Override
    public String toString() {
        return "CACHED";
    }
}
