package net.mehvahdjukaar.moonlight.api.resources.textures;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableList.Builder;
import com.google.common.collect.Lists;
import net.mehvahdjukaar.moonlight.api.resources.ResType;
import net.mehvahdjukaar.moonlight.api.util.math.colors.RGBColor;
import net.mehvahdjukaar.moonlight.core.misc.McMetaFile;
import net.minecraft.class_1011;
import net.minecraft.class_1079;
import net.minecraft.class_1080;
import net.minecraft.class_2470;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3532;
import net.minecraft.class_7771;
import org.apache.logging.log4j.util.TriConsumer;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

//like a native image that also has its metadata
public class TextureImage implements AutoCloseable, Sampler2D {

    /**
     * Opens a texture image from the given resource path. A texture image is composed of a NativeImage and its associated McMeta file
     *
     * @param manager      resource manager
     * @param relativePath relative texture path (does not include /textures)
     */
    public static TextureImage open(class_3300 manager, class_2960 relativePath) throws IOException {
        try {
            if (relativePath.method_12832().endsWith(".png")) {
                relativePath = relativePath.method_45136(relativePath.method_12832().substring(0, relativePath.method_12832().length() - 4));
            }
            class_2960 textureLoc = ResType.TEXTURES.getPath(relativePath);
            class_1011 i = SpriteUtils.readImage(manager, textureLoc);
            //try getting metadata for animated textures
            class_2960 metadataLoc = ResType.MCMETA.getPath(relativePath);
            McMetaFile metadata = null;

            var res = manager.method_14486(metadataLoc);
            if (res.isPresent()) {
                try {
                    metadata = McMetaFile.read(res.get());
                } catch (Exception e) {
                    throw new IOException("Failed to open texture at location " + relativePath + ": failed to read mcmeta file", e);
                }
            }

            TextureImage image = of(i, metadata);
            image.path = relativePath.toString();
            return image;
        } catch (Exception e) {
            throw new IOException("Failed to open texture at location " + relativePath + ": no such file");
        }
    }

    public static TextureImage createNew(int width, int height) {
        return createNew(width, height, (McMetaFile) null);
    }

    public static TextureImage createNew(int width, int height, @Nullable McMetaFile metadata) {
        if (width <= 0 || height <= 0) {
            throw new IllegalArgumentException("Width and height must be positive integers");
        }
        var v = new TextureImage(new class_1011(width, height, false), metadata);
        v.clear();
        return v;
    }

    public static TextureImage of(class_1011 image) {
        return of(image, (McMetaFile) null);
    }

    public static TextureImage of(class_1011 image, @Nullable McMetaFile metadata) {
        return new TextureImage(image, metadata);
    }


    @Nullable
    private final McMetaFile metadata;
    private final class_1011 image;
    //width of a frame
    private final class_7771 frameSize;
    //All frames. Includes unused ones
    private final int frameCount;
    private final int frameScale;

    String path = "none";

    private TextureImage(class_1011 image, @Nullable McMetaFile metadata) {
        this.image = image;
        this.metadata = metadata;
        int imgWidth = this.imageWidth(); // 16
        int imgHeight = this.imageHeight(); // 48
        if (metadata == null || metadata.hasEmptyAnimation()) {
            this.frameSize = new class_7771(imgWidth, imgHeight);
        } else {
            this.frameSize = metadata.animation().method_24143(imgWidth, imgHeight);
        }
        this.frameScale = imgWidth / frameSize.comp_1049(); // 1
        int frameScaleHeight = imgHeight / frameSize.comp_1050(); // 2
        this.frameCount = frameScale * frameScaleHeight; // 2
    }

    public int imageWidth() {
        return this.image.method_4307();
    }

    public int imageHeight() {
        return this.image.method_4323();
    }

    public int frameCount() {
        return frameCount;
    }

    public int frameWidth() {
        return frameSize.comp_1049();
    }

    public int frameHeight() {
        return frameSize.comp_1050();
    }

    public McMetaFile getMcMeta() {
        return metadata;
    }

    @ApiStatus.Internal
    public class_1011 getImage() {
        return image;
    }

    //local frame coord from global
    public int getFrameStartX(int frameIndex) {
        return (frameIndex % frameScale) * frameWidth(); //(2 % 1) * 16
    }

    public int getFrameStartY(int frameIndex) {
        return (frameIndex / frameScale) * frameHeight(); // (2/1) * 32
    }

    public int getFramePixel(int frameIndex, int x, int y) {
        return image.method_4315(getFrameStartX(frameIndex) + x, getFrameStartY(frameIndex) + y);
    }

    public int getPixel(int x, int y) {
        return image.method_4315(x, y);
    }

    @Override
    public int sample(float x, float y) {
        int iy = class_3532.method_15340(Math.round(x), 0, imageHeight() - 1);
        int ix = class_3532.method_15340(Math.round(y), 0, imageWidth() - 1);
        return getPixel(iy, ix);
    }

    public Sampler2D frameSampler(int frameIndex) {
        return (x, y) -> {
            int ix = class_3532.method_15340(Math.round(x), 0, frameWidth() - 1);
            int iy = class_3532.method_15340(Math.round(y), 0, frameHeight() - 1);
            return getFramePixel(frameIndex, ix, iy);
        };
    }

    public void setFramePixel(int frameIndex, int x, int y, int color) {
        image.method_4305(getFrameStartX(frameIndex) + x, getFrameStartY(frameIndex) + y, color);
    }

    public void setPixel(int x, int y, int color) {
        image.method_4305(x, y, color);
    }

    public void blendPixel(int x, int y, int color) {
        image.method_35624(x, y, color);
    }

    public void blendFramePixel(int frameIndex, int x, int y, int color) {
        image.method_35624(getFrameStartX(frameIndex) + x, getFrameStartY(frameIndex) + y, color);
    }

    public void forEachPixel(Consumer<PixelContext> consumer) {
        PixelContext pixel = new PixelContext(this);
        for (int frameIdx = 0; frameIdx < frameCount; frameIdx++) {
            int xOff = getFrameStartX(frameIdx);
            int yOff = getFrameStartY(frameIdx);
            for (int x = 0; x < frameWidth(); x++) {
                for (int y = 0; y < frameHeight(); y++) {
                    pixel.frameIndex = frameIdx;
                    pixel.localX = x;
                    pixel.localY = y;
                    pixel.globalX = x + xOff;
                    pixel.globalY = y + yOff;
                    consumer.accept(pixel);
                }
            }
        }
    }

    public TextureImage makeCopy() {
        return makeCopyWithMetadata(this.metadata);
    }

    public TextureImage makeCopyWithMetadata(McMetaFile mcMetaFile) {
        class_1011 im = new class_1011(this.imageWidth(), this.imageHeight(), false);
        im.method_4317(image);
        return new TextureImage(im, mcMetaFile);
    }

    @Override
    public void close() {
        this.image.close();
    }


    @FunctionalInterface
    public interface ThrowingRunnable {
        void run() throws Exception;
    }

    // alternative to try with resources
    public void doAndClose(ThrowingRunnable action) {
        try (this) {
            action.run();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void clear() {
        image.method_4326(0, 0, image.method_4307(), image.method_4323(), 0);
    }

    @Deprecated(forRemoval = true)
    public RGBColor getAverageColor() {
        return SpriteUtils.averageColor(this.image);
    }

    public ImmutableList<class_1011> splitFrames() {
        var builder = ImmutableList.<class_1011>builder();
        if (metadata == null) {
            builder.add(image);
            return builder.build();
        }
        int imgWidth = this.imageWidth(); // 16
        int imgHeight = this.imageHeight(); // 48
        var fs = metadata.animation().method_24143(imgWidth, imgHeight);


        int frameScaleWidth = imgWidth / fs.comp_1049(); // 1
        int frameScaleHeight = imgHeight / fs.comp_1050(); // 2
        int maxFrames = frameScaleWidth * frameScaleHeight; // 2

        List<Integer> indexList = Lists.newArrayList();

        metadata.animation().method_33460((index, time) -> indexList.add(index));
        if (indexList.isEmpty()) {
            for (int l = 0; l < maxFrames; ++l) {
                indexList.add(l);
            }
        }

        if (indexList.size() <= 1) {
            builder.add(image);
        } else {
            for (int index : indexList) { // 2, 1

                int xOffset = (index % frameScaleWidth) * frameWidth(); //(2 % 1) * 16
                int yOffset = (index / frameScaleWidth) * frameHeight(); // (2/1) * 32 =

                if (index >= 0 && xOffset + frameWidth() < imgWidth && yOffset + frameHeight() < imgHeight) {
                    class_1011 f = new class_1011(frameWidth(), frameHeight(), false);
                    for (int x = 0; x < frameWidth(); x++) {
                        for (int y = 0; y < frameHeight(); y++) {
                            f.method_4305(x, y, this.image.method_4315(x + xOffset, y + yOffset));
                        }
                    }
                    builder.add(f);
                }
            }
        }
        return builder.build();
    }


    //deprecated stuff

    //texture ops stuff


    @Deprecated
    public void toGrayscale() {
        TextureOps.grayscale(this);
    }


    @Deprecated(forRemoval = true)
    public TextureImage createAnimationTemplate(int length, McMetaFile useDataFrom) {
        return TextureOps.createSingleFrameAnimation(this, useDataFrom);
    }


    @Deprecated(forRemoval = true)
    public void applyOverlay(TextureImage... overlays) throws IllegalStateException {
        TextureOps.applyOverlay(this, overlays);
        Arrays.stream(overlays).forEach(TextureImage::close);
    }

    @Deprecated(forRemoval = true)
    public void applyOverlayOnExisting(TextureImage... overlays) throws IllegalStateException {
        TextureOps.applyOverlayOnExisting(this, overlays);
        Arrays.stream(overlays).forEach(TextureImage::close);
    }


    @Deprecated(forRemoval = true)
    public void removeAlpha(int backgroundColor) {
        TextureOps.makeOpaque(this, backgroundColor);
    }

    @Deprecated(forRemoval = true)
    public TextureImage createRotated(class_2470 rotation) {
        return TextureOps.createRotated(this, rotation);
    }

    @Deprecated(forRemoval = true)
    public TextureImage createResized(float widthScale, float heightScale) {
        return TextureOps.createScaled(this, widthScale, heightScale);
    }

    @Deprecated(forRemoval = true)
    public void crop(TextureImage mask) {
        crop(mask, true);
    }

    @Deprecated(forRemoval = true)
    public void crop(TextureImage mask, boolean discardInner) {
        if (discardInner) TextureOps.applyMask(this, mask);
        else TextureOps.applyMaskInverted(this, mask);
        mask.close();
    }


    //old stuff

    @Deprecated(forRemoval = true)
    public TextureImage createAnimationTemplate(int length, @NotNull class_1079 useDataFrom) {
        return createAnimationTemplate(length, McMetaFile.of(useDataFrom));
    }

    @Deprecated(forRemoval = true)
    public TextureImage createAnimationTemplate(int length, List<class_1080> frameData, int frameTime, boolean interpolate) {
        return createAnimationTemplate(length, new class_1079(frameData, this.frameWidth(), this.frameHeight(), frameTime, interpolate));
    }

    @Deprecated(forRemoval = true)
    public void forEachFrame(FramePixelConsumer e) {
        forEachFramePixel(e);
    }

    /**
     * Accepts a consumer that iterates over all image pixels, ordered by frame.
     * The given coordinates are global texture coordinates while the index represents the currently viewed frame
     */
    @Deprecated(forRemoval = true)
    public void forEachFramePixel(FramePixelConsumer framePixelConsumer) {
        for (int ind = 0; ind < frameCount; ind++) {
            int xOff = getFrameStartX(ind);
            int yOff = getFrameStartY(ind);
            for (int x = 0; x < frameWidth(); x++) {
                for (int y = 0; y < frameHeight(); y++) {
                    framePixelConsumer.accept(ind, x + xOff, y + yOff);
                }
            }
        }
    }


    @Deprecated(forRemoval = true)
    @Nullable
    public class_1079 getMetadata() {
        return metadata == null ? null : metadata.animation();
    }


    @Deprecated(forRemoval = true)
    public static TextureImage createNew(int width, int height, @Nullable class_1079 animation) {
        return createNew(width, height, animation == null ? null : McMetaFile.of(animation));
    }

    @Deprecated(forRemoval = true)
    public static TextureImage of(class_1011 image, @Nullable class_1079 animation) {
        return of(image, animation == null ? null : McMetaFile.of(animation));
    }

    //ind, x, y
    @Deprecated(forRemoval = true)
    @FunctionalInterface
    public interface FramePixelConsumer extends TriConsumer<Integer, Integer, Integer> {
        @Override
        void accept(Integer frameIndex, Integer globalX, Integer globalY);
    }

}