package net.mehvahdjukaar.moonlight.api.trades;

import ;
import I;
import com.google.common.base.Preconditions;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.DynamicOps;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.MapCodec;
import it.unimi.dsi.fastutil.ints.Int2ObjectArrayMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import net.mehvahdjukaar.moonlight.api.misc.CodecMapRegistry;
import net.mehvahdjukaar.moonlight.api.misc.MapRegistry;
import net.mehvahdjukaar.moonlight.api.misc.SidedInstance;
import net.mehvahdjukaar.moonlight.api.platform.ForgeHelper;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1914;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_3852;
import net.minecraft.class_3853;
import net.minecraft.class_4309;
import net.minecraft.class_5819;
import net.minecraft.class_7225;
import net.minecraft.class_7923;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.Map.Entry;

public class ItemListingManager extends class_4309 {

    private static final SidedInstance<ItemListingManager> INSTANCE = SidedInstance.of(ItemListingManager::new);
    protected static final CodecMapRegistry<ModItemListing> LISTING_TYPES = MapRegistry.ofCodec();

    @ApiStatus.Internal
    public static void init() {
        LISTING_TYPES.register(class_2960.method_60654("simple"), SimpleItemListing.CODEC);
        LISTING_TYPES.register(class_2960.method_60654("remove_all_non_data"), RemoveNonDataListingListing.CODEC);
        LISTING_TYPES.register(class_2960.method_60654("no_op"), NoOpListing.CODEC);
        LISTING_TYPES.register(class_2960.method_60654("villager_type_variant"), BiomeVariantItemListing.CODEC);
    }

    private final Map<class_1299<?>, Set<ModItemListing>> specialTradesAdded = new HashMap<>();
    private final Map<class_3852, Set<ModItemListing>> tradesAdded = new HashMap<>();

    //removed trades
    private final Map<class_1299<?>, Int2ObjectArrayMap<Set<class_3853.class_1652>>> specialTradesRemoved = new HashMap<>();
    private final Map<class_3852, Int2ObjectArrayMap<Set<class_3853.class_1652>>> tradesRemoved = new HashMap<>();


    private final class_7225.class_7874 registryAccess;

    public ItemListingManager(class_7225.class_7874 provider) {
        super(new Gson(), "moonlight/villager_trade");
        this.registryAccess = provider;

        INSTANCE.set(registryAccess, this);
    }

    @Override
    protected void apply(Map<class_2960, JsonElement> jsons, class_3300 resourceManager, class_3695 profiler) {

        //restore
        restoreVanillaState();


        List<Pair<ModItemListing, class_3852>> toAdd = new ArrayList<>();
        List<Pair<ModItemListing, class_1299<?>>> toAddSpecial = new ArrayList<>();
        List<Pair<RemoveNonDataListingListing, class_3852>> toRemove = new ArrayList<>();
        List<Pair<RemoveNonDataListingListing, class_1299<?>>> toRemoveSpecial = new ArrayList<>();

        DynamicOps<JsonElement> ops = ForgeHelper.conditionalOps(JsonOps.INSTANCE, registryAccess, this);
        for (var e : jsons.entrySet()) {
            JsonElement json = e.getValue();
            class_2960 id = e.getKey();
            if (!id.method_12832().contains("/")) continue;
            class_2960 targetId = id.method_45134(p -> p.substring(0, p.lastIndexOf('/')));
            var profession = class_7923.field_41195.method_17966(targetId);
            if (profession.isPresent()) {
                ModItemListing trade = parseOrThrow(json, id, ops).orElse(null);
                if (trade == null || (trade instanceof NoOpListing)) {
                    continue;
                } else if (trade instanceof RemoveNonDataListingListing rl) {
                    toRemove.add(Pair.of(rl, profession.get()));
                } else {
                    toAdd.add(Pair.of(trade, profession.get()));
                }
                continue;
            }
            var entityType = class_7923.field_41177.method_17966(targetId);
            if (entityType.isPresent()) {
                ModItemListing trade = parseOrThrow(json, id, ops).orElse(null);
                if (trade == null || (trade instanceof NoOpListing)) {
                    continue;
                } else if (trade instanceof RemoveNonDataListingListing rl) {
                    toRemoveSpecial.add(Pair.of(rl, entityType.get()));
                } else {
                    toAddSpecial.add(Pair.of(trade, entityType.get()));
                }
            } else {
                Moonlight.LOGGER.warn("Unknown villager type: {}", targetId);
            }
        }


        // Apply removals for profession-based trades
        for (var pair : toRemove) {
            class_3852 profession = pair.getSecond();
            RemoveNonDataListingListing listing = pair.getFirst();
            Int2ObjectMap<class_3853.class_1652[]> tradeMap = getTradeMapForProfession(profession);
            var removed = removeMatchingTrades(listing, tradeMap);
            if (!removed.isEmpty()) {
                tradesRemoved.computeIfAbsent(profession, k -> new Int2ObjectArrayMap<>())
                        .putAll(removed);
            }
        }

        // Apply removals for entity-based trades
        for (var pair : toRemoveSpecial) {
            class_1299<?> entity = pair.getSecond();
            if (entity == class_1299.field_17713) {
                RemoveNonDataListingListing listing = pair.getFirst();
                Int2ObjectMap<class_3853.class_1652[]> wanderingTraderTrades = class_3853.field_17724;
                var removed = removeMatchingTrades(listing, wanderingTraderTrades);
                if (!removed.isEmpty()) {
                    specialTradesRemoved.computeIfAbsent(entity, k -> new Int2ObjectArrayMap<>())
                            .putAll(removed);
                }
            }
        }


        // Apply profession-based additions
        for (var pair : toAdd) {
            ModItemListing listing = pair.getFirst();
            class_3852 profession = pair.getSecond();
            Int2ObjectMap<class_3853.class_1652[]> tradeMap = getTradeMapForProfession(profession);
                addTrade(tradeMap, listing, true);
                tradesAdded.computeIfAbsent(profession, k -> new HashSet<>()).add(listing);
        }

        // Apply entity-based additions
        for (var pair : toAddSpecial) {
            ModItemListing listing = pair.getFirst();
            class_1299<?> entity = pair.getSecond();
            if (entity == class_1299.field_17713) {
                Int2ObjectMap<class_3853.class_1652[]> wanderingTraderTrades = class_3853.field_17724;
                addTrade(wanderingTraderTrades, listing, true);
            }
            specialTradesAdded.computeIfAbsent(entity, k -> new HashSet<>()).add(listing);
        }

        int added = specialTradesAdded.values().stream()
                .mapToInt(Set::size)
                .sum() + tradesAdded.values().stream()
                .mapToInt(Set::size)
                .sum();
        int removed = tradesRemoved.values().stream().mapToInt(map -> map.values().stream().mapToInt(Set::size).sum()).sum()
                + specialTradesRemoved.values().stream().mapToInt(map -> map.values().stream().mapToInt(Set::size).sum()).sum();

        if (added > 0) {
            Moonlight.LOGGER.info("Applied {} data villager trades", added);
        }
        if (removed > 0) {
            Moonlight.LOGGER.info("Removed {} data villager trades", removed);
        }
    }

    private static @NotNull Int2ObjectMap<class_3853.class_1652[]> getTradeMapForProfession(
            class_3852 profession) {
        return class_3853.field_17067.computeIfAbsent(profession, k -> new Int2ObjectArrayMap<>());
    }

    private static void addTrade(Int2ObjectMap<class_3853.class_1652[]> tradeMap, @NotNull ModItemListing listing, boolean add) {
        var level = listing.getLevel();
        // Ensure an array exists for this level, or create a new empty array if absent
        var existing = tradeMap.computeIfAbsent(level, k -> new class_3853.class_1652[0]);
        tradeMap.put(listing.getLevel(), mergeArrays(existing, add, listing));
    }

    private static class_3853.class_1652[] mergeArrays(class_3853.class_1652[] existing, boolean add,
                                                            class_3853.class_1652... toAdd) {
        var list = new ArrayList<>(List.of(existing));
        if (add) list.addAll(List.of(toAdd));
        else list.removeAll(List.of(toAdd));
        return list.toArray(class_3853.class_1652[]::new);
    }

    private Int2ObjectArrayMap<Set<class_3853.class_1652>> removeMatchingTrades(
            RemoveNonDataListingListing removal,
            Int2ObjectMap<class_3853.class_1652[]> originalTrades
    ) {
        Int2ObjectArrayMap<Set<class_3853.class_1652>> removedTrades = new Int2ObjectArrayMap<>();

        // Temporary map to hold updated trade arrays after removals
        Map<Integer, class_3853.class_1652[]> updatedTrades = new HashMap<>();

        for (var entry : originalTrades.int2ObjectEntrySet()) {
            int level = entry.getIntKey();
            class_3853.class_1652[] trades = entry.getValue();

            List<class_3853.class_1652> remaining = new ArrayList<>();
            Set<class_3853.class_1652> removedAtLevel = new HashSet<>();

            for (class_3853.class_1652 trade : trades) {
                if (removal.matches(level, trade)) {
                    removedAtLevel.add(trade);
                } else {
                    remaining.add(trade);
                }
            }

            if (!removedAtLevel.isEmpty()) {
                removedTrades.put(level, removedAtLevel);
                updatedTrades.put(level, remaining.toArray(class_3853.class_1652[]::new));
            }
        }

        // Apply updates after iteration
        originalTrades.putAll(updatedTrades);

        return removedTrades;
    }

    private void restoreVanillaState() {
        // Undo added profession-based trades
        for (var entry : tradesAdded.entrySet()) {
            class_3852 profession = entry.getKey();
            Set<ModItemListing> listings = entry.getValue();
            Int2ObjectMap<class_3853.class_1652[]> tradeMap = getTradeMapForProfession(profession);

            for (ModItemListing listing : listings) {
                int level = listing.getLevel();
                class_3853.class_1652[] array = tradeMap.get(level);
                if (array == null) continue;
                addTrade(tradeMap, listing, false);
            }
        }

        // Undo added special/entity-based trades
        for (var entry : specialTradesAdded.entrySet()) {
            class_1299<?> entity = entry.getKey();
            Set<ModItemListing> listings = entry.getValue();

            if (entity == class_1299.field_17713) {
                Int2ObjectMap<class_3853.class_1652[]> tradeMap = class_3853.field_17724;

                for (ModItemListing listing : listings) {
                    int level = listing.getLevel();
                    class_3853.class_1652[] array = tradeMap.get(level);
                    if (array == null) continue;
                    addTrade(tradeMap, listing, false);
                }
            }
        }

        // Restore removed profession-based trades
        for (var entry : tradesRemoved.entrySet()) {
            class_3852 profession = entry.getKey();
            Int2ObjectMap<Set<class_3853.class_1652>> removedPerLevel = entry.getValue();
            Int2ObjectMap<class_3853.class_1652[]> tradeMap = getTradeMapForProfession(profession);

            restoreMap(tradeMap, removedPerLevel);
        }

        // Restore removed special/entity-based trades
        for (var entry : specialTradesRemoved.entrySet()) {
            class_1299<?> entity = entry.getKey();
            if (entity == class_1299.field_17713) {
                Int2ObjectMap<class_3853.class_1652[]> tradeMap = class_3853.field_17724;
                Int2ObjectMap<Set<class_3853.class_1652>> removedPerLevel = entry.getValue();

                restoreMap(tradeMap, removedPerLevel);
            }
        }

        tradesAdded.clear();
        specialTradesAdded.clear();
        tradesRemoved.clear();
        specialTradesRemoved.clear();
    }

    private void restoreMap(Int2ObjectMap<class_3853.class_1652[]> tradeMap,
                            Int2ObjectMap<Set<class_3853.class_1652>> removedPerLevel) {
        for (var levelEntry : removedPerLevel.int2ObjectEntrySet()) {
            int level = levelEntry.getIntKey();
            Set<class_3853.class_1652> removedTrades = levelEntry.getValue();
            class_3853.class_1652[] currentArray = tradeMap.get(level);
            tradeMap.put(level, mergeArrays(currentArray, true, removedTrades.toArray(class_3853.class_1652[]::new)));
        }
    }


    private static Optional<ModItemListing> parseOrThrow(JsonElement j, class_2960 id, DynamicOps<JsonElement> ops) {
        return ForgeHelper.conditionalCodec(ModItemListing.CODEC)
                .parse(ops, j)
                .getOrThrow();
    }

    public static List<? extends class_3853.class_1652> getVillagerListings(class_3852 profession, int level) {
        class_3853.class_1652[] array = getTradeMapForProfession(profession).get(level);
        if (array == null) return List.of();
        return Arrays.stream(array).toList();
    }

    public static List<? extends class_3853.class_1652> getSpecialListings(class_1299<?> entityType, int level, class_7225.class_7874 provider) {
        if (entityType == class_1299.field_17713) {
            class_3853.class_1652[] array = class_3853.field_17724.get(level);
            if (array == null) return List.of();
            return Arrays.stream(array).toList();
        } else {
            var special = INSTANCE.get(provider).specialTradesAdded.get(entityType);
            if (special == null) return List.of();
            List<class_3853.class_1652> listings = new ArrayList<>();
            for (ModItemListing listing : special) {
                if (listing.getLevel() == level) {
                    listings.add(listing);
                }
            }
            return listings;
        }
    }

    @Deprecated(forRemoval = true)
    public static List<? extends class_3853.class_1652> getSpecialListings(class_1299<?> entityType, int level) {
        return getSpecialListings(entityType, level, Utils.hackyGetRegistryAccess());
    }

    /**
     * Call on mod setup. Register a new serializer for your trade
     */
    public static void registerSerializer(class_2960 id, MapCodec<? extends ModItemListing> trade) {
        LISTING_TYPES.register(id, trade);
    }

    /**
     * Registers a simple special trade
     */
    public static void registerSimple(class_2960 id, class_3853.class_1652 instance, int level) {
        SpecialListing specialListing = new SpecialListing(instance, level);
        registerSerializer(id, specialListing.getCodec());
    }

    private static class SpecialListing implements ModItemListing {

        private final MapCodec<ModItemListing> codec = MapCodec.unit(this);
        private final class_3853.class_1652 listing;
        private final int level;

        public SpecialListing(class_3853.class_1652 listing, int level) {
            this.listing = listing;
            this.level = level;
        }

        @Override
        public MapCodec<? extends ModItemListing> getCodec() {
            return codec;
        }

        @Nullable
        @Override
        public class_1914 method_7246(class_1297 trader, class_5819 random) {
            return listing.method_7246(trader, random);
        }

        @Override
        public int getLevel() {
            return level;
        }
    }

}
