package net.mehvahdjukaar.moonlight.api.platform.configs.forge;

import ;
import com.electronwill.nightconfig.core.CommentedConfig;
import com.electronwill.nightconfig.core.ConcurrentConfigSpec;
import com.electronwill.nightconfig.core.ConfigFormat;
import com.electronwill.nightconfig.core.file.CommentedFileConfig;
import com.electronwill.nightconfig.core.io.ParsingMode;
import com.electronwill.nightconfig.core.io.WritingMode;
import com.electronwill.nightconfig.toml.TomlFormat;
import net.mehvahdjukaar.moonlight.api.misc.EventCalled;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.platform.configs.ConfigSpec;
import net.mehvahdjukaar.moonlight.api.platform.configs.ConfigType;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;
import net.minecraftforge.client.ConfigScreenHandler;
import net.minecraftforge.common.ForgeConfigSpec;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.entity.player.PlayerEvent;
import net.minecraftforge.eventbus.api.IEventBus;
import net.minecraftforge.fml.ModContainer;
import net.minecraftforge.fml.ModList;
import net.minecraftforge.fml.ModLoadingContext;
import net.minecraftforge.fml.config.ConfigFileTypeHandler;
import net.minecraftforge.fml.config.IConfigEvent;
import net.minecraftforge.fml.config.ModConfig;
import net.minecraftforge.fml.event.config.ModConfigEvent;
import net.minecraftforge.fml.javafmlmod.FMLJavaModLoadingContext;
import net.minecraftforge.fml.loading.FMLPaths;
import net.minecraftforge.fml.util.ObfuscationReflectionHelper;
import org.jetbrains.annotations.Nullable;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

public final class ConfigSpecWrapper extends ConfigSpec {

    private static final Method SET_CONFIG_DATA = ObfuscationReflectionHelper.findMethod(ModConfig.class, "setConfigData", CommentedConfig.class);
    private static final Method SETUP_CONFIG_FILE = ObfuscationReflectionHelper.findMethod(ConfigFileTypeHandler.class,
            "setupConfigFile", ModConfig.class, Path.class, ConfigFormat.class);

    private final ForgeConfigSpec spec;

    private final ModConfig modConfig;
    private final ModContainer modContainer;

    private final Map<ForgeConfigSpec.ConfigValue<?>, Object> requireRestartValues;
    private final List<ConfigBuilderImpl.SpecialValue<?,?>> specialValues;

    public ConfigSpecWrapper(ResourceLocation name, ForgeConfigSpec spec, ConfigType type, boolean synced,
                             @Nullable Runnable onChange, List<ForgeConfigSpec.ConfigValue<?>> requireRestart,
                             List<ConfigBuilderImpl.SpecialValue<?,?>> specialValues) {
        super(name.m_135827_(), name.m_135827_() + "-" + name.m_135815_() + ".toml",
                FMLPaths.CONFIGDIR.get(), type, synced, onChange);
        this.spec = spec;
        this.specialValues = specialValues;

        ModConfig.Type t = this.getConfigType() == ConfigType.COMMON ? ModConfig.Type.COMMON : ModConfig.Type.CLIENT;

        this.modContainer = ModLoadingContext.get().getActiveContainer();
        this.modConfig = new ModConfig(t, spec, modContainer, this.getFileName());

        var bus = FMLJavaModLoadingContext.get().getModEventBus();
        if (onChange != null || this.isSynced() || !specialValues.isEmpty()) bus.addListener(this::onConfigChange);
        if (this.isSynced()) {

            MinecraftForge.EVENT_BUS.addListener(this::onPlayerLoggedIn);
            MinecraftForge.EVENT_BUS.addListener(this::onPlayerLoggedOut);
        }
        //for event
        ConfigSpec.addTrackedSpec(this);

        if (!requireRestart.isEmpty()) {
            loadFromFile(); //Early load if this has world reload ones as we need to get their current values. Isn't there a better way?
        }
        this.requireRestartValues = requireRestart.stream().collect(Collectors.toMap(e -> e, ForgeConfigSpec.ConfigValue::get));

    }

    @Override
    public Component getName() {
        return Component.m_237113_(getFileName());
    }

    @Override
    public Path getFullPath() {
        return FMLPaths.CONFIGDIR.get().resolve(this.getFileName());
        // return modConfig.getFullPath();
    }

    @Override
    public void register() {
        ModContainer modContainer = ModLoadingContext.get().getActiveContainer();
        modContainer.addConfig(this.modConfig);
    }

    @Override
    public void loadFromFile() {
        //same stuff that forge config tracker does
        try {
            CommentedFileConfig configData = readConfig(modConfig.getHandler(), FMLPaths.CONFIGDIR.get(), modConfig);
            SET_CONFIG_DATA.setAccessible(true);
            SET_CONFIG_DATA.invoke(modConfig, configData);
            modContainer.dispatchConfigEvent(IConfigEvent.loading(modConfig));
            modConfig.save();
        } catch (Exception e) {
            throw new ConfigLoadingException(modConfig, e);
        }
    }

    //we need this so we don't add a second file watcher. Same as handler::reader
    private CommentedFileConfig readConfig(ConfigFileTypeHandler handler, Path configBasePath, ModConfig c) {
        Path configPath = configBasePath.resolve(c.getFileName());
        CommentedFileConfig configData = CommentedFileConfig.builder(configPath).sync().
                preserveInsertionOrder().
                autosave().
                onFileNotFound((newfile, configFormat) -> {
                    try {
                        return (Boolean) SETUP_CONFIG_FILE.invoke(handler, c, newfile, configFormat);
                    } catch (Exception e) {
                        throw new ConfigLoadingException(c, e);
                    }
                }).
                writingMode(WritingMode.REPLACE).
                build();
        configData.load();
        return configData;
    }

    private static class ConfigLoadingException extends RuntimeException {
        public ConfigLoadingException(ModConfig config, Exception cause) {
            super("Failed loading config file " + config.getFileName() + " of type " + config.getType() + " for modid " + config.getModId() + ". Try deleting it", cause);
        }
    }

    public ForgeConfigSpec getSpec() {
        return spec;
    }

    @Nullable
    public ModConfig getModConfig() {
        return modConfig;
    }

    public ModConfig.Type getModConfigType() {
        return this.getConfigType() == ConfigType.CLIENT ? ModConfig.Type.CLIENT : ModConfig.Type.COMMON;
    }

    @Override
    public boolean isLoaded() {
        return spec.isLoaded();
    }

    @Nullable
    @Override
    @OnlyIn(Dist.CLIENT)
    public Screen makeScreen(Screen parent, @Nullable ResourceLocation background) {
        var container = ModList.get().getModContainerById(this.getModId());
        if (container.isPresent()) {
            var factory = container.get().getCustomExtension(ConfigScreenHandler.ConfigScreenFactory.class);
            if (factory.isPresent()) return factory.get().screenFunction().apply(Minecraft.m_91087_(), parent);
        }
        return null;
    }

    @Override
    public boolean hasConfigScreen() {
        return ModList.get().getModContainerById(this.getModId())
                .map(container -> container.getCustomExtension(ConfigScreenHandler.ConfigScreenFactory.class)
                        .isPresent()).orElse(false);
    }

    @EventCalled
    private void onPlayerLoggedIn(PlayerEvent.PlayerLoggedInEvent event) {
        if (event.getEntity() instanceof ServerPlayer serverPlayer) {
            //send this configuration to connected clients
            syncConfigsToPlayer(serverPlayer);
        }
    }

    @EventCalled
    protected void onPlayerLoggedOut(PlayerEvent.PlayerLoggedOutEvent event) {
        if (event.getEntity().m_9236_().f_46443_) {
            onRefresh();
        }
    }

    @EventCalled
    protected void onConfigChange(ModConfigEvent event) {
        if (event.getConfig().getSpec() == this.getSpec()) {
            //send this configuration to connected clients if on server
            if (this.isSynced() && PlatHelper.getPhysicalSide().isServer()) sendSyncedConfigsToAllPlayers();
            onRefresh();
            specialValues.forEach(ConfigBuilderImpl.SpecialValue::clearCache);
        }
    }

    @Override
    public void loadFromBytes(InputStream stream) {
        try { //this should work the same as below and internaly calls refresh
            var b = stream.readAllBytes();
            this.modConfig.acceptSyncedConfig(b);
        } catch (Exception e) {
            Moonlight.LOGGER.warn("Failed to sync config file {}:", this.getFileName(), e);
        }

        //using this isntead so we dont fire the config changes event otherwise this will loop
        //this.getSpec().setConfig(TomlFormat.instance().createParser().parse(stream));
        //this.onRefresh();
    }


    public static void acceptConfig(ModConfig modConfig, byte[] bytes) {
        /*
        if (modConfig.getConfigData() instanceof ConcurrentConfigSpec cc) {
            cc.bulkCommentedUpdate(view -> {
                TomlFormat.instance().createParser().parse(new ByteArrayInputStream(bytes), view, ParsingMode.REPLACE);
            });
        }*/
    }


    public boolean requiresGameRestart(ForgeConfigSpec.ConfigValue<?> value) {
        var v = requireRestartValues.get(value);
        if (v == null) return false;
        else return v != value.get();
    }


}
