package net.mehvahdjukaar.moonlight.api.resources;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.internal.Streams;
import com.google.gson.stream.JsonWriter;
import net.mehvahdjukaar.moonlight.api.client.TextureCache;
import net.mehvahdjukaar.moonlight.api.resources.pack.DynamicTexturePack;
import net.mehvahdjukaar.moonlight.api.resources.recipe.IRecipeTemplate;
import net.mehvahdjukaar.moonlight.api.resources.recipe.TemplateRecipeManager;
import net.mehvahdjukaar.moonlight.api.set.BlockType;
import net.mehvahdjukaar.moonlight.api.set.wood.WoodType;
import net.mehvahdjukaar.moonlight.api.set.wood.WoodTypeRegistry;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.mehvahdjukaar.moonlight.core.Moonlight;
import net.minecraft.client.renderer.block.model.ItemOverride;
import net.minecraft.core.NonNullList;
import net.minecraft.core.RegistryAccess;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.util.GsonHelper;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.*;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.block.Block;
import org.jetbrains.annotations.NotNull;

import javax.management.openmbean.InvalidOpenTypeException;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.Map.Entry;
import java.util.function.Consumer;
import java.util.function.Predicate;

public class RPUtils {

    public static String serializeJson(JsonElement json) throws IOException {
        try (StringWriter stringWriter = new StringWriter();
             JsonWriter jsonWriter = new JsonWriter(stringWriter)) {

            jsonWriter.setLenient(true);
            jsonWriter.setIndent("  ");

            Streams.write(json, jsonWriter);

            return stringWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    //remember to close this stream
    public static JsonObject deserializeJson(InputStream stream) {
        return GsonHelper.m_13859_(new InputStreamReader(stream, StandardCharsets.UTF_8));
    }

    public static ResourceLocation findFirstBlockTextureLocation(ResourceManager manager, Block block) throws FileNotFoundException {
        return findFirstBlockTextureLocation(manager, block, t -> true);
    }

    /**
     * Grabs the first texture from a given block
     *
     * @param manager          resource manager
     * @param block            target block
     * @param texturePredicate predicate that will be applied to the texture name
     * @return found texture location
     */
    public static ResourceLocation findFirstBlockTextureLocation(ResourceManager manager, Block block, Predicate<String> texturePredicate) throws FileNotFoundException {
        var cached = TextureCache.getCached(block, texturePredicate);
        if (cached != null) {
            return new ResourceLocation(cached);
        }
        ResourceLocation blockId = Utils.getID(block);
        var blockState = manager.m_213713_(ResType.BLOCKSTATES.getPath(blockId));
        try (var bsStream = blockState.orElseThrow().m_215507_()) {

            JsonElement bsElement = RPUtils.deserializeJson(bsStream);
            //grabs the first resource location of a model
            Set<String> models = findAllResourcesInJsonRecursive(bsElement.getAsJsonObject(), s -> s.equals("model"));

            for (var modelPath : models) {
                List<String> textures = findAllTexturesInModelRecursive(manager, modelPath);

                for (var t : textures) {
                    TextureCache.add(block, t);
                    if (texturePredicate.test(t)) return new ResourceLocation(t);
                }
            }
        } catch (Exception ignored) {
        }
        //if texture is not there try to guess location. Hack for better end
        var hack = guessBlockTextureLocation(blockId, block);
        for (var t : hack) {
            TextureCache.add(block, t);
            if (texturePredicate.test(t)) return new ResourceLocation(t);
        }

        throw new FileNotFoundException("Could not find any texture associated to the given block " + blockId);
    }

    //if texture is not there try to guess location. Hack for better end
    private static Set<String> guessItemTextureLocation(ResourceLocation id, Item item) {
        return Set.of(id.m_135827_() + ":item/" + item);
    }

    private static List<String> guessBlockTextureLocation(ResourceLocation id, Block block) {
        String name = id.m_135815_();
        List<String> textures = new ArrayList<>();
        //just works for wood types
        WoodType w = WoodTypeRegistry.INSTANCE.getBlockTypeOf(block);
        if (w != null) {
            String key = w.getChildKey(block);
            if (Objects.equals(key, "log") || Objects.equals(key, "stripped_log")) {
                textures.add(id.m_135827_() + ":block/" + name + "_top");
                textures.add(id.m_135827_() + ":block/" + name + "_side");
            }
        }
        textures.add(id.m_135827_() + ":block/" + name);
        return textures;
    }

    @NotNull
    private static List<String> findAllTexturesInModelRecursive(ResourceManager manager, String modelPath) throws Exception {
        JsonObject modelElement;
        try (var modelStream = manager.m_213713_(ResType.MODELS.getPath(modelPath)).get().m_215507_()) {
            modelElement = RPUtils.deserializeJson(modelStream).getAsJsonObject();
        } catch (Exception e) {
            throw new Exception("Failed to parse model at " + modelPath);
        }
        var textures = new ArrayList<>(findAllResourcesInJsonRecursive(modelElement.getAsJsonObject("textures")));
        if (textures.isEmpty()) {
            if (modelElement.has("parent")) {
                var parentPath = modelElement.get("parent").getAsString();
                textures.addAll(findAllTexturesInModelRecursive(manager, parentPath));
            }
        }
        return textures;
    }

    //TODO: account for parents

    public static ResourceLocation findFirstItemTextureLocation(ResourceManager manager, Item block) throws FileNotFoundException {
        return findFirstItemTextureLocation(manager, block, t -> true);
    }

    /**
     * Grabs the first texture from a given item
     *
     * @param manager          resource manager
     * @param item             target item
     * @param texturePredicate predicate that will be applied to the texture name
     * @return found texture location
     */
    public static ResourceLocation findFirstItemTextureLocation(ResourceManager manager, Item item, Predicate<String> texturePredicate) throws FileNotFoundException {
        var cached = TextureCache.getCached(item, texturePredicate);
        if (cached != null) return new ResourceLocation(cached);
        ResourceLocation itemId = Utils.getID(item);

        Set<String> textures;
        var itemModel = manager.m_213713_(ResType.ITEM_MODELS.getPath(itemId));
        try (var stream = itemModel.orElseThrow().m_215507_()) {
            JsonElement bsElement = RPUtils.deserializeJson(stream);

            textures = findAllResourcesInJsonRecursive(bsElement.getAsJsonObject().getAsJsonObject("textures"));
        } catch (Exception ignored) {
            //if texture is not there try to guess location. Hack for better end
            textures = guessItemTextureLocation(itemId, item);
        }
        for (var t : textures) {
            TextureCache.add(item, t);
            if (texturePredicate.test(t)) return new ResourceLocation(t);
        }

        throw new FileNotFoundException("Could not find any texture associated to the given item " + itemId);
    }

    public static String findFirstResourceInJsonRecursive(JsonElement element) throws NoSuchElementException {
        if (element instanceof JsonArray array) {
            return findFirstResourceInJsonRecursive(array.get(0));
        } else if (element instanceof JsonObject) {
            var entries = element.getAsJsonObject().entrySet();
            JsonElement child = entries.stream().findAny().get().getValue();
            return findFirstResourceInJsonRecursive(child);
        } else return element.getAsString();
    }

    public static Set<String> findAllResourcesInJsonRecursive(JsonElement element) {
        return findAllResourcesInJsonRecursive(element, s -> true);
    }

    public static Set<String> findAllResourcesInJsonRecursive(JsonElement element, Predicate<String> filter) {
        if (element instanceof JsonArray array) {
            Set<String> list = new HashSet<>();

            array.forEach(e -> list.addAll(findAllResourcesInJsonRecursive(e, filter)));
            return list;
        } else if (element instanceof JsonObject json) {
            var entries = json.entrySet();

            Set<String> list = new HashSet<>();
            for (var c : entries) {
                if (c.getValue().isJsonPrimitive() && !filter.test(c.getKey())) continue;
                var l = findAllResourcesInJsonRecursive(c.getValue(), filter);

                list.addAll(l);
            }
            return list;
        } else {
            return Set.of(element.getAsString());
        }
    }
    //recipe stuff

    public static Recipe<?> readRecipe(ResourceManager manager, String location) {
        return readRecipe(manager, ResType.RECIPES.getPath(location));
    }

    public static Recipe<?> readRecipe(ResourceManager manager, ResourceLocation location) {
        var resource = manager.m_213713_(location);
        try (var stream = resource.orElseThrow().m_215507_()) {
            JsonObject element = RPUtils.deserializeJson(stream);
            return RecipeManager.m_44045_(location, element);
        } catch (Exception e) {
            throw new InvalidOpenTypeException(String.format("Failed to get recipe at %s: %s", location, e));
        }
    }

    public static IRecipeTemplate<?> readRecipeAsTemplate(ResourceManager manager, String location) {
        return readRecipeAsTemplate(manager, ResType.RECIPES.getPath(location));
    }


    public static IRecipeTemplate<?> readRecipeAsTemplate(ResourceManager manager, ResourceLocation location) {
        var resource = manager.m_213713_(location);
        try (var stream = resource.orElseThrow().m_215507_()) {
            JsonObject element = RPUtils.deserializeJson(stream);
            try {
                return TemplateRecipeManager.read(element);
            } catch (Exception e) {
                Moonlight.LOGGER.error(element);
                Moonlight.LOGGER.error(location);
                throw e;
            }

        } catch (Exception e) {
            throw new InvalidOpenTypeException(String.format("Failed to get recipe at %s: %s", location, e));
        }
    }

    public static <T extends BlockType> Recipe<?> makeSimilarRecipe(Recipe<?> original, T originalMat, T destinationMat, String baseID) {
        if (original instanceof ShapedRecipe or) {
            List<Ingredient> newList = new ArrayList<>();
            for (var ingredient : or.m_7527_()) {
                if (ingredient != null && ingredient.m_43908_().length > 0) {
                    ItemLike i = BlockType.changeItemType(ingredient.m_43908_()[0].m_41720_(), originalMat, destinationMat);
                    if (i != null) newList.add(Ingredient.m_43929_(i));
                }
            }
            Item originalRes = or.m_8043_(RegistryAccess.f_243945_).m_41720_();
            ItemLike newRes = BlockType.changeItemType(originalRes, originalMat, destinationMat);
            if (newRes == null) throw new UnsupportedOperationException("Failed to convert recipe");
            ItemStack result = newRes.m_5456_().m_7968_();
            ResourceLocation newId = new ResourceLocation(baseID + "/" + destinationMat.getAppendableId());
            NonNullList<Ingredient> ingredients = NonNullList.m_122783_(Ingredient.f_43901_, newList.toArray(Ingredient[]::new));
            return new ShapedRecipe(newId, or.m_6076_(), or.m_245232_(), or.m_44220_(), or.m_44221_(), ingredients, result);
        } else if (original instanceof ShapelessRecipe or) {
            List<Ingredient> newList = new ArrayList<>();
            for (var ingredient : or.m_7527_()) {
                if (ingredient != null && ingredient.m_43908_().length > 0) {
                    ItemLike i = BlockType.changeItemType(ingredient.m_43908_()[0].m_41720_(), originalMat, destinationMat);
                    if (i != null) newList.add(Ingredient.m_43929_(i));
                }
            }
            Item originalRes = or.m_8043_(RegistryAccess.f_243945_).m_41720_();
            ItemLike newRes = BlockType.changeItemType(originalRes, originalMat, destinationMat);
            if (newRes == null) throw new UnsupportedOperationException("Failed to convert recipe");
            ItemStack result = newRes.m_5456_().m_7968_();
            ResourceLocation newId = new ResourceLocation(baseID + "/" + destinationMat.getAppendableId());
            NonNullList<Ingredient> ingredients = NonNullList.m_122783_(Ingredient.f_43901_, newList.toArray(Ingredient[]::new));
            return new ShapelessRecipe(newId, or.m_6076_(), or.m_245232_(), result, ingredients);
        } else {
            throw new UnsupportedOperationException(String.format("Original recipe %s must be Shaped or Shapeless", original));
        }
    }

    @FunctionalInterface
    public interface OverrideAppender {
        void add(ItemOverride override);

    }

    /**
     * Utility method to add models overrides in a non-destructive way. Provided overrides will be added on top of whatever model is currently provided by vanilla or mod resources. IE: crossbows
     */
    @Deprecated(forRemoval = true)
    public static void appendModelOverride(ResourceManager manager, DynamicTexturePack pack,
                                           ResourceLocation modelRes, Consumer<OverrideAppender> modelConsumer) {
        var json = makeModelOverride(manager, modelRes, modelConsumer);
        pack.addItemModel(modelRes, json);
    }

    public static JsonElement makeModelOverride(ResourceManager manager,
                                                 ResourceLocation modelRes, Consumer<OverrideAppender> modelConsumer) {
        try (var model =  manager.m_215593_(ResType.ITEM_MODELS.getPath(modelRes)).m_215507_()) {
            var json = RPUtils.deserializeJson(model);
            JsonArray overrides;
            if (json.has("overrides")) {
                overrides = json.getAsJsonArray("overrides");
            } else overrides = new JsonArray();

            modelConsumer.accept(ov -> overrides.add(serializeModelOverride(ov)));

            json.add("overrides", overrides);
            return json;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    private static JsonObject serializeModelOverride(ItemOverride override) {
        JsonObject json = new JsonObject();
        json.addProperty("model", override.m_111718_().toString());
        JsonObject predicates = new JsonObject();
        override.m_173449_().forEach(p -> {
            predicates.addProperty(p.m_173459_().toString(), p.m_173460_());
        });
        json.add("predicate", predicates);
        return json;
    }


}
