package net.mehvahdjukaar.moonlight.api.entity;

import net.mehvahdjukaar.moonlight.api.platform.ForgeHelper;
import net.mehvahdjukaar.moonlight.api.util.math.MthUtils;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.core.particles.ParticleTypes;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.network.syncher.EntityDataAccessor;
import net.minecraft.network.syncher.EntityDataSerializers;
import net.minecraft.network.syncher.SynchedEntityData;
import net.minecraft.tags.BlockTags;
import net.minecraft.util.Mth;
import net.minecraft.world.entity.*;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.entity.projectile.AbstractArrow;
import net.minecraft.world.entity.projectile.ProjectileUtil;
import net.minecraft.world.entity.projectile.ThrowableItemProjectile;
import net.minecraft.world.level.ClipContext;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.entity.TheEndGatewayBlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.gameevent.GameEvent;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.EntityHitResult;
import net.minecraft.world.phys.HitResult;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Improved version of the projectile entity. Combines the functionality of AbstractArrow and ThrowableItemProjectile
 * Main features are:
 * - Improved collision handling, onHit and onBlockHit are called after the pos has been set and not before
 * - Correctly handles weird cases like portal and end portal hit, which arrows and snowballs both do only one of
 * - Collision can now use swept AABB collision instead of ray collision, greatly improving accuracy for blocks
 * - Handy overrides such as spawnTrailParticles and hasReachedEndOfLife
 * - Streamlined deceleration and gravity logic
 */
public abstract class ImprovedProjectileEntity extends ThrowableItemProjectile {

    private static final EntityDataAccessor<Byte> ID_FLAGS = SynchedEntityData.m_135353_(ImprovedProjectileEntity.class, EntityDataSerializers.f_135027_);

    protected Vec3 movementOld;

    // Renamed inGround. This is used to check if the projectile is not moving
    protected boolean isStuck = false;
    protected int stuckTime = 0;

    protected int maxAge = 300;
    protected int maxStuckTime = 20;

    protected ImprovedProjectileEntity(EntityType<? extends ThrowableItemProjectile> type, Level world) {
        super(type, world);
        this.movementOld = this.m_20184_();
        this.m_274367_(0);
    }

    protected ImprovedProjectileEntity(EntityType<? extends ThrowableItemProjectile> type, double x, double y, double z, Level world) {
        this(type, world);
        this.m_6034_(x, y, z);
    }

    protected ImprovedProjectileEntity(EntityType<? extends ThrowableItemProjectile> type, LivingEntity thrower, Level world) {
        this(type, thrower.m_20185_(), thrower.m_20188_() - 0.1F, thrower.m_20189_(), world);
        this.m_5602_(thrower);
    }

    @Override
    protected void m_8097_() {
        super.m_8097_();
        this.f_19804_.m_135372_(ID_FLAGS, (byte) 0);
    }

    public boolean hasLeftOwner() {
        return this.f_37246_;
    }

    @Override
    protected float m_6380_(Pose pose, EntityDimensions dimensions) {
        return dimensions.f_20378_ * 0.5f;
    }

    //mix of projectile + arrow code to do what both do+  fix some issues
    @SuppressWarnings("ConstantConditions")
    @Override
    public void m_8119_() {
        //entity has this param. we sync them for consistency
        this.f_19794_ = this.isNoPhysics();

        // Projectile tick stuff
        if (!this.f_150164_) {
            this.m_146852_(GameEvent.f_157778_, this.m_19749_());
            this.f_150164_ = true;
        }
        if (!this.f_37246_) {
            this.f_37246_ = this.m_37276_();
        }

        this.m_6075_();

        if (this.hasReachedEndOfLife() && !m_213877_()) {
            this.reachedEndOfLife();
        }

        // end of projectile tick stuff

        // AbstractArrow + ThrowableProjectile stuff

        //fixed vanilla arrow code. You're welcome
        Level level = this.m_9236_();
        Vec3 movement = this.m_20184_();
        this.movementOld = movement;

        if (this.f_19865_.m_82556_() > 1.0E-7) {
            movement = movement.m_82559_(this.f_19865_);
            this.f_19865_ = Vec3.f_82478_;
            this.m_20256_(Vec3.f_82478_);
        }

        if (!this.f_19794_ && this.isStuck) {
            this.stuckTime++;
        }else {
            this.stuckTime = 0;
        }

        this.m_6478_(MoverType.SELF, movement);

        // rest stuff
        this.m_146872_();
        this.updateFireState();

        // after we finished moving we can apply forces and  particles

        // update movement and particles
        float deceleration = this.m_20069_() ? this.getWaterInertia() : this.getInertia();

        this.m_20256_(this.m_20184_().m_82490_(deceleration));
        if (!this.m_20068_() && !f_19794_) {
            this.m_20256_(this.m_20184_().m_82492_(0, this.m_7139_(), 0));
        }

        if (!isStuck) {
            if (level.f_46443_) {
                this.spawnTrailParticles();
            }

            this.m_37283_();
        }

        // check if stuck
        this.isStuck = !this.f_19794_ && this.m_20182_().m_82492_(this.f_19854_, this.f_19855_, this.f_19856_).m_82556_() < (0.0001 * 0.0001);

    }

    private void updateFireState() {
        //copied bit from move method. Extracted for clarity
        this.f_146810_ = this.m_6060_();

        if (this.m_9236_().m_46847_(this.m_20191_().m_82406_(1.0E-6)).noneMatch((arg) ->
                arg.m_204336_(BlockTags.f_13076_) || arg.m_60713_(Blocks.f_49991_))) {
            if (this.m_20094_() <= 0) {
                this.m_7311_(-this.m_6101_());
            }

            if (this.f_146810_ && (this.f_146808_ || this.m_20071_() ||
                    ForgeHelper.isInFluidThatCanExtinguish(this))) {
                this.m_146873_();
            }
        }

        if (this.m_6060_() && (this.f_146808_ || this.m_20071_() || ForgeHelper.isInFluidThatCanExtinguish(this))) {
            this.m_7311_(-this.m_6101_());
        }
    }

    /**
     * Tries moving this entity by movement amount.
     * Does all the checks it needs and calls onHit if it hits something.
     * This was made from entity.move  + much stuff from both arrow and projectile code.
     * Does not check for fall damage or other living entity specific stuff
     * If blocks are hit movement is not modified. You need to react in onHitBlock if you wish to do so
     * Movement isn't modified at all here.
     * On fallOn isn't called as we call onProjectile hit. Projectiles dont fall.
     *
     * @param movement amount to travel by
     */
    @Override
    public void m_6478_(MoverType moverType, Vec3 movement) {
        // use normal movement logic if not self.. idk why compat i guess incase we were to use ray collider
        // also ued for no physics as that will just set the pos without doing any collision
        if (moverType != MoverType.SELF || this.f_19794_) {
            super.m_6478_(moverType, movement);
            return;
        }

        movement = this.m_5763_(movement, moverType);

        Level level = this.m_9236_();
        Vec3 pos = this.m_20182_();

        // Applies collisions calculating hit face and new pos
        ColliderType colliderType = this.getColliderType();

        HitResult hitResult = switch (colliderType) {
            case RAY -> level.m_45547_(new ClipContext(pos, pos.m_82549_(movement),
                    ClipContext.Block.COLLIDER, ClipContext.Fluid.NONE, this));
            case AABB -> MthUtils.collideWithSweptAABB(this, movement, 2);
            case ENTITY_COLLIDE -> {
                Vec3 vec3 = this.m_20272_(movement);
                Vec3 sub = vec3.m_82546_(movement);
                yield vec3 == movement ? BlockHitResult.m_82426_(pos.m_82549_(vec3), Direction.UP,
                        BlockPos.m_274446_(pos.m_82549_(vec3))) : new BlockHitResult(pos.m_82549_(vec3),
                        Direction.m_122366_(sub.f_82479_, sub.f_82480_, sub.f_82481_), BlockPos.m_274446_(pos.m_82549_(vec3)), false);
            }
        };

        Vec3 newPos = hitResult.m_82450_();
        Vec3 newMovement = newPos.m_82546_(pos);
        this.m_6034_(newPos.f_82479_, newPos.f_82480_, newPos.f_82481_);

        //this is mainly used for players
        boolean bl = !Mth.m_14082_(newMovement.f_82479_, movement.f_82479_);
        boolean bl2 = !Mth.m_14082_(newMovement.f_82481_, movement.f_82481_);
        this.f_19862_ = bl || bl2;
        this.f_19863_ = newMovement.f_82480_ != movement.f_82480_;
        this.f_201939_ = this.f_19863_ && newMovement.f_82480_ < 0.0;
        if (this.f_19862_) {
            this.f_185931_ = this.m_196406_(newMovement);
        } else {
            this.f_185931_ = false;
        }

        //try hit entity
        EntityHitResult entityHitResult = ProjectileUtil.m_37304_(level, this, pos, newPos,
                this.m_20191_().m_82369_(newPos.m_82546_(pos)).m_82400_(1.0D), this::m_5603_);

        if (entityHitResult != null) {
            hitResult = entityHitResult;
        }

        boolean portalHit = false;
        if (hitResult instanceof EntityHitResult ei) {
            Entity hitEntity = ei.m_82443_();
            if (hitEntity == this.m_19749_()) {
                if (!canHarmOwner()) {
                    hitResult = null;
                }
            } else if (hitEntity instanceof Player p1 && this.m_19749_() instanceof Player p2 && !p2.m_7099_(p1)) {
                hitResult = null;
            }
        } else if (hitResult instanceof BlockHitResult bi) {
            //portals. done here and not in onBlockHit to prevent any further calls
            BlockPos hitPos = bi.m_82425_();
            BlockState hitState = level.m_8055_(hitPos);

            if (hitState.m_60713_(Blocks.f_50142_)) {
                this.m_20221_(hitPos);
                portalHit = true;
            } else if (hitState.m_60713_(Blocks.f_50446_)) {
                if (level.m_7702_(hitPos) instanceof TheEndGatewayBlockEntity tile && TheEndGatewayBlockEntity.m_59940_(this)) {
                    TheEndGatewayBlockEntity.m_155828_(level, hitPos, hitState, this, tile);
                }
                portalHit = true;
            }
        }

        if (!portalHit && hitResult != null && hitResult.m_6662_() != HitResult.Type.MISS &&
                !ForgeHelper.onProjectileImpact(this, hitResult)) {
            this.m_6532_(hitResult);
        }
    }

    public boolean canHarmOwner() {
        if (m_19749_() instanceof Player) {
            return m_9236_().m_46791_().m_19028_() >= 1;
        }
        return false;
    }

    protected float getInertia() {
        // normally 0.99 for everything
        return 0.99F;
    }

    protected float getWaterInertia() {
        // normally 0.6 for arrows and 0.99 for tridents and 0.8 for other projectiles
        return 0.6F;
    }

    /**
     * do stuff before removing, then call remove. Called when age reaches max age
     */
    public boolean hasReachedEndOfLife() {
        return this.f_19797_ > this.maxAge || this.stuckTime > maxStuckTime;
    }

    /**
     * remove condition
     */
    public void reachedEndOfLife() {
        this.m_142687_(RemovalReason.DISCARDED);
    }

    @Deprecated(forRemoval = true)
    public void spawnTrailParticles(Vec3 oldPos, Vec3 newPos) {
    }

    public void spawnTrailParticles() {
        spawnTrailParticles(new Vec3(f_19854_, f_19855_, f_19856_), this.m_20182_());

        if (this.m_20069_()) {
            // Projectile particle code
            var movement = this.m_20184_();
            double velX = movement.f_82479_;
            double velY = movement.f_82480_;
            double velZ = movement.f_82481_;
            for (int j = 0; j < 4; ++j) {
                double pY = this.m_20188_();
                m_9236_().m_7106_(ParticleTypes.f_123795_,
                        m_20185_() - velX * 0.25D, pY - velY * 0.25D, m_20189_() - velZ * 0.25D,
                        velX, velY, velZ);
            }
        }
    }

    @Override
    public void m_7380_(@NotNull CompoundTag tag) {
        super.m_7380_(tag);
        tag.m_128379_("stuck", this.isStuck);
        tag.m_128405_("stuckTime", this.stuckTime);
        tag.m_128379_("noPhysics", this.isNoPhysics());
    }

    @Override
    public void m_7378_(@NotNull CompoundTag tag) {
        super.m_7378_(tag);
        this.isStuck = tag.m_128471_("stuck");
        this.stuckTime = tag.m_128451_("stuckTime");
        this.setNoPhysics(tag.m_128471_("noPhysics"));
    }

    @Override
    public void m_37251_(Entity shooter, float x, float y, float z, float velocity, float inaccuracy) {
        super.m_37251_(shooter, x, y, z, velocity, inaccuracy);
    }

    @Override
    public void m_6686_(double x, double y, double z, float velocity, float inaccuracy) {
        super.m_6686_(x, y, z, velocity, inaccuracy);
    }

    // Has no effect. Give this to shoot method manually
    public float getDefaultShootVelocity() {
        return 1.5F;
    }

    protected void setFlag(int id, boolean value) {
        byte b0 = this.f_19804_.m_135370_(ID_FLAGS);
        if (value) {
            this.f_19804_.m_135381_(ID_FLAGS, (byte) (b0 | id));
        } else {
            this.f_19804_.m_135381_(ID_FLAGS, (byte) (b0 & ~id));
        }
    }

    protected boolean getFlag(int id) {
        return (this.f_19804_.m_135370_(ID_FLAGS) & id) != 0;
    }

    // 2 cause its same as arrows for consistency
    public void setNoPhysics(boolean noPhysics) {
        this.f_19794_ = noPhysics;
        this.setFlag(2, noPhysics);
    }

    public boolean isNoPhysics() {
        return this.getFlag(2);
    }

    @Deprecated(forRemoval = true)
    public boolean touchedGround;
    @Deprecated(forRemoval = true)
    public int groundTime = 0;

    @Deprecated(forRemoval = true)
    protected float getDeceleration() {
        return getInertia();
    }

    @Deprecated(forRemoval = true)
    @Nullable
    protected EntityHitResult findHitEntity(Vec3 oPos, Vec3 pos) {
        return ProjectileUtil.m_37304_(this.m_9236_(), this, oPos, pos, this.m_20191_().m_82369_(this.m_20184_()).m_82400_(1.0D), this::m_5603_);
    }

    /**
     * AABB: Swept AABB collision, gives very accurate block collisions and stops the entity on the first detected collision
     * RAY: Ray collision, fast but only accurate in the center of the projectile. Ok to use for small projectiles. What arrows use
     */
    protected ColliderType getColliderType() {
        return ColliderType.AABB;
    }

    protected enum ColliderType {
        RAY,
        AABB,
        ENTITY_COLLIDE
    }
}
