package net.mehvahdjukaar.moonlight.api.fluids;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.mehvahdjukaar.moonlight.api.misc.StrOpt;
import net.mehvahdjukaar.moonlight.api.platform.ForgeHelper;
import net.mehvahdjukaar.moonlight.api.platform.PlatHelper;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.world.effect.MobEffect;
import net.minecraft.world.effect.MobEffectInstance;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.food.FoodProperties;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.level.Level;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.function.Consumer;

public class FoodProvider {

    public static final Codec<FoodProvider> CODEC = RecordCodecBuilder.create((instance) -> instance.group(
            BuiltInRegistries.f_257033_.m_194605_().fieldOf("item").forGetter(f -> f.food),
            StrOpt.of(SoftFluid.Capacity.INT_CODEC, "divider", 1).forGetter(f -> f.divider)
    ).apply(instance, FoodProvider::create));

    public static final FoodProvider EMPTY = new FoodProvider(Items.f_41852_, 1);

    protected final Item food;
    protected final int divider;

    private FoodProvider(Item food, int divider) {
        this.food = food;
        this.divider = divider;
    }

    public Item getFood() {
        return food;
    }

    public int getDivider() {
        return divider;
    }

    public boolean isEmpty() {
        return this == EMPTY;
    }

    /**
     * Consumes some fluid and gives the player the appropriate effect
     *
     * @return if one unit has been consumed
     */
    public boolean consume(Player player, Level world, @Nullable Consumer<ItemStack> nbtApplier) {

        ItemStack stack = this.food.m_7968_();
        if (nbtApplier != null) nbtApplier.accept(stack);

        //food

        FoodProperties foodProperties = PlatHelper.getFoodProperties(this.food, stack, player);
        //single items are handled by items themselves
        if (this.divider == 1) {
            this.food.m_5922_(stack.m_41777_(), world, player);
            if (foodProperties == null || stack.m_41720_().m_41472_()) {
                player.m_5496_(this.food.m_6023_(), 1, 1);
            }
            //player already plays sound
            return true;
        }
        if (foodProperties != null && player.m_36391_(false)) {

            player.m_36324_().m_38707_(foodProperties.m_38744_() / this.divider, foodProperties.m_38745_() / (float) this.divider);
            player.m_5496_(this.food.m_6023_(), 1, 1);
            return true;
        }
        return false;
    }

    public static FoodProvider create(Item item, int divider) {
        return CUSTOM_PROVIDERS.getOrDefault(item, new FoodProvider(item, divider));
    }

    private static final FoodProvider XP = new FoodProvider(Items.f_42612_, 1) {

        @Override
        public boolean consume(Player player, Level world, @Nullable Consumer<ItemStack> nbtApplier) {
            player.m_6756_(Utils.getXPinaBottle(1, world.f_46441_));
            player.m_5496_(SoundEvents.f_11871_, 1, 1);
            return true;
        }
    };

    private static final FoodProvider MILK = new FoodProvider(Items.f_42455_, 3) {

        @Override
        public boolean consume(Player player, Level world, @Nullable Consumer<ItemStack> nbtApplier) {
            ItemStack stack = this.food.m_7968_();
            if (nbtApplier != null) nbtApplier.accept(stack);
            for (MobEffectInstance effect : player.m_21221_().values()) {
                if (ForgeHelper.isCurativeItem(stack, effect)) {
                    player.m_21195_(effect.m_19544_());
                    break;
                }
            }
            player.m_5496_(this.food.m_6023_(), 1, 1);
            return true;
        }
    };

    private static final FoodProvider SUS_STEW = new FoodProvider(Items.f_42718_, 2) {

        @Override
        public boolean consume(Player player, Level world, @Nullable Consumer<ItemStack> nbtApplier) {

            ItemStack stack = this.food.m_7968_();
            if (nbtApplier != null) nbtApplier.accept(stack);
            FoodProperties foodProperties = this.food.m_41473_();
            if (foodProperties != null && player.m_36391_(false)) {

                CompoundTag tag = stack.m_41783_();
                if (tag != null && tag.m_128425_("Effects", 9)) {
                    ListTag effects = tag.m_128437_("Effects", 10);
                    for (int i = 0; i < effects.size(); ++i) {
                        int j = 160;
                        CompoundTag effectsCompound = effects.m_128728_(i);
                        if (effectsCompound.m_128425_("EffectDuration", 3))
                            j = effectsCompound.m_128451_("EffectDuration") / this.divider;
                        MobEffect effect = MobEffect.m_19453_(effectsCompound.m_128445_("EffectId"));
                        if (effect != null) {
                            player.m_7292_(new MobEffectInstance(effect, j));
                        }
                    }
                }
                player.m_36324_().m_38707_(foodProperties.m_38744_() / this.divider, foodProperties.m_38745_() / (float) this.divider);
                player.m_5496_(this.food.m_6023_(), 1, 1);
                return true;
            }
            return false;
        }
    };

    public static final Map<Item, FoodProvider> CUSTOM_PROVIDERS = Map.of(
            Items.f_41852_, EMPTY,
            Items.f_42718_, SUS_STEW,
            Items.f_42455_, MILK,
            Items.f_42612_, XP
    );
}
