package net.mehvahdjukaar.moonlight.api.misc;

import com.mojang.datafixers.util.Either;
import net.mehvahdjukaar.moonlight.api.util.Utils;
import net.minecraft.core.Holder;
import net.minecraft.core.HolderOwner;
import net.minecraft.core.Registry;
import net.minecraft.core.RegistryAccess;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.tags.TagKey;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Stream;

//can be statically stored and persists across world loads

/**
 * A soft reference to an object in a Data pack registry
 * Like registry object but can be invalidated and works for data pack registries
 */
public class DynamicHolder<T> implements Supplier<T>, Holder<T> {

    @ApiStatus.Internal
    public static void clearCache() {
        REFERENCES.forEach(DynamicHolder::invalidateInstance);
    }

    private static final WeakHashSet<DynamicHolder<?>> REFERENCES = new WeakHashSet<>();

    private final ResourceKey<Registry<T>> registryKey;
    private final ResourceKey<T> key;

    // needs to be thread local because of data pack stuff. datapack registries will have 2 objects with the same key
    protected final ThreadLocal<Holder<T>> instance = new ThreadLocal<>();

    protected DynamicHolder(ResourceKey<Registry<T>> registryKey, ResourceKey<T> key) {
        this.registryKey = registryKey;
        this.key = key;
        REFERENCES.add(this);
    }

    public static <A> DynamicHolder<A> of(String id, ResourceKey<Registry<A>> registry) {
        return of(new ResourceLocation(id), registry);
    }

    public static <A> DynamicHolder<A> of(ResourceLocation location, ResourceKey<Registry<A>> registry) {
        return new DynamicHolder<>(registry, ResourceKey.m_135785_(registry, location));
    }

    public static <A> DynamicHolder<A> of(ResourceKey<A> key) {
        return new DynamicHolder<>(ResourceKey.m_135788_(key.m_211136_()), key);
    }

    public static <A> Opt<A> optional(ResourceLocation location, ResourceKey<Registry<A>> registry) {
        return new Opt<>(registry, ResourceKey.m_135785_(registry, location));
    }

    public static <A> Opt<A> optional(ResourceKey<A> key) {
        return new Opt<>(ResourceKey.m_135788_(key.m_211136_()), key);
    }

    public static <A> Opt<A> optional(String id, ResourceKey<Registry<A>> registry) {
        return optional(new ResourceLocation(id), registry);
    }

    private void invalidateInstance() {
        instance.remove();
    }

    @NotNull
    protected Holder<T> getInstance() {
        Holder<T> value = instance.get();
        if (value == null) {
            var r = Utils.hackyGetRegistryAccess();
            Registry<T> reg = r.m_175515_(registryKey);
            try {
                value = reg.m_246971_(key);
                instance.set(value);
            } catch (Exception e) {
                throw new RuntimeException("Failed to get object from registry: " + key +
                        ".\nCalled from " + Thread.currentThread() + ".\n" +
                        "Registry content was: " + reg.m_6579_().stream().map(b -> b.getKey().m_135782_()).toList(), e);
            }
        }
        return value;
    }

    public String getRegisteredName() {
        return key.m_135782_().toString();
    }

    public ResourceLocation getID() {
        return key.m_135782_();
    }

    public ResourceKey<T> getKey() {
        return key;
    }

    @Deprecated(forRemoval = true)
    @NotNull
    public T get() {
        return m_203334_();
    }

    @Override
    public T m_203334_() {
        return getInstance().m_203334_();
    }

    @Override
    public boolean m_203633_() {
        return true;
    }

    @Override
    public boolean m_203373_(ResourceLocation location) {
        return registryKey.m_135782_().equals(location);
    }

    @Override
    public boolean m_203565_(ResourceKey<T> resourceKey) {
        return resourceKey == key;
    }

    @Override
    public boolean m_203425_(Predicate<ResourceKey<T>> predicate) {
        return predicate.test(key);
    }

    public boolean is(Holder<T> other) {
        return other == this || other.m_203543_().get() == key;
    }

    @Override
    public boolean m_203656_(TagKey<T> tagKey) {
        return getInstance().m_203656_(tagKey);
    }

    @Override
    public Stream<TagKey<T>> m_203616_() {
        return getInstance().m_203616_();
    }

    @Override
    public Either<ResourceKey<T>, T> m_203439_() {
        return Either.left(this.key);
    }

    @Override
    public Optional<ResourceKey<T>> m_203543_() {
        return Optional.of(key);
    }

    @Override
    public Kind m_203376_() {
        return Holder.Kind.REFERENCE;
    }

    @Override
    public boolean m_203401_(HolderOwner<T> owner) {
        return getInstance().m_203401_(owner);
    }

    public static class Opt<T> extends DynamicHolder<T> {
        private boolean resolved = false;

        protected Opt(ResourceKey<Registry<T>> registryKey, ResourceKey<T> key) {
            super(registryKey, key);
        }

        @Nullable
        @Override
        protected Holder<T> getInstance() {
            if (!resolved) {
                resolved = true;
                try {
                    return super.getInstance();
                } catch (Exception ignored) {
                }
            }
            return instance.get();
        }

        @Override
        public Stream<TagKey<T>> m_203616_() {
            var i = getInstance();
            if (i != null) return i.m_203616_();
            return Stream.empty();
        }

        @Override
        public boolean m_203656_(TagKey<T> tagKey) {
            var i = getInstance();
            if (i != null) return i.m_203656_(tagKey);
            return false;
        }

        @Nullable
        @Override
        public T get() {
            return super.get();
        }

        @Nullable
        @Override
        public T m_203334_() {
            var i = getInstance();
            if (i != null) return i.m_203334_();
            return null;
        }
    }
}
